/**
 * Logo Path Fix Utility
 * Helps diagnose and fix logo path issues
 */

const fs = require('fs');
const path = require('path');
const { query } = require('../config/database');

/**
 * Check if logo file exists and fix path if needed
 */
async function checkAndFixLogoPath() {
    try {
        console.log('🔍 Checking logo paths in database...');
        
        // Get shop info
        const sql = 'SELECT * FROM tbl_shopinfo WHERE id = 1';
        const result = await query(sql);
        
        if (result.length === 0) {
            console.log('❌ No shop info found in database');
            return;
        }
        
        const shopInfo = result[0];
        console.log('📋 Current shop info:');
        console.log('  - Name:', shopInfo.name);
        console.log('  - Logo Path:', shopInfo.photo_path);
        
        if (!shopInfo.photo_path) {
            console.log('⚠️ No logo path set in database');
            return;
        }
        
        // Check if file exists
        const logoPath = path.join(__dirname, '..', shopInfo.photo_path);
        console.log('🔍 Checking file at:', logoPath);
        
        if (fs.existsSync(logoPath)) {
            console.log('✅ Logo file exists');
            
            // Get file stats
            const stats = fs.statSync(logoPath);
            console.log('📊 File stats:');
            console.log('  - Size:', stats.size, 'bytes');
            console.log('  - Modified:', stats.mtime);
            
            return {
                success: true,
                message: 'Logo file exists and is accessible',
                path: shopInfo.photo_path,
                size: stats.size
            };
        } else {
            console.log('❌ Logo file does not exist');
            
            // Try to find the file in uploads directory
            const uploadsDir = path.join(__dirname, '..', 'uploads', 'logo');
            console.log('🔍 Searching in uploads directory:', uploadsDir);
            
            if (fs.existsSync(uploadsDir)) {
                const files = fs.readdirSync(uploadsDir);
                console.log('📁 Found files in uploads/logo:', files);
                
                if (files.length > 0) {
                    // Use the most recent file
                    const latestFile = files[files.length - 1];
                    const newPath = `/uploads/logo/${latestFile}`;
                    
                    console.log('🔄 Updating database with latest logo:', newPath);
                    
                    // Update database
                    const updateSql = 'UPDATE tbl_shopinfo SET photo_path = ? WHERE id = 1';
                    await query(updateSql, [newPath]);
                    
                    console.log('✅ Database updated with new logo path');
                    
                    return {
                        success: true,
                        message: 'Logo path updated to latest file',
                        oldPath: shopInfo.photo_path,
                        newPath: newPath
                    };
                }
            }
            
            return {
                success: false,
                message: 'Logo file not found and no alternatives available',
                path: shopInfo.photo_path
            };
        }
        
    } catch (error) {
        console.error('❌ Error checking logo path:', error);
        return {
            success: false,
            message: 'Error checking logo path: ' + error.message
        };
    }
}

/**
 * List all available logo files
 */
function listAvailableLogos() {
    try {
        const uploadsDir = path.join(__dirname, '..', 'uploads', 'logo');
        
        if (!fs.existsSync(uploadsDir)) {
            console.log('❌ Uploads directory does not exist');
            return [];
        }
        
        const files = fs.readdirSync(uploadsDir);
        const logoFiles = [];
        
        files.forEach(file => {
            const filePath = path.join(uploadsDir, file);
            const stats = fs.statSync(filePath);
            
            logoFiles.push({
                name: file,
                path: `/uploads/logo/${file}`,
                size: stats.size,
                modified: stats.mtime
            });
        });
        
        console.log('📁 Available logo files:');
        logoFiles.forEach(logo => {
            console.log(`  - ${logo.name} (${logo.size} bytes, ${logo.modified})`);
        });
        
        return logoFiles;
        
    } catch (error) {
        console.error('❌ Error listing logos:', error);
        return [];
    }
}

/**
 * Set a specific logo file
 */
async function setLogoFile(filename) {
    try {
        const logoPath = `/uploads/logo/${filename}`;
        const fullPath = path.join(__dirname, '..', logoPath);
        
        if (!fs.existsSync(fullPath)) {
            throw new Error(`Logo file not found: ${filename}`);
        }
        
        // Update database
        const updateSql = 'UPDATE tbl_shopinfo SET photo_path = ? WHERE id = 1';
        await query(updateSql, [logoPath]);
        
        console.log('✅ Logo path updated to:', logoPath);
        
        return {
            success: true,
            message: 'Logo path updated successfully',
            path: logoPath
        };
        
    } catch (error) {
        console.error('❌ Error setting logo file:', error);
        return {
            success: false,
            message: 'Error setting logo file: ' + error.message
        };
    }
}

module.exports = {
    checkAndFixLogoPath,
    listAvailableLogos,
    setLogoFile
};
