/**
 * Fixed Database Migration Tool
 * Properly handles DELIMITER and multi-line statements
 */

const fs = require('fs');
const path = require('path');
const { pool } = require('../config/database');

async function runMigration() {
    try {
        console.log('🚀 Starting database migration...\n');

        const connection = await pool.getConnection();

        // Read SQL file
        const sqlFile = path.join(__dirname, '../../database/infotech_db_mysql.sql');
        const sqlContent = fs.readFileSync(sqlFile, 'utf8');

        // Split by delimiter-aware logic
        const sections = sqlContent.split('DELIMITER');
        
        let tableCount = 0;
        let procCount = 0;
        let errorCount = 0;

        // Execute first section (tables and inserts before procedures)
        if (sections.length > 0) {
            const tableStatements = sections[0]
                .split(';')
                .map(s => s.trim())
                .filter(s => s.length > 10 && !s.startsWith('--') && !s.startsWith('/*'));

            for (const statement of tableStatements) {
                if (statement.includes('CREATE DATABASE')) continue;
                if (statement.includes('USE')) continue;
                if (statement.includes('SELECT \'Database')) continue;

                try {
                    await connection.query(statement);
                    
                    if (statement.includes('CREATE TABLE')) {
                        const match = statement.match(/CREATE TABLE.*?`(\w+)`/i);
                        if (match) {
                            console.log(`✅ Table: ${match[1]}`);
                            tableCount++;
                        }
                    } else if (statement.includes('CREATE INDEX')) {
                        const match = statement.match(/CREATE INDEX.*?(\w+)/i);
                        if (match) {
                            console.log(`✅ Index: ${match[1]}`);
                        }
                    } else if (statement.includes('INSERT INTO')) {
                        // Silent for inserts
                    }
                } catch (error) {
                    if (!error.message.includes('already exists') && 
                        !error.message.includes('Duplicate')) {
                        console.log(`⚠️  ${error.message.substring(0, 80)}`);
                        errorCount++;
                    }
                }
            }
        }

        // Handle stored procedures separately
        console.log('\n📦 Creating Stored Procedures...');
        const procedures = [
            {
                name: 'auto_purchase_id',
                sql: `
                CREATE PROCEDURE auto_purchase_id()
                BEGIN
                    DECLARE v_row_count INT;
                    DECLARE v_id INT;
                    DECLARE v_temp_new_number INT;
                    
                    SET v_row_count = (SELECT COUNT(id) FROM tbl_purchase);
                    
                    IF (v_row_count = 0) THEN
                        SET v_id = 1;
                    ELSE
                        SET v_temp_new_number = (SELECT IFNULL(MAX(id), 0) FROM tbl_purchase);
                        SET v_id = v_temp_new_number + 1;
                    END IF;
                    
                    SELECT v_id AS id;
                END`
            },
            {
                name: 'AutoGeneratevoucherNo_purchase',
                sql: `
                CREATE PROCEDURE AutoGeneratevoucherNo_purchase()
                BEGIN
                    DECLARE v_row_count INT;
                    DECLARE v_voucher INT;
                    DECLARE v_temp_new_number INT;
                    
                    SET v_row_count = (SELECT COUNT(id) FROM tbl_purchase);
                    
                    IF (v_row_count = 0) THEN
                        SET v_voucher = 1000;
                    ELSE
                        SET v_temp_new_number = (SELECT IFNULL(MAX(purchase_voucher_id), 1000) FROM tbl_purchase);
                        SET v_voucher = v_temp_new_number + 1;
                    END IF;
                    
                    SELECT v_voucher AS voucher;
                END`
            },
            {
                name: 'AutoGeneratevoucherNo_Sale',
                sql: `
                CREATE PROCEDURE AutoGeneratevoucherNo_Sale()
                BEGIN
                    DECLARE v_row_count INT;
                    DECLARE v_voucher INT;
                    DECLARE v_temp_new_number INT;
                    
                    SET v_row_count = (SELECT COUNT(id) FROM tbl_sale);
                    
                    IF (v_row_count = 0) THEN
                        SET v_voucher = 1000;
                    ELSE
                        SET v_temp_new_number = (SELECT IFNULL(MAX(VNo), 1000) FROM tbl_sale);
                        SET v_voucher = v_temp_new_number + 1;
                    END IF;
                    
                    SELECT v_voucher AS voucher;
                END`
            }
        ];

        for (const proc of procedures) {
            try {
                // Drop if exists
                await connection.query(`DROP PROCEDURE IF EXISTS ${proc.name}`);
                // Create procedure
                await connection.query(proc.sql);
                console.log(`✅ Procedure: ${proc.name}`);
                procCount++;
            } catch (error) {
                console.log(`❌ Procedure ${proc.name}: ${error.message.substring(0, 80)}`);
                errorCount++;
            }
        }

        connection.release();

        console.log('\n🎉 Migration completed!');
        console.log(`✅ Tables created: ${tableCount}`);
        console.log(`✅ Procedures created: ${procCount}`);
        if (errorCount > 0) {
            console.log(`⚠️  Warnings: ${errorCount} (may be safe to ignore)`);
        }

        return true;

    } catch (error) {
        console.error('❌ Migration failed:', error.message);
        return false;
    }
}

// Run migration if called directly
if (require.main === module) {
    require('dotenv').config();
    runMigration().then(success => {
        process.exit(success ? 0 : 1);
    });
}

module.exports = { runMigration };

