/**
 * Professional Hospital Features Migration Script
 * Runs all 4 advanced feature migrations
 */

const fs = require('fs');
const path = require('path');
const { query } = require('../config/database');

const migrations = [
    {
        file: '01_laboratory_system.sql',
        name: 'Laboratory & Diagnostic Integration'
    },
    {
        file: '02_hospital_operations.sql',
        name: 'Advanced Hospital Operations'
    },
    {
        file: '03_advanced_pharmacy.sql',
        name: 'Advanced Pharmacy Features'
    },
    {
        file: '04_analytics_reporting.sql',
        name: 'Enhanced Reporting & Analytics'
    }
];

async function runMigration(filename, migrationName) {
    console.log(`\n📋 Running: ${migrationName}`);
    console.log(`   File: ${filename}`);
    
    try {
        const sqlPath = path.join(__dirname, '../../database', filename);
        
        if (!fs.existsSync(sqlPath)) {
            throw new Error(`Migration file not found: ${sqlPath}`);
        }
        
        const sql = fs.readFileSync(sqlPath, 'utf8');
        
        // Split by semicolon and filter out empty statements
        const statements = sql
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0 && !s.startsWith('--') && s !== 'USE clinic_pro_db');
        
        console.log(`   📝 Found ${statements.length} SQL statements`);
        
        let successCount = 0;
        let skipCount = 0;
        
        for (let i = 0; i < statements.length; i++) {
            try {
                await query(statements[i]);
                successCount++;
            } catch (error) {
                // If table/column already exists, skip silently
                if (error.code === 'ER_TABLE_EXISTS_ERROR' || 
                    error.code === 'ER_DUP_FIELDNAME' ||
                    error.code === 'ER_DUP_KEYNAME' ||
                    error.code === 'ER_DUP_ENTRY') {
                    skipCount++;
                } else {
                    console.error(`   ❌ Error in statement ${i + 1}:`, error.message);
                }
            }
        }
        
        console.log(`   ✅ Success: ${successCount} statements`);
        if (skipCount > 0) {
            console.log(`   ⏭️  Skipped: ${skipCount} (already exists)`);
        }
        
        return { success: true, executed: successCount, skipped: skipCount };
    } catch (error) {
        console.error(`   ❌ Failed: ${error.message}`);
        return { success: false, error: error.message };
    }
}

async function main() {
    console.log('\n╔═══════════════════════════════════════════════════════════╗');
    console.log('║                                                           ║');
    console.log('║   Professional Hospital Features Migration                ║');
    console.log('║   Clinic Pro V3 - Advanced Features Setup                 ║');
    console.log('║                                                           ║');
    console.log('╚═══════════════════════════════════════════════════════════╝\n');
    
    const startTime = Date.now();
    const results = [];
    
    for (const migration of migrations) {
        const result = await runMigration(migration.file, migration.name);
        results.push({ name: migration.name, ...result });
    }
    
    const endTime = Date.now();
    const duration = ((endTime - startTime) / 1000).toFixed(2);
    
    console.log('\n╔═══════════════════════════════════════════════════════════╗');
    console.log('║                    Migration Summary                      ║');
    console.log('╚═══════════════════════════════════════════════════════════╝\n');
    
    const successfulMigrations = results.filter(r => r.success).length;
    const totalExecuted = results.reduce((sum, r) => sum + (r.executed || 0), 0);
    const totalSkipped = results.reduce((sum, r) => sum + (r.skipped || 0), 0);
    
    console.log(`✅ Successful Migrations: ${successfulMigrations}/${migrations.length}`);
    console.log(`📝 Total Statements Executed: ${totalExecuted}`);
    if (totalSkipped > 0) {
        console.log(`⏭️  Total Statements Skipped: ${totalSkipped}`);
    }
    console.log(`⏱️  Duration: ${duration} seconds`);
    
    console.log('\n📊 Detailed Results:');
    results.forEach((result, index) => {
        const status = result.success ? '✅' : '❌';
        console.log(`   ${status} ${result.name}`);
        if (result.success) {
            console.log(`      Executed: ${result.executed}, Skipped: ${result.skipped}`);
        } else {
            console.log(`      Error: ${result.error}`);
        }
    });
    
    if (successfulMigrations === migrations.length) {
        console.log('\n╔═══════════════════════════════════════════════════════════╗');
        console.log('║                                                           ║');
        console.log('║   🎉 ALL MIGRATIONS COMPLETED SUCCESSFULLY! 🎉            ║');
        console.log('║                                                           ║');
        console.log('║   Your system now includes:                               ║');
        console.log('║   ✅ Laboratory & Diagnostic Integration                 ║');
        console.log('║   ✅ Advanced Hospital Operations                        ║');
        console.log('║   ✅ Advanced Pharmacy Features                          ║');
        console.log('║   ✅ Enhanced Reporting & Analytics                      ║');
        console.log('║                                                           ║');
        console.log('║   Next Steps:                                             ║');
        console.log('║   1. Restart your backend server (npm start)              ║');
        console.log('║   2. Test the new APIs (see implementation guide)         ║');
        console.log('║   3. Create frontend pages for new features               ║');
        console.log('║                                                           ║');
        console.log('╚═══════════════════════════════════════════════════════════╝\n');
    } else {
        console.log('\n⚠️  Some migrations failed. Please check the errors above and retry.\n');
    }
    
    process.exit(successfulMigrations === migrations.length ? 0 : 1);
}

// Run migrations
main().catch(error => {
    console.error('\n❌ Migration failed:', error);
    process.exit(1);
});

