/**
 * Database Migration Tool
 * Creates all tables based on the schema
 */

const fs = require('fs');
const path = require('path');
const { pool } = require('../config/database');

async function runMigration() {
    try {
        console.log('🚀 Starting database migration...\n');

        // Read SQL file
        const sqlFile = path.join(__dirname, '../../database/infotech_db_mysql.sql');
        const sql = fs.readFileSync(sqlFile, 'utf8');

        // Split SQL file into individual statements
        const statements = sql
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0 && !s.startsWith('--') && !s.startsWith('/*'));

        const connection = await pool.getConnection();

        let successCount = 0;
        let errorCount = 0;

        for (let i = 0; i < statements.length; i++) {
            const statement = statements[i];
            
            // Skip comments and empty statements
            if (!statement || statement.startsWith('--')) continue;

            try {
                await connection.query(statement);
                
                // Log table creation
                if (statement.includes('CREATE TABLE')) {
                    const match = statement.match(/CREATE TABLE.*?`(\w+)`/i);
                    if (match) {
                        console.log(`✅ Created table: ${match[1]}`);
                        successCount++;
                    }
                } else if (statement.includes('CREATE PROCEDURE')) {
                    const match = statement.match(/CREATE PROCEDURE.*?`(\w+)`/i);
                    if (match) {
                        console.log(`✅ Created procedure: ${match[1]}`);
                        successCount++;
                    }
                }
            } catch (error) {
                // Ignore "already exists" errors
                if (!error.message.includes('already exists')) {
                    console.error(`❌ Error: ${error.message}`);
                    errorCount++;
                }
            }
        }

        connection.release();

        console.log('\n🎉 Migration completed!');
        console.log(`✅ Success: ${successCount}`);
        console.log(`❌ Errors: ${errorCount}`);

    } catch (error) {
        console.error('Migration failed:', error);
        process.exit(1);
    }
}

// Run migration if called directly
if (require.main === module) {
    runMigration().then(() => {
        console.log('\n✨ Done!');
        process.exit(0);
    });
}

module.exports = { runMigration };

