/**
 * Permission Helper Functions
 * Core permission checking logic with caching support
 */

const { Role, Permission, RolePermission, User } = require('../models/sequelize');

// Cache for permissions (in-memory, can be replaced with Redis)
const permissionCache = new Map();
const CACHE_TTL = 5 * 60 * 1000; // 5 minutes

/**
 * Check if user has a specific permission
 * @param {number} userId - User ID
 * @param {string} resource - Resource name (e.g., 'users', 'patients')
 * @param {string} action - Action name (e.g., 'create', 'read', 'update', 'delete')
 * @param {object} sessionData - Optional session data with cached permissions
 * @returns {Promise<boolean>} - True if user has permission
 */
async function hasPermission(userId, resource, action, sessionData = null) {
    try {
        // NOTE: Do not trust session-stored permissions here because they can
        // become stale after role updates. Always evaluate via cache/DB.

        // Do not use in-memory cache for permissions; it can go stale across role updates.
        const cacheKey = `${userId}:${resource}:${action}`;

        // Get user with role
        const user = await User.findByPk(userId, {
            include: [{
                model: Role,
                as: 'role',
                required: false // Allow users without roles to be found
            }]
        });

        if (!user || !user.role) {
            return false;
        }

        // Check if role is active
        if (!user.role.active) {
            return false;
        }

        // Admin role bypass (check metadata for bypassChecks flag)
        const roleMetadata = typeof user.role.metadata === 'string' 
            ? JSON.parse(user.role.metadata) 
            : user.role.metadata || {};
        
        const isAdminBypass = roleMetadata.bypassChecks === true ||
            (user.role.level >= 100 && String(user.role.name || '').toLowerCase() === 'admin');
        if (isAdminBypass) return true;

        // Check permission in database
        const permission = await Permission.findOne({
            where: {
                resource,
                action,
                active: true
            }
        });

        if (!permission) {
            return false;
        }

        // Check role-permission assignment
        const rolePermission = await RolePermission.findOne({
            where: {
                roleId: user.role.id,
                permissionId: permission.id,
                active: true,
                granted: true
            }
        });

        const hasPerm = !!rolePermission;

        return hasPerm;
    } catch (error) {
        console.error('Error checking permission:', error);
        return false; // Fail closed - deny access on error
    }
}

/**
 * Get all permissions for a user as a structured object
 * @param {number} userId - User ID
 * @param {object} sessionData - Optional session data
 * @returns {Promise<object>} - Object with structure {resource: {action: true/false}}
 */
async function getUserPermissions(userId, sessionData = null) {
    try {
        // Return cached permissions from session if available
        if (sessionData && sessionData.permissions) {
            return sessionData.permissions;
        }

        // Get user with role
        const user = await User.findByPk(userId, {
            include: [{
                model: Role,
                as: 'role',
                required: false, // Allow users without roles to be found
                include: [{
                    model: Permission,
                    as: 'permissions',
                    through: {
                        where: {
                            active: true,
                            granted: true
                        }
                    },
                    where: { active: true },
                    required: false
                }]
            }]
        });

        if (!user || !user.role || !user.role.active) {
            return {};
        }

        // Admin role gets all permissions
        const roleMetadata = typeof user.role.metadata === 'string' 
            ? JSON.parse(user.role.metadata) 
            : user.role.metadata || {};
        const isAdminBypass = roleMetadata.bypassChecks === true ||
            (user.role.level >= 100 && String(user.role.name || '').toLowerCase() === 'admin');
        
        if (isAdminBypass) {
            // Return all active permissions as granted
            const allPermissions = await Permission.findAll({
                where: { active: true }
            });

            const permissions = {};
            for (const perm of allPermissions) {
                if (!permissions[perm.resource]) {
                    permissions[perm.resource] = {};
                }
                permissions[perm.resource][perm.action] = true;
            }
            return permissions;
        }

        // Build permissions object from role permissions
        const permissions = {};
        for (const perm of user.role.permissions || []) {
            if (!permissions[perm.resource]) {
                permissions[perm.resource] = {};
            }
            permissions[perm.resource][perm.action] = true;
        }

        return permissions;
    } catch (error) {
        console.error('Error getting user permissions:', error);
        return {};
    }
}

/**
 * Clear permission cache for a user
 * @param {number} userId - User ID (optional, clears all if not provided)
 */
function clearPermissionCache(userId = null) {
    if (userId) {
        // Clear all cache entries for this user
        for (const key of permissionCache.keys()) {
            if (key.startsWith(`${userId}:`)) {
                permissionCache.delete(key);
            }
        }
    } else {
        // Clear all cache
        permissionCache.clear();
    }
}

/**
 * Check if user has any of the specified permissions
 * @param {number} userId - User ID
 * @param {Array<{resource: string, action: string}>} permissionList - Array of permission objects
 * @param {object} sessionData - Optional session data
 * @returns {Promise<boolean>} - True if user has at least one permission
 */
async function hasAnyPermission(userId, permissionList, sessionData = null) {
    for (const perm of permissionList) {
        if (await hasPermission(userId, perm.resource, perm.action, sessionData)) {
            return true;
        }
    }
    return false;
}

/**
 * Check if user has all of the specified permissions
 * @param {number} userId - User ID
 * @param {Array<{resource: string, action: string}>} permissionList - Array of permission objects
 * @param {object} sessionData - Optional session data
 * @returns {Promise<boolean>} - True if user has all permissions
 */
async function hasAllPermissions(userId, permissionList, sessionData = null) {
    for (const perm of permissionList) {
        if (!(await hasPermission(userId, perm.resource, perm.action, sessionData))) {
            return false;
        }
    }
    return true;
}

module.exports = {
    hasPermission,
    getUserPermissions,
    clearPermissionCache,
    hasAnyPermission,
    hasAllPermissions
};

