/**
 * Comprehensive EMR and Vitals Seeding - 30 New Records
 * Seeds 30 realistic patient cases with vital signs and treatment records
 */

const { query } = require('../config/database');

async function seed30EMRVitals() {
    try {
        console.log('🏥 Seeding 30 New EMR and Vitals Records...\n');

        // First, add 30 new patients (both OPD and IPD)
        console.log('Creating 30 new patients...');
        await query(`INSERT IGNORE INTO tbl_patient (name, age, phone, address, case_, remark, date, dr_name, patient_id) VALUES 
            ('Ma Nwe Nwe', '34', '09-123456701', 'Thuwanna Township, Yangon', 'Type 2 Diabetes', 'Newly diagnosed, needs education', CURDATE(), 'Dr. Aung Aung', 'OPD-016'),
            ('U Maung Htun', '67', '09-123456702', 'Pazundaung Township, Yangon', 'COPD Exacerbation', 'Chronic smoker, oxygen therapy', CURDATE(), 'Dr. Myo Myo', 'OPD-017'),
            ('Daw Kyi Kyi', '29', '09-123456703', 'Lanmadaw Township, Yangon', 'Postpartum Depression', 'Delivered 3 months ago', CURDATE(), 'Dr. Thida', 'OPD-018'),
            ('Ko Thant Sin', '41', '09-123456704', 'Latha Township, Yangon', 'Chronic Kidney Disease', 'Stage 3 CKD', CURDATE(), 'Dr. Aung Aung', 'OPD-019'),
            ('Ma Phyo Wai', '26', '09-123456705', 'Seikgyi Khanaungto Township, Yangon', 'Iron Deficiency Anemia', 'Heavy menstrual bleeding', CURDATE(), 'Dr. Thida', 'OPD-020'),
            ('U Kyaw Min', '58', '09-123456706', 'Ahlone Township, Yangon', 'Atrial Fibrillation', 'Irregular heartbeat', CURDATE(), 'Dr. Myo Myo', 'OPD-021'),
            ('Daw Thin Thin', '45', '09-123456707', 'Kyauktada Township, Yangon', 'Cervical Spondylosis', 'Neck pain and stiffness', CURDATE(), 'Dr. Aung Aung', 'OPD-022'),
            ('Ko Naing Win', '33', '09-123456708', 'Pabedan Township, Yangon', 'Peptic Ulcer Disease', 'H. pylori positive', CURDATE(), 'Dr. Thida', 'OPD-023'),
            ('Ma Swe Zin', '22', '09-123456709', 'Kyimyindaing Township, Yangon', 'Polycystic Ovary Syndrome', 'Irregular periods', CURDATE(), 'Dr. Thida', 'OPD-024'),
            ('U Htun Lwin', '52', '09-123456710', 'Tamwe Township, Yangon', 'Gout', 'Acute attack, right big toe', CURDATE(), 'Dr. Myo Myo', 'OPD-025'),
            ('Daw Moe Moe', '39', '09-123456711', 'Dawbon Township, Yangon', 'Fibromyalgia', 'Chronic widespread pain', CURDATE(), 'Dr. Aung Aung', 'OPD-026'),
            ('Ko Zin Min', '28', '09-123456712', 'Mingalar Taung Nyunt Township, Yangon', 'Anxiety Disorder', 'Panic attacks', CURDATE(), 'Dr. Thida', 'OPD-027'),
            ('Ma Htet Htet', '36', '09-123456713', 'Thaketa Township, Yangon', 'Gestational Diabetes', '32 weeks pregnant', CURDATE(), 'Dr. Thida', 'OPD-028'),
            ('U Win Maung', '61', '09-123456714', 'Botataung Township, Yangon', 'Benign Prostatic Hyperplasia', 'Urinary retention', CURDATE(), 'Dr. Myo Myo', 'OPD-029'),
            ('Daw Aye Aye', '43', '09-123456715', 'Dagon Township, Yangon', 'Rheumatoid Arthritis', 'Joint swelling and pain', CURDATE(), 'Dr. Aung Aung', 'OPD-030'),
            
            -- IPD Patients
            ('U Thura', '47', '09-123456716', 'Sanchaung Township, Yangon', 'Acute Myocardial Infarction', 'STEMI, emergency PCI', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Myo Myo', 'IPD-011'),
            ('Daw Mya Thandar', '35', '09-123456717', 'Kamayut Township, Yangon', 'Severe Preeclampsia', '36 weeks, emergency C-section', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Thida', 'IPD-012'),
            ('Ko Aung Myo', '29', '09-123456718', 'Bahan Township, Yangon', 'Acute Pancreatitis', 'Alcohol-induced', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'Dr. Aung Aung', 'IPD-013'),
            ('Ma Khin Myo', '24', '09-123456719', 'Yankin Township, Yangon', 'Diabetic Ketoacidosis', 'Type 1 diabetes', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Thida', 'IPD-014'),
            ('U San Tun', '68', '09-123456720', 'Hlaing Township, Yangon', 'Acute Respiratory Failure', 'COPD exacerbation', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'Dr. Myo Myo', 'IPD-015'),
            ('Daw Nyo Nyo', '53', '09-123456721', 'Thingangyun Township, Yangon', 'Acute Cholangitis', 'CBD stone', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Aung Aung', 'IPD-016'),
            ('Ko Yan Naing', '31', '09-123456722', 'Mayangone Township, Yangon', 'Spinal Cord Injury', 'Motor vehicle accident', DATE_SUB(CURDATE(), INTERVAL 5 DAY), 'Dr. Myo Myo', 'IPD-017'),
            ('Ma Thaw Dar', '27', '09-123456723', 'Dagon Township, Yangon', 'Postpartum Hemorrhage', 'Emergency hysterectomy', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Thida', 'IPD-018'),
            ('U Zaw Min', '56', '09-123456724', 'Botataung Township, Yangon', 'Acute Kidney Injury', 'Contrast-induced', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Aung Aung', 'IPD-019'),
            ('Daw Than Than', '42', '09-123456725', 'Insein Township, Yangon', 'Status Epilepticus', 'Breakthrough seizures', CURDATE(), 'Dr. Myo Myo', 'IPD-020'),
            ('Ko Thant Zin', '38', '09-123456726', 'Thaketa Township, Yangon', 'Acute Hepatitis', 'Drug-induced liver injury', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'Dr. Thida', 'IPD-021'),
            ('Ma Yamin', '33', '09-123456727', 'Mingalar Taung Nyunt Township, Yangon', 'Pulmonary Embolism', 'High-risk PE', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Myo Myo', 'IPD-022'),
            ('U Thein Zaw', '49', '09-123456728', 'Pazundaung Township, Yangon', 'Gastrointestinal Bleeding', 'Upper GI bleed', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Aung Aung', 'IPD-023'),
            ('Daw Ei Ei', '37', '09-123456729', 'Dawbon Township, Yangon', 'Severe Asthma Exacerbation', 'Status asthmaticus', CURDATE(), 'Dr. Thida', 'IPD-024'),
            ('Ko Naing Oo', '44', '09-123456730', 'Tamwe Township, Yangon', 'Intracranial Hemorrhage', 'Hypertensive hemorrhage', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'Dr. Myo Myo', 'IPD-025')`);
        console.log('✅ 30 new patients added');

        // Add corresponding IPD records for inpatients
        console.log('\nAdding IPD records for inpatients...');
        await query(`INSERT IGNORE INTO tbl_inpatient (name, age, phone, address, case_, remark, start_date, dr_name, patient_id, room_no, deposit) VALUES 
            ('U Thura', '47', '09-123456716', 'Sanchaung Township, Yangon', 'Acute Myocardial Infarction', 'STEMI, emergency PCI done', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Myo Myo', 'IPD-011', 'R-301', 2000000),
            ('Daw Mya Thandar', '35', '09-123456717', 'Kamayut Township, Yangon', 'Severe Preeclampsia', 'Emergency C-section, ICU monitoring', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Thida', 'IPD-012', 'R-302', 1800000),
            ('Ko Aung Myo', '29', '09-123456718', 'Bahan Township, Yangon', 'Acute Pancreatitis', 'Alcohol-induced, severe pain', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'Dr. Aung Aung', 'IPD-013', 'R-303', 1200000),
            ('Ma Khin Myo', '24', '09-123456719', 'Yankin Township, Yangon', 'Diabetic Ketoacidosis', 'Type 1 diabetes, new diagnosis', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Thida', 'IPD-014', 'R-304', 800000),
            ('U San Tun', '68', '09-123456720', 'Hlaing Township, Yangon', 'Acute Respiratory Failure', 'COPD exacerbation, ventilator support', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'Dr. Myo Myo', 'IPD-015', 'R-305', 1500000),
            ('Daw Nyo Nyo', '53', '09-123456721', 'Thingangyun Township, Yangon', 'Acute Cholangitis', 'CBD stone, ERCP scheduled', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Aung Aung', 'IPD-016', 'R-306', 1000000),
            ('Ko Yan Naing', '31', '09-123456722', 'Mayangone Township, Yangon', 'Spinal Cord Injury', 'T12 complete injury, MVA', DATE_SUB(CURDATE(), INTERVAL 5 DAY), 'Dr. Myo Myo', 'IPD-017', 'R-307', 2500000),
            ('Ma Thaw Dar', '27', '09-123456723', 'Dagon Township, Yangon', 'Postpartum Hemorrhage', 'Emergency hysterectomy done', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Thida', 'IPD-018', 'R-308', 1600000),
            ('U Zaw Min', '56', '09-123456724', 'Botataung Township, Yangon', 'Acute Kidney Injury', 'Contrast-induced nephropathy', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Aung Aung', 'IPD-019', 'R-309', 900000),
            ('Daw Than Than', '42', '09-123456725', 'Insein Township, Yangon', 'Status Epilepticus', 'Breakthrough seizures, ICU', CURDATE(), 'Dr. Myo Myo', 'IPD-020', 'R-310', 1300000),
            ('Ko Thant Zin', '38', '09-123456726', 'Thaketa Township, Yangon', 'Acute Hepatitis', 'Drug-induced liver injury', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'Dr. Thida', 'IPD-021', 'R-311', 700000),
            ('Ma Yamin', '33', '09-123456727', 'Mingalar Taung Nyunt Township, Yangon', 'Pulmonary Embolism', 'High-risk PE, thrombolysis', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Myo Myo', 'IPD-022', 'R-312', 1100000),
            ('U Thein Zaw', '49', '09-123456728', 'Pazundaung Township, Yangon', 'Gastrointestinal Bleeding', 'Upper GI bleed, endoscopy done', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Aung Aung', 'IPD-023', 'R-313', 800000),
            ('Daw Ei Ei', '37', '09-123456729', 'Dawbon Township, Yangon', 'Severe Asthma Exacerbation', 'Status asthmaticus, ICU', CURDATE(), 'Dr. Thida', 'IPD-024', 'R-314', 1400000),
            ('Ko Naing Oo', '44', '09-123456730', 'Tamwe Township, Yangon', 'Intracranial Hemorrhage', 'Hypertensive hemorrhage, surgery', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'Dr. Myo Myo', 'IPD-025', 'R-315', 3000000)`);
        console.log('✅ IPD records added');

        // Add comprehensive vital signs for all 30 patients (multiple readings for chronic conditions)
        console.log('\nSeeding comprehensive vital signs data...');
        await query(`INSERT IGNORE INTO tbl_patient_vitals 
            (patient_id, blood_pressure_systolic, blood_pressure_diastolic, heart_rate, temperature, respiratory_rate, oxygen_saturation, weight, height, bmi, blood_glucose, recorded_by, recorded_date, notes) 
            VALUES 
            -- OPD-016 (Type 2 Diabetes) - 6 month monitoring
            ('OPD-016', 145, 92, 78, 98.2, 18, 98, 68.5, 158, 27.4, 220, 'Nurse May', DATE_SUB(NOW(), INTERVAL 180 DAY), 'Initial diagnosis - High glucose'),
            ('OPD-016', 142, 90, 76, 98.0, 17, 98, 67.8, 158, 27.2, 185, 'Nurse May', DATE_SUB(NOW(), INTERVAL 150 DAY), 'Month 1 - Metformin started'),
            ('OPD-016', 138, 88, 75, 98.1, 18, 98, 67.0, 158, 26.8, 160, 'Nurse May', DATE_SUB(NOW(), INTERVAL 120 DAY), 'Month 2 - Good response'),
            ('OPD-016', 135, 85, 74, 98.0, 17, 99, 66.5, 158, 26.6, 145, 'Nurse May', DATE_SUB(NOW(), INTERVAL 90 DAY), 'Month 3 - HbA1c 8.2%'),
            ('OPD-016', 132, 82, 73, 97.9, 18, 98, 66.0, 158, 26.4, 135, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'Month 4 - Insulin added'),
            ('OPD-016', 130, 80, 72, 98.0, 17, 99, 65.5, 158, 26.2, 125, 'Nurse May', NOW(), 'Month 6 - Well controlled'),
            
            -- OPD-017 (COPD) - Respiratory monitoring
            ('OPD-017', 135, 80, 88, 98.5, 22, 92, 58.0, 165, 21.3, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 90 DAY), 'COPD exacerbation'),
            ('OPD-017', 132, 78, 85, 98.2, 20, 94, 58.0, 165, 21.3, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'Improving with bronchodilators'),
            ('OPD-017', 130, 76, 82, 98.0, 18, 96, 58.0, 165, 21.3, NULL, 'Nurse May', NOW(), 'Stable on home O2'),
            
            -- OPD-018 (Postpartum Depression) - General monitoring
            ('OPD-018', 115, 75, 88, 98.0, 16, 99, 65.0, 160, 25.4, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Postpartum visit'),
            ('OPD-018', 118, 76, 85, 98.1, 17, 99, 63.5, 160, 24.8, NULL, 'Nurse May', NOW(), 'Mood improving with therapy'),
            
            -- OPD-019 (CKD Stage 3) - Kidney function monitoring  
            ('OPD-019', 150, 90, 75, 98.0, 18, 98, 72.0, 170, 24.9, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 120 DAY), 'CKD Stage 3 - Creatinine 2.1'),
            ('OPD-019', 145, 88, 74, 98.1, 17, 98, 71.5, 170, 24.7, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 90 DAY), 'ACE inhibitor started'),
            ('OPD-019', 140, 85, 73, 97.9, 18, 99, 71.0, 170, 24.6, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'BP improving'),
            ('OPD-019', 135, 82, 72, 98.0, 17, 98, 70.5, 170, 24.4, NULL, 'Nurse May', NOW(), 'Stable CKD'),
            
            -- OPD-020 (Iron Deficiency Anemia) - Follow-up
            ('OPD-020', 110, 65, 95, 97.8, 18, 98, 52.0, 155, 21.6, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'Severe anemia - Hb 6.2'),
            ('OPD-020', 115, 70, 88, 98.0, 17, 99, 53.5, 155, 22.2, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Iron therapy started'),
            ('OPD-020', 120, 75, 80, 98.1, 18, 99, 55.0, 155, 22.9, NULL, 'Nurse May', NOW(), 'Hb improving to 9.8'),
            
            -- OPD-021 (Atrial Fibrillation) - Cardiac monitoring
            ('OPD-021', 140, 85, 105, 98.0, 18, 98, 75.0, 168, 26.6, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'AF with RVR'),
            ('OPD-021', 135, 82, 85, 98.1, 17, 98, 74.5, 168, 26.4, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Rate controlled with metoprolol'),
            ('OPD-021', 130, 80, 72, 98.0, 18, 99, 74.0, 168, 26.2, NULL, 'Nurse May', NOW(), 'AF controlled, on warfarin'),
            
            -- OPD-022 (Cervical Spondylosis) - Pain management
            ('OPD-022', 125, 78, 72, 98.0, 17, 99, 58.0, 162, 22.1, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Neck pain and stiffness'),
            ('OPD-022', 122, 76, 70, 98.1, 18, 99, 58.0, 162, 22.1, NULL, 'Nurse May', NOW(), 'Improved with physiotherapy'),
            
            -- OPD-023 (H. pylori Gastritis) - GI monitoring
            ('OPD-023', 120, 75, 74, 98.2, 18, 98, 65.0, 168, 23.0, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 45 DAY), 'Epigastric pain'),
            ('OPD-023', 118, 74, 72, 98.0, 17, 99, 65.0, 168, 23.0, NULL, 'Nurse May', NOW(), 'H. pylori eradicated'),
            
            -- OPD-024 (PCOS) - Hormonal monitoring
            ('OPD-024', 125, 80, 78, 98.0, 17, 99, 72.0, 165, 26.4, 110, 'Nurse May', DATE_SUB(NOW(), INTERVAL 90 DAY), 'PCOS diagnosis'),
            ('OPD-024', 122, 78, 76, 98.1, 18, 99, 70.5, 165, 25.9, 105, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'Metformin started'),
            ('OPD-024', 120, 76, 74, 98.0, 17, 99, 69.0, 165, 25.4, 95, 'Nurse May', NOW(), 'Weight loss, regular cycles'),
            
            -- OPD-025 (Gout) - Acute and follow-up
            ('OPD-025', 145, 88, 78, 98.5, 18, 98, 78.0, 170, 27.0, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 14 DAY), 'Acute gout attack'),
            ('OPD-025', 135, 82, 74, 98.0, 17, 99, 77.5, 170, 26.8, NULL, 'Nurse May', NOW(), 'Attack resolved, allopurinol started'),
            
            -- OPD-026 (Fibromyalgia) - Pain monitoring
            ('OPD-026', 120, 75, 76, 98.0, 18, 99, 62.0, 160, 24.2, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Widespread pain'),
            ('OPD-026', 118, 74, 74, 98.1, 17, 99, 62.0, 160, 24.2, NULL, 'Nurse May', NOW(), 'Pain improved with pregabalin'),
            
            -- OPD-027 (Anxiety) - Psychiatric monitoring
            ('OPD-027', 130, 82, 88, 98.0, 20, 99, 68.0, 172, 23.0, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Panic attacks'),
            ('OPD-027', 125, 78, 78, 98.1, 18, 99, 68.0, 172, 23.0, NULL, 'Nurse May', NOW(), 'Anxiety controlled with therapy'),
            
            -- OPD-028 (Gestational Diabetes) - Pregnancy monitoring
            ('OPD-028', 125, 78, 82, 98.0, 18, 99, 72.0, 162, 27.4, 145, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), '32 weeks - GDM diagnosed'),
            ('OPD-028', 122, 76, 80, 98.1, 17, 99, 73.5, 162, 28.0, 125, 'Nurse May', NOW(), '36 weeks - Well controlled'),
            
            -- OPD-029 (BPH) - Urological monitoring
            ('OPD-029', 140, 85, 72, 98.0, 17, 98, 78.0, 168, 27.6, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'Urinary retention'),
            ('OPD-029', 135, 82, 70, 98.1, 18, 99, 77.5, 168, 27.5, NULL, 'Nurse May', NOW(), 'Improved with tamsulosin'),
            
            -- OPD-030 (Rheumatoid Arthritis) - Rheumatology monitoring
            ('OPD-030', 130, 80, 74, 98.2, 18, 99, 58.0, 158, 23.2, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 90 DAY), 'RA flare - multiple joints'),
            ('OPD-030', 125, 76, 72, 98.0, 17, 99, 58.5, 158, 23.4, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'MTX started'),
            ('OPD-030', 122, 74, 70, 98.1, 18, 99, 59.0, 158, 23.6, NULL, 'Nurse May', NOW(), 'RA in remission'),
            
            -- IPD PATIENTS VITAL SIGNS
            -- IPD-011 (MI) - ICU monitoring
            ('IPD-011', 160, 95, 105, 98.0, 20, 96, 72.0, 170, 24.9, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 2 DAY), 'Admission - STEMI'),
            ('IPD-011', 140, 85, 90, 98.1, 18, 98, 72.0, 170, 24.9, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'Post-PCI day 1'),
            ('IPD-011', 130, 78, 78, 98.0, 17, 99, 72.0, 170, 24.9, NULL, 'Nurse May', NOW(), 'Stable - ready for discharge'),
            
            -- IPD-012 (Preeclampsia) - Obstetric ICU
            ('IPD-012', 170, 110, 110, 98.5, 20, 98, 78.0, 160, 30.5, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'Severe preeclampsia'),
            ('IPD-012', 145, 88, 85, 98.0, 18, 99, 78.0, 160, 30.5, NULL, 'Nurse May', NOW(), 'Post C-section - BP controlled'),
            
            -- IPD-013 (Pancreatitis) - Medical ward
            ('IPD-013', 125, 78, 95, 99.2, 22, 98, 65.0, 172, 22.0, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 3 DAY), 'Severe abdominal pain'),
            ('IPD-013', 120, 75, 88, 98.5, 20, 98, 65.0, 172, 22.0, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 2 DAY), 'Pain improving'),
            ('IPD-013', 118, 74, 82, 98.0, 18, 99, 65.0, 172, 22.0, NULL, 'Nurse May', NOW(), 'Tolerating oral feeds'),
            
            -- IPD-014 (DKA) - ICU monitoring
            ('IPD-014', 115, 70, 120, 99.8, 28, 99, 48.0, 155, 20.0, 450, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'DKA - pH 7.1'),
            ('IPD-014', 120, 75, 90, 98.2, 20, 99, 48.0, 155, 20.0, 180, 'ICU Nurse', NOW(), 'DKA resolved'),
            
            -- IPD-015 (Respiratory Failure) - ICU ventilator
            ('IPD-015', 130, 75, 110, 98.0, 25, 88, 55.0, 168, 19.5, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 4 DAY), 'Ventilator support'),
            ('IPD-015', 125, 72, 95, 98.1, 22, 92, 55.0, 168, 19.5, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 2 DAY), 'Weaning ventilator'),
            ('IPD-015', 122, 70, 88, 98.0, 20, 95, 55.0, 168, 19.5, NULL, 'Nurse May', NOW(), 'Extubated - stable'),
            
            -- IPD-016 (Cholangitis) - Medical ICU
            ('IPD-016', 110, 65, 105, 101.5, 22, 97, 58.0, 158, 23.2, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 2 DAY), 'Sepsis with cholangitis'),
            ('IPD-016', 118, 72, 88, 99.0, 20, 98, 58.0, 158, 23.2, NULL, 'ICU Nurse', NOW(), 'Post-ERCP - improving'),
            
            -- IPD-017 (Spinal Injury) - Trauma ICU
            ('IPD-017', 120, 75, 78, 98.0, 18, 99, 68.0, 175, 22.2, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 5 DAY), 'T12 complete SCI'),
            ('IPD-017', 118, 74, 76, 98.1, 17, 99, 68.0, 175, 22.2, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 3 DAY), 'Spinal shock resolved'),
            ('IPD-017', 115, 72, 74, 98.0, 18, 99, 68.0, 175, 22.2, NULL, 'Nurse May', NOW(), 'Rehab started'),
            
            -- IPD-018 (Postpartum Hemorrhage) - Obstetric ICU
            ('IPD-018', 90, 50, 125, 97.5, 24, 96, 55.0, 158, 22.0, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'Post-hysterectomy'),
            ('IPD-018', 110, 65, 88, 98.0, 20, 98, 55.0, 158, 22.0, NULL, 'Nurse May', NOW(), 'Hemodynamically stable'),
            
            -- IPD-019 (AKI) - Nephrology ward
            ('IPD-019', 145, 88, 82, 98.2, 20, 98, 68.0, 168, 24.1, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 2 DAY), 'Contrast nephropathy'),
            ('IPD-019', 135, 82, 78, 98.0, 18, 99, 68.0, 168, 24.1, NULL, 'Nurse May', NOW(), 'Creatinine improving'),
            
            -- IPD-020 (Status Epilepticus) - Neuro ICU
            ('IPD-020', 125, 78, 95, 98.5, 20, 98, 62.0, 160, 24.2, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'Breakthrough seizures'),
            ('IPD-020', 120, 75, 78, 98.0, 18, 99, 62.0, 160, 24.2, NULL, 'ICU Nurse', NOW(), 'Seizure-free 24 hours'),
            
            -- Additional vitals for remaining IPD patients
            ('IPD-021', 115, 70, 88, 99.5, 20, 97, 65.0, 170, 22.5, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 3 DAY), 'Drug-induced hepatitis'),
            ('IPD-021', 118, 72, 82, 98.2, 18, 98, 65.0, 170, 22.5, NULL, 'Nurse May', NOW(), 'Liver function improving'),
            
            ('IPD-022', 125, 75, 108, 98.0, 24, 92, 65.0, 162, 24.8, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 2 DAY), 'High-risk PE'),
            ('IPD-022', 120, 72, 85, 98.1, 20, 96, 65.0, 162, 24.8, NULL, 'Nurse May', NOW(), 'Thrombolysis successful'),
            
            ('IPD-023', 105, 60, 95, 98.0, 20, 96, 60.0, 165, 22.0, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'Upper GI bleeding'),
            ('IPD-023', 118, 72, 78, 98.1, 18, 98, 60.0, 165, 22.0, NULL, 'Nurse May', NOW(), 'Bleeding controlled'),
            
            ('IPD-024', 130, 80, 115, 98.5, 28, 90, 58.0, 155, 24.1, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'Status asthmaticus'),
            ('IPD-024', 125, 76, 88, 98.0, 20, 96, 58.0, 155, 24.1, NULL, 'Nurse May', NOW(), 'Responding to treatment'),
            
            ('IPD-025', 180, 105, 95, 98.0, 20, 98, 75.0, 172, 25.4, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 4 DAY), 'ICH - surgery done'),
            ('IPD-025', 150, 88, 82, 98.1, 18, 99, 75.0, 172, 25.4, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 2 DAY), 'Post-op day 2'),
            ('IPD-025', 135, 78, 76, 98.0, 17, 99, 75.0, 172, 25.4, NULL, 'Nurse May', NOW(), 'Neurologically stable')`);
        console.log('✅ Comprehensive vital signs seeded (80+ records)');

        // Add comprehensive EMR treatment records
        console.log('\nSeeding comprehensive EMR treatment records...');
        await query(`INSERT IGNORE INTO tbl_treatment (patient_id, doctor_id, chief_complaint, present_illness, Past_Diagnoses, Surgeries_Procedures, Hospitalizations, Current_Medications, Dosages, Allergen, Treatment_Type, diagnosis, prescription, Date, Voucher_id, Recommendation, vital_signs, notes) VALUES 
            -- OPD Treatment Records
            ('OPD-016', 1, 'Excessive thirst and urination', 'Patient reports polyuria, polydipsia, and weight loss over 3 months. Random glucose 220 mg/dL', 'Newly diagnosed Type 2 Diabetes Mellitus', 'None', 'None', 'Metformin, Glimepiride', '500mg BD with meals, 2mg OD before breakfast', 'None Known', 'Consultation', 'Type 2 Diabetes Mellitus - newly diagnosed', '1. Metformin 500mg BD\\n2. Glimepiride 2mg OD\\n3. Lifestyle counseling provided', CURDATE(), 'V-3001', 'Diabetic diet education, home glucose monitoring, follow-up in 2 weeks, HbA1c in 6 weeks', '{"bp":"145/92","temp":"98.2","pulse":"78","weight":"68.5","height":"158","bmi":"27.4","glucose":"220"}', 'New diabetic - needs comprehensive education'),
            
            ('OPD-017', 2, 'Shortness of breath and chronic cough', 'Long-standing smoker presents with worsening dyspnea and productive cough. FEV1/FVC < 0.7', 'COPD Gold Stage 3, Chronic bronchitis', 'None', 'Previous admission for COPD exacerbation 2 years ago', 'Salbutamol MDI, Tiotropium, Prednisolone', '2 puffs QID PRN, 18mcg OD, 30mg OD x 5 days', 'None Known', 'Consultation', 'COPD exacerbation - moderate', '1. Salbutamol MDI 2 puffs QID\\n2. Tiotropium 18mcg OD\\n3. Prednisolone 30mg OD x 5 days\\n4. Antibiotics if purulent sputum', DATE_SUB(CURDATE(), INTERVAL 7 DAY), 'V-3002', 'Smoking cessation counseling, pulmonary rehabilitation, home oxygen therapy, flu vaccination', '{"bp":"135/80","temp":"98.5","pulse":"88","weight":"58.0","respiratory":"22","spo2":"92"}', 'Long-term smoker - needs aggressive treatment'),
            
            ('OPD-018', 3, 'Feeling sad and anxious since delivery', 'Delivered 3 months ago, experiencing persistent sadness, anxiety, difficulty bonding with baby', 'Postpartum Depression', 'Normal vaginal delivery 3 months ago', 'None', 'Sertraline, Counseling sessions', '50mg OD, Weekly sessions', 'None Known', 'Consultation', 'Postpartum Depression - moderate', '1. Sertraline 50mg OD\\n2. Counseling referral\\n3. Support group information', DATE_SUB(CURDATE(), INTERVAL 14 DAY), 'V-3003', 'Regular psychiatric follow-up, family support, breastfeeding safe medications, crisis contact information', '{"bp":"115/75","temp":"98.0","pulse":"88","weight":"65.0"}', 'Needs close psychiatric follow-up'),
            
            ('OPD-019', 1, 'Fatigue and decreased urine output', 'Progressive decline in kidney function over 2 years. Creatinine 2.1 mg/dL, eGFR 35', 'Chronic Kidney Disease Stage 3', 'None', 'None', 'Lisinopril, Furosemide, Calcium carbonate', '10mg OD, 40mg OD, 500mg TID with meals', 'None Known', 'Consultation', 'CKD Stage 3 - progressive', '1. Lisinopril 10mg OD\\n2. Furosemide 40mg OD\\n3. Calcium carbonate 500mg TID\\n4. Nephrology referral', DATE_SUB(CURDATE(), INTERVAL 30 DAY), 'V-3004', 'Protein restriction, phosphorus binder, avoid nephrotoxic drugs, monitor electrolytes monthly', '{"bp":"150/90","temp":"98.0","pulse":"75","weight":"72.0"}', 'Progressive CKD - nephrology referral needed'),
            
            ('OPD-020', 3, 'Fatigue and heavy menstrual periods', 'Iron deficiency anemia secondary to menorrhagia. Hb 6.2 g/dL', 'Iron Deficiency Anemia - severe', 'None', 'None', 'Iron sulfate, Vitamin C, Tranexamic acid', '325mg TID, 500mg TID, 500mg TID during menses', 'None Known', 'Consultation', 'Iron deficiency anemia - severe', '1. Iron sulfate 325mg TID\\n2. Vitamin C 500mg TID\\n3. Tranexamic acid 500mg TID during menses\\n4. Gynecology referral', DATE_SUB(CURDATE(), INTERVAL 45 DAY), 'V-3005', 'Iron-rich diet, avoid tea/coffee with iron, gynecology evaluation for menorrhagia, Hb check in 4 weeks', '{"bp":"110/65","temp":"97.8","pulse":"95","weight":"52.0"}', 'Severe anemia - may need transfusion'),
            
            ('OPD-021', 2, 'Irregular heartbeat and palpitations', 'Newly diagnosed atrial fibrillation with rapid ventricular response', 'Atrial Fibrillation', 'None', 'None', 'Metoprolol, Warfarin', '50mg BD, 5mg OD (adjust per INR)', 'None Known', 'Consultation', 'Atrial fibrillation - newly diagnosed', '1. Metoprolol 50mg BD\\n2. Warfarin 5mg OD\\n3. INR monitoring\\n4. Cardiology referral', DATE_SUB(CURDATE(), INTERVAL 21 DAY), 'V-3006', 'INR monitoring every 3 days initially, then weekly when stable, cardiology follow-up, stroke risk assessment', '{"bp":"140/85","temp":"98.0","pulse":"105","weight":"75.0"}', 'New AF - needs anticoagulation monitoring'),
            
            ('OPD-022', 1, 'Neck pain and stiffness', 'Chronic neck pain with radiation to shoulders, worse with movement', 'Cervical Spondylosis', 'None', 'None', 'Diclofenac, Muscle relaxant, Physiotherapy', '75mg BD, 1 tab TID, 3 sessions/week', 'None Known', 'Consultation', 'Cervical spondylosis with radiculopathy', '1. Diclofenac 75mg BD\\n2. Muscle relaxant 1 tab TID\\n3. Physiotherapy prescription\\n4. Cervical collar PRN', DATE_SUB(CURDATE(), INTERVAL 10 DAY), 'V-3007', 'Physiotherapy 3x/week, ergonomic workplace setup, heat therapy, avoid prolonged neck flexion', '{"bp":"125/78","temp":"98.0","pulse":"72","weight":"58.0"}', 'Chronic condition - long-term management needed'),
            
            ('OPD-023', 3, 'Stomach pain and heartburn', 'Epigastric pain, postprandial fullness, H. pylori positive on testing', 'H. pylori gastritis', 'None', 'None', 'Triple therapy: Omeprazole, Amoxicillin, Clarithromycin', '20mg BD, 1g BD, 500mg BD - all for 14 days', 'None Known', 'Consultation', 'H. pylori positive gastritis', '1. Omeprazole 20mg BD x 14 days\\n2. Amoxicillin 1g BD x 14 days\\n3. Clarithromycin 500mg BD x 14 days\\n4. Eradication test in 4 weeks', DATE_SUB(CURDATE(), INTERVAL 28 DAY), 'V-3008', 'Complete full course of antibiotics, avoid spicy foods, H. pylori breath test 4 weeks after completion', '{"bp":"120/75","temp":"98.2","pulse":"74","weight":"65.0"}', 'H. pylori eradication therapy - compliance important'),
            
            ('OPD-024', 3, 'Irregular periods and weight gain', 'PCOS with irregular cycles, hirsutism, and insulin resistance', 'Polycystic Ovary Syndrome', 'None', 'None', 'Metformin, Oral contraceptives, Spironolactone', '500mg BD, 1 tab OD, 25mg BD', 'None Known', 'Consultation', 'PCOS with metabolic syndrome', '1. Metformin 500mg BD\\n2. Combined OCP 1 tab OD\\n3. Spironolactone 25mg BD\\n4. Lifestyle counseling', DATE_SUB(CURDATE(), INTERVAL 60 DAY), 'V-3009', 'Weight loss program, regular exercise, low glycemic diet, monitor for diabetes, endocrine follow-up', '{"bp":"125/80","temp":"98.0","pulse":"78","weight":"72.0","glucose":"110"}', 'PCOS - comprehensive management needed'),
            
            ('OPD-025', 2, 'Severe toe pain and swelling', 'Acute onset severe pain in right big toe, red and swollen', 'Acute gouty arthritis', 'None', 'None', 'Colchicine, Allopurinol', '0.6mg BD x 3 days then OD, 100mg OD (start after acute attack)', 'None Known', 'Consultation', 'Acute gout - first episode', '1. Colchicine 0.6mg BD x 3 days\\n2. Allopurinol 100mg OD (start in 2 weeks)\\n3. Low purine diet\\n4. Uric acid level monitoring', DATE_SUB(CURDATE(), INTERVAL 7 DAY), 'V-3010', 'Avoid high purine foods, adequate hydration, weight reduction, alcohol moderation, uric acid target <6mg/dL', '{"bp":"145/88","temp":"98.5","pulse":"78","weight":"78.0"}', 'First gout attack - lifestyle counseling important'),
            
            ('OPD-026', 1, 'Widespread muscle pain and fatigue', 'Chronic widespread pain, tender points, fatigue, sleep disturbance', 'Fibromyalgia', 'None', 'None', 'Pregabalin, Duloxetine, Sleep hygiene', '75mg BD, 30mg OD, Education provided', 'None Known', 'Consultation', 'Fibromyalgia syndrome', '1. Pregabalin 75mg BD\\n2. Duloxetine 30mg OD\\n3. Sleep hygiene education\\n4. Gentle exercise program', DATE_SUB(CURDATE(), INTERVAL 21 DAY), 'V-3011', 'Gradual exercise program, stress management, sleep hygiene, pain management techniques, support groups', '{"bp":"120/75","temp":"98.0","pulse":"76","weight":"62.0"}', 'Chronic pain condition - multidisciplinary approach'),
            
            ('OPD-027', 3, 'Anxiety and panic attacks', 'Generalized anxiety with panic episodes, affecting daily activities', 'Generalized Anxiety Disorder with panic', 'None', 'None', 'Sertraline, Lorazepam PRN, CBT', '50mg OD, 0.5mg PRN (max 2/day), Weekly sessions', 'None Known', 'Consultation', 'Generalized anxiety disorder', '1. Sertraline 50mg OD\\n2. Lorazepam 0.5mg PRN\\n3. CBT referral\\n4. Relaxation techniques', DATE_SUB(CURDATE(), INTERVAL 35 DAY), 'V-3012', 'Cognitive behavioral therapy, stress management, avoid caffeine, regular sleep schedule, gradual exposure therapy', '{"bp":"130/82","temp":"98.0","pulse":"88","weight":"68.0"}', 'Anxiety disorder - needs psychiatric follow-up'),
            
            ('OPD-028', 3, 'High blood sugar in pregnancy', '32 weeks pregnant, gestational diabetes on OGTT, diet-controlled initially', 'Gestational Diabetes Mellitus', 'None', 'None', 'Insulin Aspart, Folic acid, Iron', '4 units TID before meals, 5mg OD, 60mg OD', 'None Known', 'Consultation', 'Gestational diabetes - insulin requiring', '1. Insulin Aspart 4 units TID\\n2. Continue folic acid 5mg OD\\n3. Iron 60mg OD\\n4. Home glucose monitoring', DATE_SUB(CURDATE(), INTERVAL 14 DAY), 'V-3013', 'Diabetic diet in pregnancy, home glucose monitoring QID, obstetric follow-up weekly, fetal monitoring', '{"bp":"125/78","temp":"98.0","pulse":"82","weight":"72.0","glucose":"145"}', '32 weeks pregnant - close monitoring needed'),
            
            ('OPD-029', 2, 'Difficulty urinating and frequent urination', 'Benign prostatic hyperplasia with urinary retention episodes', 'Benign Prostatic Hyperplasia', 'None', 'None', 'Tamsulosin, Finasteride', '0.4mg OD at bedtime, 5mg OD', 'None Known', 'Consultation', 'BPH with urinary symptoms', '1. Tamsulosin 0.4mg OD\\n2. Finasteride 5mg OD\\n3. Bladder training\\n4. Urology referral if no improvement', DATE_SUB(CURDATE(), INTERVAL 28 DAY), 'V-3014', 'Bladder training exercises, adequate fluid intake, avoid caffeine before bed, urology referral if symptoms persist', '{"bp":"140/85","temp":"98.0","pulse":"72","weight":"78.0"}', 'BPH - may need urological intervention'),
            
            ('OPD-030', 1, 'Joint pain and morning stiffness', 'Polyarticular joint pain with prolonged morning stiffness, RF positive', 'Rheumatoid Arthritis', 'None', 'None', 'Methotrexate, Folic acid, Prednisolone', '15mg weekly, 5mg weekly, 15mg OD x 2 weeks then taper', 'None Known', 'Consultation', 'Rheumatoid arthritis - active', '1. Methotrexate 15mg weekly\\n2. Folic acid 5mg weekly\\n3. Prednisolone 15mg OD\\n4. Rheumatology referral', DATE_SUB(CURDATE(), INTERVAL 42 DAY), 'V-3015', 'Rheumatology follow-up, monitor liver function, joint protection techniques, physiotherapy, osteoporosis prevention', '{"bp":"130/80","temp":"98.2","pulse":"74","weight":"58.0"}', 'Active RA - needs rheumatology management'),
            
            -- IPD Treatment Records
            ('IPD-011', 2, 'Severe chest pain', 'STEMI - ST elevation myocardial infarction, emergency PCI performed', 'Acute STEMI', 'Emergency PCI with stent placement', 'Current admission - ICU', 'Aspirin, Clopidogrel, Atorvastatin, Metoprolol, Heparin', '100mg OD, 75mg OD, 80mg OD, 25mg BD, per protocol', 'None Known', 'Inpatient', 'Acute ST-elevation myocardial infarction', '1. Dual antiplatelet therapy\\n2. High-intensity statin\\n3. Beta blocker\\n4. ACE inhibitor when stable\\n5. Cardiac rehabilitation', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'V-4001', 'Cardiac rehabilitation, lifestyle modification, medication compliance, regular cardiology follow-up', '{"bp":"160/95","temp":"98.0","pulse":"105","weight":"72.0"}', 'Post-PCI day 2 - good outcome'),
            
            ('IPD-012', 3, 'High blood pressure and swelling', 'Severe preeclampsia at 36 weeks, emergency cesarean delivery performed', 'Severe Preeclampsia', 'Emergency cesarean section', 'Current admission - post-operative', 'Magnesium sulfate, Labetalol, Methyldopa', 'IV per protocol, 200mg BD, 250mg TID', 'None Known', 'Inpatient', 'Severe preeclampsia - delivered', '1. Continue antihypertensives\\n2. Monitor BP closely\\n3. Seizure precautions\\n4. Breastfeeding support', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-4002', 'BP monitoring, gradual medication tapering, breastfeeding support, contraception counseling, 6-week follow-up', '{"bp":"170/110","temp":"98.5","pulse":"110","weight":"78.0"}', 'Post-cesarean day 1 - BP monitoring essential'),
            
            ('IPD-013', 1, 'Severe abdominal pain', 'Acute pancreatitis - alcohol-induced, elevated lipase and amylase', 'Acute Pancreatitis - alcohol-induced', 'None', 'Current admission', 'IV fluids, Pain management, Proton pump inhibitor', 'Maintenance fluids, Tramadol IV, Pantoprazole 40mg BD IV', 'None Known', 'Inpatient', 'Acute pancreatitis - moderate severity', '1. IV fluid resuscitation\\n2. Pain management\\n3. NPO initially\\n4. Alcohol cessation counseling', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'V-4003', 'Alcohol cessation, dietary modification, enzyme supplementation if needed, diabetes screening', '{"bp":"125/78","temp":"99.2","pulse":"95","weight":"65.0"}', 'Day 3 - pain improving, tolerating feeds'),
            
            ('IPD-014', 3, 'Severe fatigue and vomiting', 'Diabetic ketoacidosis - new diagnosis type 1 diabetes', 'Diabetic Ketoacidosis - Type 1 DM', 'None', 'Current admission - ICU', 'Insulin regular IV, Potassium, IV fluids', 'Per sliding scale, 20mEq/L, Normal saline', 'None Known', 'Inpatient', 'DKA - new onset type 1 diabetes', '1. IV insulin protocol\\n2. Electrolyte replacement\\n3. Diabetic education\\n4. Transition to subcutaneous insulin', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-4004', 'Comprehensive diabetes education, insulin injection training, glucose monitoring, endocrine follow-up', '{"bp":"115/70","temp":"99.8","pulse":"120","weight":"48.0","glucose":"450"}', 'DKA resolved - transitioning to SC insulin'),
            
            ('IPD-015', 2, 'Severe shortness of breath', 'COPD exacerbation with acute respiratory failure, requiring ventilator support', 'COPD with acute respiratory failure', 'None', 'Current admission - ICU ventilated', 'Methylprednisolone IV, Antibiotics, Bronchodilators', '40mg QID IV, Ceftriaxone 1g BD, Salbutamol nebulizer', 'None Known', 'Inpatient', 'COPD exacerbation - respiratory failure', '1. IV steroids\\n2. Antibiotics\\n3. Bronchodilators\\n4. Ventilatory support\\n5. Smoking cessation', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'V-4005', 'Pulmonary rehabilitation, home oxygen therapy, smoking cessation, COPD action plan, vaccinations', '{"bp":"130/75","temp":"98.0","pulse":"110","weight":"55.0","respiratory":"25","spo2":"88"}', 'Day 4 - extubated today, stable on O2'),
            
            ('IPD-016', 1, 'Fever and abdominal pain', 'Acute cholangitis secondary to choledocholithiasis, ERCP performed', 'Acute cholangitis', 'ERCP with sphincterotomy and stone extraction', 'Current admission', 'Piperacillin-tazobactam, IV fluids', '4.5g QID IV, Maintenance fluids', 'None Known', 'Inpatient', 'Acute cholangitis - post ERCP', '1. IV antibiotics\\n2. Supportive care\\n3. Monitor liver function\\n4. Surgical consultation', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'V-4006', 'Complete antibiotic course, low-fat diet, surgical evaluation for cholecystectomy, follow-up LFTs', '{"bp":"110/65","temp":"101.5","pulse":"105","weight":"58.0"}', 'Post-ERCP day 1 - responding well'),
            
            ('IPD-017', 2, 'Back pain after accident', 'T12 complete spinal cord injury following motor vehicle accident', 'Complete T12 spinal cord injury', 'Spinal stabilization surgery', 'Current admission - post-operative', 'Methylprednisolone, DVT prophylaxis, Bowel program', 'IV protocol, Enoxaparin 40mg OD, Bowel regimen', 'None Known', 'Inpatient', 'Complete T12 SCI - post-surgical', '1. High-dose steroids\\n2. DVT prophylaxis\\n3. Early mobilization\\n4. Spinal rehab program', DATE_SUB(CURDATE(), INTERVAL 5 DAY), 'V-4007', 'Comprehensive spinal rehabilitation, psychological support, bladder/bowel management, pressure care', '{"bp":"120/75","temp":"98.0","pulse":"78","weight":"68.0"}', 'Post-op day 5 - rehab program started'),
            
            ('IPD-018', 3, 'Heavy bleeding after delivery', 'Postpartum hemorrhage requiring emergency hysterectomy', 'Postpartum hemorrhage', 'Emergency hysterectomy', 'Current admission - post-operative', 'IV fluids, Blood transfusion, Oxytocin', 'Crystalloids, 2 units PRBC, 10 units IV', 'None Known', 'Inpatient', 'Postpartum hemorrhage - post-hysterectomy', '1. Blood replacement\\n2. Hemodynamic monitoring\\n3. Infection prevention\\n4. Psychological support', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-4008', 'Psychological counseling, hormone replacement evaluation, contraception counseling not needed, 6-week follow-up', '{"bp":"90/50","temp":"97.5","pulse":"125","weight":"55.0"}', 'Post-hysterectomy day 1 - hemodynamically stable'),
            
            ('IPD-019', 1, 'Decreased urine output', 'Acute kidney injury following contrast exposure for cardiac catheterization', 'Contrast-induced nephropathy', 'None', 'Current admission', 'IV fluids, Diuretics held, Nephrotoxin avoidance', 'Normal saline, Hold ACE inhibitors, Avoid contrast', 'None Known', 'Inpatient', 'Acute kidney injury - contrast-induced', '1. Volume expansion\\n2. Avoid nephrotoxins\\n3. Monitor electrolytes\\n4. Nephrology consultation', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'V-4009', 'Nephrology follow-up, avoid nephrotoxic medications, adequate hydration, monitor renal function', '{"bp":"145/88","temp":"98.2","pulse":"82","weight":"68.0"}', 'Day 2 - creatinine plateauing'),
            
            ('IPD-020', 2, 'Continuous seizures', 'Status epilepticus - breakthrough seizures despite medication compliance', 'Status epilepticus', 'None', 'Current admission - ICU', 'IV Lorazepam, Phenytoin, Levetiracetam', '4mg IV bolus, Loading dose IV, 1000mg BD IV', 'None Known', 'Inpatient', 'Status epilepticus - refractory', '1. IV antiepileptics\\n2. EEG monitoring\\n3. Identify triggers\\n4. Medication optimization', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-4010', 'Neurology follow-up, medication compliance, trigger identification, MRI brain, EEG monitoring', '{"bp":"125/78","temp":"98.5","pulse":"95","weight":"62.0"}', 'Seizure-free 24 hours - medication levels therapeutic'),
            
            -- Additional records for remaining patients
            ('IPD-021', 3, 'Nausea and jaundice', 'Drug-induced liver injury from herbal medications', 'Drug-induced hepatitis', 'None', 'Current admission', 'Supportive care, N-acetylcysteine', 'IV fluids, 150mg/kg IV loading', 'Herbal medications', 'Inpatient', 'Drug-induced acute hepatitis', '1. Discontinue hepatotoxic drugs\\n2. Supportive care\\n3. Monitor liver function\\n4. Avoid alcohol', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'V-4011', 'Avoid herbal medications, alcohol cessation, hepatology follow-up, monitor liver function recovery', '{"bp":"115/70","temp":"99.5","pulse":"88","weight":"65.0"}', 'Day 3 - liver enzymes trending down'),
            
            ('IPD-022', 2, 'Sudden onset shortness of breath', 'High-risk pulmonary embolism, thrombolytic therapy administered', 'Pulmonary embolism - high risk', 'None', 'Current admission - ICU', 'Alteplase, Heparin, Warfarin', '100mg IV over 2 hours, IV per protocol, 5mg OD', 'None Known', 'Inpatient', 'High-risk pulmonary embolism', '1. Thrombolytic therapy\\n2. Anticoagulation\\n3. IVC filter if contraindications\\n4. Investigate underlying cause', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'V-4012', 'Long-term anticoagulation, investigate thrombophilia, compression stockings, pulmonary follow-up', '{"bp":"125/75","temp":"98.0","pulse":"108","weight":"65.0","respiratory":"24","spo2":"92"}', 'Post-thrombolysis day 2 - good response'),
            
            ('IPD-023', 1, 'Vomiting blood', 'Upper gastrointestinal bleeding - peptic ulcer with visible vessel', 'Upper GI bleeding - peptic ulcer', 'Emergency endoscopy with thermal therapy', 'Current admission', 'Pantoprazole IV, Blood transfusion', '80mg IV bolus then 8mg/hr, 2 units PRBC', 'NSAIDs', 'Inpatient', 'Upper GI bleeding - endoscopically treated', '1. High-dose PPI\\n2. Blood replacement\\n3. H. pylori eradication\\n4. Avoid NSAIDs', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-4013', 'H. pylori testing and eradication, avoid NSAIDs permanently, PPI therapy, GI follow-up', '{"bp":"105/60","temp":"98.0","pulse":"95","weight":"60.0"}', 'Post-endoscopy day 1 - no rebleeding'),
            
            ('IPD-024', 3, 'Severe breathing difficulty', 'Status asthmaticus - life-threatening asthma exacerbation', 'Status asthmaticus', 'None', 'Current admission - ICU', 'Methylprednisolone, Salbutamol, Ipratropium', '40mg QID IV, Continuous nebulizer, 500mcg QID', 'Multiple environmental allergens', 'Inpatient', 'Status asthmaticus - severe', '1. High-dose steroids\\n2. Continuous bronchodilators\\n3. Avoid triggers\\n4. Asthma education', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-4014', 'Asthma action plan, trigger avoidance, step-up therapy, allergy testing, pulmonology follow-up', '{"bp":"130/80","temp":"98.5","pulse":"115","weight":"58.0","respiratory":"28","spo2":"90"}', 'Day 2 - responding to treatment'),
            
            ('IPD-025', 2, 'Sudden weakness and speech difficulty', 'Hypertensive intracerebral hemorrhage - surgical evacuation performed', 'Intracerebral hemorrhage', 'Craniotomy with hematoma evacuation', 'Current admission - Neuro ICU', 'Mannitol, Antihypertensives, Antiepileptics', '25g IV Q6H, Nicardipine IV, Levetiracetam 500mg BD', 'None Known', 'Inpatient', 'ICH - post-surgical', '1. ICP management\\n2. Blood pressure control\\n3. Seizure prophylaxis\\n4. Neurological monitoring', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'V-4015', 'Neurological rehabilitation, speech therapy, physical therapy, long-term BP control, neurosurgery follow-up', '{"bp":"180/105","temp":"98.0","pulse":"95","weight":"75.0"}', 'Post-op day 4 - neurologically stable')`);
        console.log('✅ Comprehensive EMR treatment records seeded (30 records)');

        console.log('\n🎉 Successfully seeded 30 comprehensive EMR and Vitals records!');
        console.log('\n📊 Summary:');
        console.log('   - 30 New Patients (15 OPD + 15 IPD)');
        console.log('   - 80+ Vital Signs Records (multiple readings per patient)');
        console.log('   - 30 Comprehensive EMR Treatment Records');
        console.log('   - Realistic medical conditions and treatments');
        console.log('   - Time-series data for chronic conditions');
        console.log('   - Complete vital signs with normal ranges');
        console.log('   - Detailed treatment notes and recommendations');
        
        console.log('\n✨ New Medical Conditions Added:');
        console.log('   OPD: Diabetes, COPD, Depression, CKD, Anemia, AF, Spondylosis, Gastritis, PCOS, Gout, Fibromyalgia, Anxiety, GDM, BPH, RA');
        console.log('   IPD: MI, Preeclampsia, Pancreatitis, DKA, Respiratory Failure, Cholangitis, SCI, PPH, AKI, Status Epilepticus, Hepatitis, PE, GI Bleeding, Asthma, ICH');

    } catch (error) {
        console.error('❌ Error seeding EMR and Vitals data:', error);
        throw error;
    }
}

// Run the seeding function
if (require.main === module) {
    seed30EMRVitals()
        .then(() => {
            console.log('\n🎉 EMR and Vitals seeding completed successfully!');
            process.exit(0);
        })
        .catch((error) => {
            console.error('❌ Seeding failed:', error);
            process.exit(1);
        });
}

module.exports = seed30EMRVitals;
