/**
 * Database Seeding Tool
 * Seeds initial data into the database
 */

const { query } = require('../config/database');
const bcrypt = require('bcryptjs');

async function seedDatabase() {
    try {
        console.log('🌱 Starting database seeding...\n');

        // Seed Roles
        console.log('Seeding roles...');
        await query(`INSERT IGNORE INTO tbl_role (id, name) VALUES 
            (1, 'Admin'),
            (2, 'Manager'),
            (3, 'Cashier'),
            (4, 'Doctor')`);
        console.log('✅ Roles seeded');

        // Seed Default Users
        console.log('\nSeeding default users...');
        
        // Check if admin exists
        const existingAdmin = await query('SELECT id FROM tbl_user WHERE name = ?', ['admin']);
        
        if (existingAdmin.length === 0) {
            const hashedPassword = await bcrypt.hash('admin', 10);
            await query(
                `INSERT INTO tbl_user (id, role_id, name, passwd, ph_no, nrc_no, address, date) VALUES (?, ?, ?, ?, ?, ?, ?, CURDATE())`,
                [1, '1', 'admin', hashedPassword, '09-123456789', '12/YGNXXX', 'Yangon']
            );
            console.log('✅ Default user created (username: admin, password: admin)');
        } else {
            // Update existing admin with correct password
            const hashedPassword = await bcrypt.hash('admin', 10);
            await query('UPDATE tbl_user SET passwd = ? WHERE name = ?', [hashedPassword, 'admin']);
            console.log('✅ Admin password updated (username: admin, password: admin)');
        }

        // Seed Payment Methods
        console.log('\nSeeding payment methods...');
        await query(`INSERT IGNORE INTO tbl_payment_method (id, name) VALUES 
            (1, 'Cash'),
            (2, 'KBZ Pay'),
            (3, 'CB Pay'),
            (4, 'Aya Pay'),
            (5, 'Wave Money'),
            (6, 'Bank Transfer')`);
        console.log('✅ Payment methods seeded');

        // Seed Categories
        console.log('\nSeeding categories...');
        await query(`INSERT IGNORE INTO tbl_catalog (name) VALUES 
            ('Tablets'),
            ('Capsules'),
            ('Syrup'),
            ('Injection'),
            ('Medical Equipment'),
            ('First Aid'),
            ('Vitamins & Supplements')`);
        console.log('✅ Categories seeded');

        // Seed Service Categories
        console.log('\nSeeding service categories...');
        await query(`INSERT IGNORE INTO tbl_service_catalog (name) VALUES 
            ('Consultation'),
            ('Laboratory'),
            ('X-Ray'),
            ('Ultrasound'),
            ('ECG'),
            ('Vaccination'),
            ('Medical Procedures'),
            ('Emergency Care'),
            ('Pharmacy Services'),
            ('Administrative')`);
        console.log('✅ Service categories seeded');

        // Seed 50 Sample Services
        console.log('\nSeeding 50 sample services...');
        
        const services = [
            // Consultation Services (Category 1)
            ['S1001234567', 'General Doctor Consultation', 1, 5000, 15000, 'Basic medical consultation and examination'],
            ['S1002345678', 'Specialist Consultation', 1, 8000, 25000, 'Consultation with medical specialist'],
            ['S1003456789', 'Follow-up Consultation', 1, 3000, 10000, 'Follow-up appointment after treatment'],
            ['S1004567890', 'Emergency Consultation', 1, 10000, 35000, '24/7 emergency medical consultation'],
            ['S1005678901', 'Pediatric Consultation', 1, 6000, 20000, 'Specialized consultation for children'],
            
            // Laboratory Services (Category 2) 
            ['S2001234567', 'Complete Blood Count (CBC)', 2, 2000, 8000, 'Full blood analysis and cell count'],
            ['S2002345678', 'Blood Sugar Test', 2, 1000, 5000, 'Glucose level measurement'],
            ['S2003456789', 'Lipid Profile', 2, 2500, 10000, 'Cholesterol and triglyceride test'],
            ['S2004567890', 'Liver Function Test', 2, 3000, 12000, 'Liver enzyme and function analysis'],
            ['S2005678901', 'Kidney Function Test', 2, 2800, 11000, 'Creatinine and BUN test'],
            ['S2006789012', 'Thyroid Function Test', 2, 3500, 15000, 'TSH, T3, T4 hormone levels'],
            ['S2007890123', 'Urine Analysis', 2, 800, 3000, 'Complete urine examination'],
            ['S2008901234', 'Hepatitis B Test', 2, 4000, 18000, 'Hepatitis B surface antigen test'],
            ['S2009012345', 'HIV Test', 2, 5000, 20000, 'HIV antibody screening test'],
            ['S2010123456', 'Pregnancy Test', 2, 1500, 6000, 'Beta HCG pregnancy test'],
            
            // X-Ray Services (Category 3)
            ['S3001234567', 'Chest X-Ray', 3, 3000, 15000, 'Standard chest radiography'],
            ['S3002345678', 'Abdominal X-Ray', 3, 3500, 18000, 'Abdominal region X-ray imaging'],
            ['S3003456789', 'Spine X-Ray', 3, 4000, 20000, 'Spinal column X-ray examination'],
            ['S3004567890', 'Dental X-Ray', 3, 2500, 12000, 'Dental and jaw X-ray imaging'],
            ['S3005678901', 'Bone X-Ray', 3, 3200, 16000, 'Bone fracture and joint X-ray'],
            
            // Ultrasound Services (Category 4)
            ['S4001234567', 'Abdominal Ultrasound', 4, 5000, 25000, 'Ultrasound scan of abdominal organs'],
            ['S4002345678', 'Pregnancy Ultrasound', 4, 6000, 30000, 'Prenatal ultrasound examination'],
            ['S4003456789', 'Pelvic Ultrasound', 4, 5500, 28000, 'Pelvic region ultrasound scan'],
            ['S4004567890', 'Thyroid Ultrasound', 4, 4500, 22000, 'Thyroid gland ultrasound'],
            ['S4005678901', 'Cardiac Ultrasound', 4, 8000, 40000, 'Heart ultrasound examination'],
            
            // ECG Services (Category 5)
            ['S5001234567', 'Resting ECG', 5, 2000, 8000, '12-lead electrocardiogram at rest'],
            ['S5002345678', 'Exercise Stress Test', 5, 8000, 35000, 'ECG during physical exercise'],
            ['S5003456789', 'Holter Monitor (24hr)', 5, 15000, 60000, '24-hour continuous ECG monitoring'],
            
            // Vaccination Services (Category 6)
            ['S6001234567', 'Hepatitis B Vaccine', 6, 8000, 20000, 'Hepatitis B immunization shot'],
            ['S6002345678', 'Flu Vaccine', 6, 5000, 15000, 'Annual influenza vaccination'],
            ['S6003456789', 'COVID-19 Vaccine', 6, 0, 5000, 'COVID-19 immunization'],
            ['S6004567890', 'Tetanus Vaccine', 6, 3000, 10000, 'Tetanus immunization shot'],
            ['S6005678901', 'Pneumonia Vaccine', 6, 12000, 30000, 'Pneumococcal vaccination'],
            
            // Medical Procedures (Category 7)
            ['S7001234567', 'Blood Pressure Monitoring', 7, 500, 2000, 'Blood pressure measurement and monitoring'],
            ['S7002345678', 'Wound Dressing', 7, 1000, 5000, 'Professional wound cleaning and dressing'],
            ['S7003456789', 'Injection Administration', 7, 800, 3000, 'Intramuscular or intravenous injection'],
            ['S7004567890', 'Suture Removal', 7, 1200, 5000, 'Safe removal of surgical sutures'],
            ['S7005678901', 'IV Drip Administration', 7, 2000, 8000, 'Intravenous fluid administration'],
            ['S7006789012', 'Minor Surgery', 7, 15000, 50000, 'Minor surgical procedures'],
            ['S7007890123', 'Ear Cleaning', 7, 800, 3500, 'Professional ear wax removal'],
            ['S7008901234', 'Physical Examination', 7, 2000, 8000, 'Complete physical health examination'],
            
            // Emergency Care (Category 8)
            ['S8001234567', 'Emergency Treatment', 8, 10000, 40000, '24/7 emergency medical care'],
            ['S8002345678', 'Trauma Care', 8, 15000, 60000, 'Emergency trauma treatment'],
            ['S8003456789', 'CPR Service', 8, 5000, 25000, 'Cardiopulmonary resuscitation'],
            ['S8004567890', 'First Aid Treatment', 8, 2000, 8000, 'Basic first aid medical treatment'],
            
            // Pharmacy Services (Category 9)
            ['S9001234567', 'Medicine Home Delivery', 9, 1000, 3000, 'Free home delivery for medicines'],
            ['S9002345678', 'Medication Counseling', 9, 2000, 8000, 'Professional medication guidance'],
            ['S9003456789', 'Prescription Review', 9, 1500, 5000, 'Pharmacist prescription review'],
            ['S9004567890', 'Medicine Packaging', 9, 500, 2000, 'Weekly/monthly medicine packaging'],
            ['S9005678901', 'Drug Interaction Check', 9, 1000, 4000, 'Check for medication interactions'],
            
            // Administrative Services (Category 10)
            ['S0001234567', 'Medical Certificate', 10, 500, 3000, 'Official medical fitness certificate'],
            ['S0002345678', 'Health Record Copy', 10, 200, 1000, 'Copy of patient medical records'],
            ['S0003456789', 'Insurance Form Processing', 10, 800, 3500, 'Medical insurance form completion'],
            ['S0004567890', 'Referral Letter', 10, 300, 1500, 'Medical referral to specialist'],
            ['S0005678901', 'Medical Report', 10, 1000, 5000, 'Detailed medical examination report'],
            ['S0006789012', 'Prescription Copy', 10, 200, 800, 'Copy of medical prescription'],
            ['S0007890123', 'Vaccination Certificate', 10, 300, 1500, 'Official vaccination record certificate']
        ];
        
        // Insert services in batches to handle large dataset
        const batchSize = 10;
        for (let i = 0; i < services.length; i += batchSize) {
            const batch = services.slice(i, i + batchSize);
            const placeholders = batch.map(() => '(?, ?, ?, ?, ?, ?)').join(', ');
            const values = batch.flat();
            
            await query(`INSERT IGNORE INTO tbl_service (barcode, service_name, cat_id, buy_price, sale_price, remark) VALUES ${placeholders}`, values);
        }
        
        console.log(`✅ ${services.length} services seeded successfully`);

        // Seed Expense Categories
        console.log('\nSeeding expense categories...');
        await query(`INSERT IGNORE INTO tbl_cat_expenses (name) VALUES 
            ('Salary'),
            ('Utilities'),
            ('Rent'),
            ('Maintenance'),
            ('Office Supplies'),
            ('Transportation')`);
        console.log('✅ Expense categories seeded');

        // Seed Units
        console.log('\nSeeding units...');
        await query(`INSERT IGNORE INTO tbl_unit (unit_name, remark) VALUES 
            ('Tablet', 'Per tablet'),
            ('Box', 'Per box'),
            ('Bottle', 'Per bottle'),
            ('Vial', 'Per vial'),
            ('Pack', 'Per pack'),
            ('Piece', 'Per piece')`);
        console.log('✅ Units seeded');

        // Seed Customers
        console.log('\nSeeding customers...');
        await query(`INSERT IGNORE INTO tbl_customer (name, ph_no, nrc_no, shop, address, date, remark) VALUES 
            ('Aung Aung', '09-111111111', '12/YKNXXX', 'Aung Aung Pharmacy', 'Downtown, Yangon', CURDATE(), 'Regular customer'),
            ('Ma Mya', '09-222222222', '12/YGNXXX', 'Mya Mya Medical Store', 'Sanchaung, Yangon', CURDATE(), 'VIP customer'),
            ('Ko Htun', '09-333333333', '14/MGBXXX', 'Htun Medical Center', 'Mandalay', CURDATE(), 'Good credit'),
            ('Daw Tin', '09-444444444', '12/DGNXXX', 'Tin Tin Clinic', 'Dagon, Yangon', CURDATE(), 'Monthly payment'),
            ('U Myint', '09-555555555', '12/THNXXX', 'Myint Medical Shop', 'Thingangyun, Yangon', CURDATE(), 'Wholesale buyer'),
            ('Ma Su', '09-666666666', '12/KMNXXX', 'Su Su Pharmacy', 'Kamayut, Yangon', CURDATE(), 'Retail customer'),
            ('Ko Win', '09-777777777', '13/PYTXXX', 'Win Win Clinic', 'Pyay', CURDATE(), 'Regional customer'),
            ('Daw Khin', '09-888888888', '12/BTNXXX', 'Khin Medical', 'Botataung, Yangon', CURDATE(), 'Cash only'),
            ('U Zaw', '09-999999999', '12/YKNXXX', 'Zaw Health Center', 'Yankin, Yangon', CURDATE(), 'Credit customer'),
            ('Ma Hla', '09-101010101', '12/HGLXXX', 'Hla Pharmacy', 'Hlaing, Yangon', CURDATE(), 'New customer')`);
        console.log('✅ Customers seeded');

        // Seed Suppliers
        console.log('\nSeeding suppliers...');
        await query(`INSERT IGNORE INTO tbl_supplier (name, ph_no, nrc_no, shop, address, date, remark) VALUES 
            ('Myanmar Pharma Co.', '01-234567', '12/YGNXXX', 'Myanmar Pharma Headquarters', 'Industrial Zone, Yangon', CURDATE(), 'Main pharmaceutical supplier'),
            ('Asia Med Import', '01-345678', '12/YGNXXX', 'Asia Med Trading', 'Bo Aung Kyaw Street, Yangon', CURDATE(), 'Import medicines'),
            ('Global Health Supplies', '01-456789', '12/YGNXXX', 'Global Health Office', 'Downtown, Yangon', CURDATE(), 'International supplier'),
            ('Local Medical Store', '01-567890', '12/YGNXXX', 'Local Medical Warehouse', 'Tharketa, Yangon', CURDATE(), 'Local distributor'),
            ('Wellcare Distribution', '01-678901', '12/YGNXXX', 'Wellcare HQ', 'Mayangone, Yangon', CURDATE(), 'National distributor'),
            ('HealthPlus Trading', '01-789012', '12/YGNXXX', 'HealthPlus Office', 'Sanchaung, Yangon', CURDATE(), 'Medical equipment'),
            ('MedSupply Myanmar', '01-890123', '12/YGNXXX', 'MedSupply Center', 'Insein, Yangon', CURDATE(), 'Bulk supplier'),
            ('PharmaLink Co.', '01-901234', '12/YGNXXX', 'PharmaLink Branch', 'Bahan, Yangon', CURDATE(), 'Generic medicines'),
            ('CareFirst Suppliers', '01-012345', '12/YGNXXX', 'CareFirst Depot', 'Hlaing, Yangon', CURDATE(), 'Fast delivery'),
            ('Unity Medical Trading', '01-123456', '12/YGNXXX', 'Unity Medical Office', 'Pazundaung, Yangon', CURDATE(), 'Competitive prices')`);
        console.log('✅ Suppliers seeded');

        // Seed OPD Patients
        console.log('\nSeeding OPD patients...');
        await query(`INSERT IGNORE INTO tbl_patient (name, age, phone, address, case_, remark, date, dr_name, patient_id) VALUES 
            ('Ko Kyaw', '35', '09-111222333', 'Sanchaung Township, Yangon', 'Common Cold', 'Follow-up in 3 days', CURDATE(), 'Dr. Aung Aung', 'OPD-001'),
            ('Ma Thandar', '28', '09-222333444', 'Kamayut Township, Yangon', 'Hypertension', 'Regular checkup needed', CURDATE(), 'Dr. Myo Myo', 'OPD-002'),
            ('U Tin Win', '55', '09-333444555', 'Bahan Township, Yangon', 'Diabetes Type 2', 'Monitor blood sugar', CURDATE(), 'Dr. Aung Aung', 'OPD-003'),
            ('Daw Aye', '48', '09-444555666', 'Yankin Township, Yangon', 'Back Pain', 'Physiotherapy recommended', CURDATE(), 'Dr. Myo Myo', 'OPD-004'),
            ('Ko Zaw', '42', '09-555666777', 'Thingangyun Township, Yangon', 'Gastritis', 'Dietary advice given', CURDATE(), 'Dr. Aung Aung', 'OPD-005'),
            ('Ma Su', '25', '09-666777888', 'Mayangone Township, Yangon', 'Migraine', 'Stress management needed', CURDATE(), 'Dr. Myo Myo', 'OPD-006'),
            ('U Htun', '60', '09-777888999', 'Hlaing Township, Yangon', 'Arthritis', 'Pain management plan', CURDATE(), 'Dr. Aung Aung', 'OPD-007'),
            ('Daw Khin', '38', '09-888999000', 'Dagon Township, Yangon', 'Asthma', 'Inhaler prescribed', CURDATE(), 'Dr. Myo Myo', 'OPD-008'),
            ('Ko Win', '30', '09-999000111', 'Botataung Township, Yangon', 'Skin Allergy', 'Avoid allergens', CURDATE(), 'Dr. Aung Aung', 'OPD-009'),
            ('Ma Hla', '33', '09-000111222', 'Pazundaung Township, Yangon', 'Fever', 'Rest and hydration', CURDATE(), 'Dr. Myo Myo', 'OPD-010'),
            ('U Myint', '50', '09-111222444', 'Insein Township, Yangon', 'High Cholesterol', 'Diet control needed', CURDATE(), 'Dr. Aung Aung', 'OPD-011'),
            ('Daw Mya', '45', '09-222333555', 'Thaketa Township, Yangon', 'Thyroid Disorder', 'Regular medication', CURDATE(), 'Dr. Myo Myo', 'OPD-012'),
            ('Ko Aung', '27', '09-333444666', 'North Okkalapa, Yangon', 'Sports Injury', 'Rest for 2 weeks', CURDATE(), 'Dr. Aung Aung', 'OPD-013'),
            ('Ma Wai', '31', '09-444555777', 'South Okkalapa, Yangon', 'Pregnancy Checkup', 'All normal', CURDATE(), 'Dr. Myo Myo', 'OPD-014'),
            ('U Sein', '65', '09-555666888', 'Mingalar Taung Nyunt, Yangon', 'Heart Disease', 'Cardiologist referral', CURDATE(), 'Dr. Aung Aung', 'OPD-015')`);
        console.log('✅ OPD patients seeded');

        // Seed IPD Patients
        console.log('\nSeeding IPD patients...');
        await query(`INSERT IGNORE INTO tbl_inpatient (name, age, phone, address, case_, remark, start_date, dr_name, patient_id, room_no, deposit) VALUES 
            ('U Than Tun', '58', '09-121212121', 'Sanchaung, Yangon', 'Pneumonia', 'Admitted for IV antibiotics', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'Dr. Aung Aung', 'IPD-001', 'R-101', 500000),
            ('Daw Sandar', '47', '09-232323232', 'Kamayut, Yangon', 'Appendicitis', 'Post-surgery recovery', DATE_SUB(CURDATE(), INTERVAL 5 DAY), 'Dr. Myo Myo', 'IPD-002', 'R-102', 800000),
            ('Ko Nyi', '35', '09-343434343', 'Bahan, Yangon', 'Dengue Fever', 'Platelet monitoring', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Aung Aung', 'IPD-003', 'R-103', 400000),
            ('Ma Hnin', '29', '09-454545454', 'Yankin, Yangon', 'Delivery - C-Section', 'Mother and baby healthy', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Myo Myo', 'IPD-004', 'R-201', 1000000),
            ('U Kyaw Soe', '62', '09-565656565', 'Hlaing, Yangon', 'Stroke', 'Stable condition, physiotherapy started', DATE_SUB(CURDATE(), INTERVAL 7 DAY), 'Dr. Aung Aung', 'IPD-005', 'R-104', 1500000),
            ('Daw Myint', '52', '09-676767676', 'Thingangyun, Yangon', 'Kidney Stones', 'Awaiting procedure', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'Dr. Myo Myo', 'IPD-006', 'R-105', 600000),
            ('Ko Htet', '40', '09-787878787', 'Mayangone, Yangon', 'Traffic Accident', 'Fracture treatment', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'Dr. Aung Aung', 'IPD-007', 'R-106', 700000),
            ('Ma Phyu', '26', '09-898989898', 'Dagon, Yangon', 'Severe Dehydration', 'IV fluids treatment', CURDATE(), 'Dr. Myo Myo', 'IPD-008', 'R-202', 300000),
            ('U Win Htun', '55', '09-909090909', 'Botataung, Yangon', 'Gastric Ulcer', 'Medical management', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'Dr. Aung Aung', 'IPD-009', 'R-107', 450000),
            ('Daw Yin', '43', '09-101010101', 'Insein, Yangon', 'Gallbladder Surgery', 'Post-op day 3', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'Dr. Myo Myo', 'IPD-010', 'R-203', 900000)`);
        console.log('✅ IPD patients seeded');

        // Seed Comprehensive EMR Treatment Records
        console.log('\nSeeding comprehensive EMR treatment records...');
        await query(`INSERT IGNORE INTO tbl_treatment (patient_id, Past_Diagnoses, Surgeries_Procedures, Hospitalizations, Current_Medications, Dosages, Allergen, Treatment_Type, Date, Voucher_id, Recommendation) VALUES 
            -- OPD Treatment Records
            ('OPD-001', 'Seasonal Allergies', 'None', 'None', 'Paracetamol, Vitamin C', '500mg TID, 1000mg OD', 'None Known', 'Outpatient', CURDATE(), 'V-1001', 'Rest, plenty of fluids, avoid cold exposure'),
            ('OPD-002', 'Hypertension (5 years)', 'None', 'None', 'Amlodipine, Losartan', '5mg OD, 50mg OD', 'None Known', 'Outpatient', CURDATE(), 'V-1002', 'Continue medications, low salt diet, regular exercise'),
            ('OPD-003', 'Diabetes Type 2 (10 years)', 'None', 'None', 'Metformin, Glimepiride', '500mg BD, 2mg OD', 'Sulfa drugs', 'Outpatient', CURDATE(), 'V-1003', 'Monitor blood sugar daily, diabetic diet, foot care'),
            ('OPD-004', 'Chronic Back Pain', 'None', 'None', 'Ibuprofen, Muscle Relaxant', '400mg TID, 1 tab BD', 'None Known', 'Outpatient', CURDATE(), 'V-1004', 'Physiotherapy 3x/week, avoid heavy lifting'),
            ('OPD-005', 'Gastritis (2 years)', 'None', 'None', 'Omeprazole, Antacid', '20mg BD, PRN', 'Penicillin', 'Outpatient', CURDATE(), 'V-1005', 'Avoid spicy food, eat small meals, reduce stress'),
            ('OPD-006', 'Migraine (Chronic)', 'None', 'None', 'Sumatriptan, Propranolol', '50mg PRN, 40mg BD', 'Aspirin', 'Outpatient', CURDATE(), 'V-1006', 'Avoid triggers, stress management, adequate sleep'),
            ('OPD-007', 'Osteoarthritis', 'None', 'None', 'Diclofenac, Glucosamine', '75mg BD, 500mg OD', 'None Known', 'Outpatient', CURDATE(), 'V-1007', 'Weight management, gentle exercises, heat therapy'),
            ('OPD-008', 'Bronchial Asthma', 'None', 'None', 'Salbutamol Inhaler, Budesonide', '2 puffs PRN, 200mcg BD', 'Pollen', 'Outpatient', CURDATE(), 'V-1008', 'Avoid allergens, proper inhaler technique'),
            ('OPD-009', 'Atopic Dermatitis', 'None', 'None', 'Cetirizine, Hydrocortisone Cream', '10mg OD, Apply BD', 'Latex', 'Outpatient', CURDATE(), 'V-1009', 'Moisturize skin, avoid irritants'),
            ('OPD-010', 'Viral Fever', 'None', 'None', 'Paracetamol, Antihistamine', '500mg TID, 1 tab HS', 'None Known', 'Outpatient', CURDATE(), 'V-1010', 'Rest, hydration, symptomatic treatment'),
            ('OPD-011', 'Hyperlipidemia', 'None', 'None', 'Atorvastatin, Omega-3', '20mg OD, 1000mg OD', 'None Known', 'Outpatient', CURDATE(), 'V-1011', 'Low fat diet, regular exercise, follow-up in 3 months'),
            ('OPD-012', 'Hypothyroidism', 'None', 'None', 'Levothyroxine', '100mcg OD (morning)', 'None Known', 'Outpatient', CURDATE(), 'V-1012', 'Take on empty stomach, thyroid function test in 6 weeks'),
            ('OPD-013', 'Ankle Sprain', 'None', 'None', 'Diclofenac, Ice therapy', '75mg BD, Apply 4x/day', 'None Known', 'Outpatient', CURDATE(), 'V-1013', 'RICE protocol, avoid weight bearing, follow-up in 1 week'),
            ('OPD-014', 'Prenatal Care - 24 weeks', 'None', 'None', 'Folic Acid, Iron, Calcium', '5mg OD, 60mg OD, 500mg BD', 'None Known', 'Outpatient', CURDATE(), 'V-1014', 'Regular antenatal checkups, balanced diet, adequate rest'),
            ('OPD-015', 'Coronary Artery Disease', 'CABG 2 years ago', 'Cardiac ICU admission post-CABG', 'Aspirin, Clopidogrel, Metoprolol, Rosuvastatin', '100mg OD, 75mg OD, 50mg BD, 20mg OD', 'NSAIDs', 'Outpatient', CURDATE(), 'V-1015', 'Cardiac rehab, stress test in 3 months, low cholesterol diet'),
            
            -- IPD Treatment Records
            ('IPD-001', 'Pneumonia', 'None', 'Previous pneumonia 3 years ago', 'Ceftriaxone IV, Azithromycin', '1g BD IV, 500mg OD', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'V-2001', 'Complete antibiotic course, chest physiotherapy'),
            ('IPD-002', 'Appendicitis', 'Appendectomy', 'Current admission', 'Cefuroxime IV, Metronidazole', '750mg TID IV, 500mg TID', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 5 DAY), 'V-2002', 'Post-op care, gradual diet advancement'),
            ('IPD-003', 'Dengue Fever', 'None', 'None', 'IV Fluids, Paracetamol', 'Maintenance, 500mg TID', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'V-2003', 'Monitor platelet count, adequate hydration'),
            ('IPD-004', 'Pregnancy - Term', 'C-Section Delivery', 'Current admission', 'Cefazolin, Oxytocin', '1g IV, As needed', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-2004', 'Post-operative care, breastfeeding support'),
            ('IPD-005', 'Ischemic Stroke', 'None', 'Current admission', 'Aspirin, Atorvastatin, Clopidogrel', '100mg OD, 40mg OD, 75mg OD', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 7 DAY), 'V-2005', 'Physiotherapy, speech therapy, blood pressure control'),
            ('IPD-006', 'Nephrolithiasis', 'None', 'None', 'IV Fluids, Ketorolac, Tamsulosin', 'Maintenance, 30mg IM PRN, 0.4mg OD', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-2006', 'Hydration, pain management, lithotripsy scheduled'),
            ('IPD-007', 'Multiple Fractures', 'ORIF Left Femur', 'Current admission', 'Cefazolin IV, Tramadol', '1g TID IV, 50mg QID', 'Morphine', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'V-2007', 'Bed rest, wound care, physiotherapy after 2 weeks'),
            ('IPD-008', 'Severe Dehydration', 'None', 'None', 'IV Ringer Lactate, Ondansetron', '2L over 4hrs, 4mg TID', 'None Known', 'Inpatient', CURDATE(), 'V-2008', 'Oral rehydration when tolerated, monitor electrolytes'),
            ('IPD-009', 'Peptic Ulcer Disease', 'None', 'Previous GI bleed 1 year ago', 'Pantoprazole IV, Sucralfate', '40mg BD IV, 1g QID', 'NSAIDs', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'V-2009', 'H. pylori testing, endoscopy scheduled, avoid NSAIDs'),
            ('IPD-010', 'Acute Cholecystitis', 'Laparoscopic Cholecystectomy', 'Current admission', 'Ceftriaxone IV, Metronidazole IV', '1g BD IV, 500mg TID IV', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'V-2010', 'Post-op recovery, low fat diet after discharge'),
            
            -- Follow-up Treatment Records
            ('OPD-001', 'URI resolved, mild cough persists', 'None', 'None', 'Dextromethorphan Syrup', '10ml TID', 'None Known', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'V-1011', 'Complete cough syrup, return if fever develops'),
            ('OPD-002', 'BP controlled, no complications', 'None', 'None', 'Amlodipine, Losartan', '5mg OD, 50mg OD', 'None Known', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 7 DAY), 'V-1012', 'Continue current medications, BP monitoring at home'),
            ('OPD-003', 'HbA1c improved to 7.2%', 'None', 'None', 'Metformin, Glimepiride, Insulin Glargine', '500mg BD, 2mg OD, 10 units HS', 'Sulfa drugs', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 14 DAY), 'V-1013', 'Good control, continue regimen, next HbA1c in 3 months'),
            ('OPD-006', 'Migraine frequency reduced', 'None', 'None', 'Propranolol, Magnesium', '40mg BD, 400mg OD', 'Aspirin', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 10 DAY), 'V-1014', 'Maintain stress diary, continue prophylaxis'),
            ('OPD-007', 'Joint pain improved', 'None', 'None', 'Diclofenac, Glucosamine, Physiotherapy', '75mg BD, 500mg OD, 3x/week', 'None Known', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 5 DAY), 'V-1015', 'Continue therapy, consider intra-articular injection'),
            
            -- Additional New Cases
            ('OPD-001', 'Common Cold - New episode', 'None', 'None', 'Paracetamol, Cetirizine, Vitamin C', '500mg TID, 10mg OD, 1000mg OD', 'None Known', 'Outpatient', DATE_SUB(CURDATE(), INTERVAL 30 DAY), 'V-1016', 'Symptomatic treatment, adequate rest'),
            ('OPD-002', 'Hypertension - Routine checkup', 'None', 'None', 'Amlodipine, Losartan, Aspirin', '5mg OD, 50mg OD, 100mg OD', 'None Known', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 60 DAY), 'V-1017', 'BP well controlled, continue medications'),
            ('OPD-011', 'Lipid panel improved', 'None', 'None', 'Atorvastatin, Omega-3, Fenofibrate', '20mg OD, 1000mg OD, 145mg OD', 'None Known', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 90 DAY), 'V-1018', 'Excellent response, maintain lifestyle changes'),
            ('OPD-012', 'TSH normalized', 'None', 'None', 'Levothyroxine', '100mcg OD', 'None Known', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 42 DAY), 'V-1019', 'Optimal dose achieved, annual monitoring'),
            ('OPD-015', 'Stable angina', 'CABG 2 years ago', 'Cardiac ICU admission post-CABG', 'Aspirin, Clopidogrel, Metoprolol, Rosuvastatin, Isosorbide', '100mg OD, 75mg OD, 50mg BD, 20mg OD, 5mg BD', 'NSAIDs', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 21 DAY), 'V-1020', 'Exercise tolerance improved, continue cardiac rehab'),
            
            -- Additional Diverse Cases
            ('OPD-013', 'Ankle fully healed', 'None', 'None', 'Stopped medications', 'None', 'None Known', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 14 DAY), 'V-1021', 'Gradual return to activities, strengthening exercises'),
            ('OPD-014', 'Normal pregnancy progression - 28 weeks', 'None', 'None', 'Folic Acid, Iron, Calcium, Vitamin D', '5mg OD, 60mg OD, 500mg BD, 1000 IU OD', 'None Known', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 28 DAY), 'V-1022', 'Ultrasound scheduled, glucose tolerance test done'),
            ('OPD-008', 'Asthma well controlled', 'None', 'None', 'Salbutamol Inhaler, Budesonide', '2 puffs PRN, 200mcg BD', 'Pollen', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 15 DAY), 'V-1023', 'Peak flow monitoring, continue preventer'),
            ('OPD-009', 'Eczema improved', 'None', 'None', 'Cetirizine, Hydrocortisone Cream', '10mg OD, Apply BD', 'Latex', 'Follow-up', DATE_SUB(CURDATE(), INTERVAL 7 DAY), 'V-1024', 'Taper steroid cream, emollient use'),
            ('OPD-004', 'Back pain recurrence', 'None', 'None', 'Diclofenac, Muscle Relaxant, Physiotherapy', '75mg BD, 1 tab TID, Daily sessions', 'None Known', 'Outpatient', DATE_SUB(CURDATE(), INTERVAL 20 DAY), 'V-1025', 'MRI scheduled, core strengthening exercises'),
            
            -- IPD Treatment Records
            ('IPD-001', 'Pneumonia', 'None', 'Previous pneumonia 3 years ago', 'Ceftriaxone IV, Azithromycin', '1g BD IV, 500mg OD', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'V-2001', 'Complete antibiotic course, chest physiotherapy'),
            ('IPD-002', 'Appendicitis', 'Appendectomy', 'Current admission', 'Cefuroxime IV, Metronidazole', '750mg TID IV, 500mg TID', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 5 DAY), 'V-2002', 'Post-op care, gradual diet advancement'),
            ('IPD-003', 'Dengue Fever', 'None', 'None', 'IV Fluids, Paracetamol', 'Maintenance, 500mg TID', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'V-2003', 'Monitor platelet count, adequate hydration'),
            ('IPD-004', 'Pregnancy - Term', 'C-Section Delivery', 'Current admission', 'Cefazolin, Oxytocin', '1g IV, As needed', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-2004', 'Post-operative care, breastfeeding support'),
            ('IPD-005', 'Ischemic Stroke', 'None', 'Current admission', 'Aspirin, Atorvastatin, Clopidogrel', '100mg OD, 40mg OD, 75mg OD', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 7 DAY), 'V-2005', 'Physiotherapy, speech therapy, blood pressure control'),
            ('IPD-006', 'Nephrolithiasis', 'None', 'None', 'IV Fluids, Ketorolac, Tamsulosin', 'Maintenance, 30mg IM PRN, 0.4mg OD', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-2006', 'Hydration, pain management, lithotripsy scheduled'),
            ('IPD-007', 'Multiple Fractures', 'ORIF Left Femur', 'Current admission', 'Cefazolin IV, Tramadol', '1g TID IV, 50mg QID', 'Morphine', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 4 DAY), 'V-2007', 'Bed rest, wound care, physiotherapy after 2 weeks'),
            ('IPD-008', 'Severe Dehydration', 'None', 'None', 'IV Ringer Lactate, Ondansetron', '2L over 4hrs, 4mg TID', 'None Known', 'Inpatient', CURDATE(), 'V-2008', 'Oral rehydration when tolerated, monitor electrolytes'),
            ('IPD-009', 'Peptic Ulcer Disease', 'None', 'Previous GI bleed 1 year ago', 'Pantoprazole IV, Sucralfate', '40mg BD IV, 1g QID', 'NSAIDs', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 2 DAY), 'V-2009', 'H. pylori testing, endoscopy scheduled, avoid NSAIDs'),
            ('IPD-010', 'Acute Cholecystitis', 'Laparoscopic Cholecystectomy', 'Current admission', 'Ceftriaxone IV, Metronidazole IV', '1g BD IV, 500mg TID IV', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'V-2010', 'Post-op recovery, low fat diet after discharge'),
            
            -- Additional Follow-up Records
            ('IPD-001', 'Pneumonia resolving', 'None', 'Current admission', 'Oral Amoxicillin, Bromhexine', '500mg TID, 8mg TID', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-2011', 'Ready for discharge tomorrow, complete oral antibiotics'),
            ('IPD-003', 'Dengue - Platelets recovering', 'None', 'Current admission', 'IV Fluids, Paracetamol', 'Maintenance, 500mg TID', 'None Known', 'Inpatient', CURDATE(), 'V-2012', 'Platelets 95k, continue monitoring, discharge in 2 days'),
            ('IPD-005', 'Stroke - Rehabilitation progress', 'None', 'Current admission', 'Aspirin, Atorvastatin, Clopidogrel, Physiotherapy', '100mg OD, 40mg OD, 75mg OD, BD sessions', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 3 DAY), 'V-2013', 'Motor function improving, continue intensive rehab'),
            ('IPD-007', 'Fracture healing well', 'ORIF Left Femur', 'Current admission', 'Oral Cefuroxime, Tramadol', '500mg BD, 50mg QID', 'Morphine', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-2014', 'X-ray shows good alignment, start partial weight bearing'),
            ('IPD-010', 'Post-cholecystectomy recovery', 'Laparoscopic Cholecystectomy', 'Current admission', 'Oral Cefuroxime, Paracetamol', '500mg BD, 500mg TID', 'None Known', 'Inpatient', DATE_SUB(CURDATE(), INTERVAL 1 DAY), 'V-2015', 'Wound healing well, tolerating diet, discharge tomorrow')`);
        console.log('✅ Comprehensive EMR treatment records seeded (40 records)');

        // Seed Sample Doctors
        console.log('\nSeeding sample doctors...');
        await query(`INSERT IGNORE INTO tbl_doctor (name, phone, education, specialist, remark) VALUES 
            ('Dr. Aung Aung', '09-111111111', 'MBBS, MD', 'General Physician', 'Available Mon-Fri'),
            ('Dr. Su Su', '09-222222222', 'MBBS, MD (Peds)', 'Pediatrician', 'Available Mon-Sat'),
            ('Dr. Kyaw Kyaw', '09-333333333', 'MBBS, MS (Ortho)', 'Orthopedic Surgeon', 'Available on appointment')`);
        console.log('✅ Sample doctors seeded');

        // Seed Cases
        console.log('\nSeeding medical cases...');
        await query(`INSERT IGNORE INTO tbl_case (name) VALUES 
            ('Fever'),
            ('Cold & Flu'),
            ('Headache'),
            ('Stomach Pain'),
            ('Injury'),
            ('Diabetes'),
            ('Hypertension'),
            ('Check-up'),
            ('Follow-up'),
            ('Vaccination')`);
        console.log('✅ Medical cases seeded');

        // Seed Sample Shop Info
        console.log('\nSeeding shop information...');
        await query(`INSERT IGNORE INTO tbl_shopinfo (id, name, type, ph_no, email, address, remark, thanks, printer) VALUES 
            (1, 'Clinic Pro Medical Center', 'Clinic & Pharmacy', '09-999999999', 'info@clinicpro.com', 
            'No. 123, Main Street, Yangon', 'Open 24/7', 
            'Thank you for choosing Clinic Pro! Stay healthy!', '80mm')`);
        console.log('✅ Shop information seeded');

        // Seed Sample Stock Items
        console.log('\nSeeding sample stock items...');
        const catalogResult = await query('SELECT id FROM tbl_catalog LIMIT 1');
        const catId = catalogResult[0]?.id || 1;
        
        await query(`INSERT IGNORE INTO tbl_stock (barcode, name, cat_id, qty, qty_alert, unit_name, cost_price, sale_price_latli, sale_price_latkar, remark) VALUES 
            ('8850987180001', 'Paracetamol 500mg', '${catId}', 1000, 50, 'Tablet', 50, 100, 90, 'For fever and pain'),
            ('8850987180002', 'Amoxicillin 500mg', '${catId}', 500, 30, 'Capsule', 200, 350, 300, 'Antibiotic'),
            ('8850987180003', 'Vitamin C 1000mg', '${catId}', 800, 40, 'Tablet', 100, 200, 180, 'Immune support')`);
        console.log('✅ Sample stock items seeded');
        
        // Check if comprehensive products already exist
        const existingProducts = await query('SELECT COUNT(*) as count FROM tbl_stock');
        if (existingProducts[0].count < 50) {
            console.log('\n💡 To seed 100+ comprehensive products, run: node seed-products.js');
        } else {
            console.log(`\n📦 Stock database contains ${existingProducts[0].count} products`);
        }

        // Check patient count for pagination testing
        const patientCount = await query('SELECT COUNT(*) as count FROM tbl_patient');
        if (patientCount[0].count < 25) {
            console.log('\n💡 To seed more patients for pagination testing, run: node seed-patients.js');
        } else {
            console.log(`\n👥 Patient database contains ${patientCount[0].count} patients`);
        }

        console.log('\n🎉 Database seeding completed successfully!\n');
        
        console.log('📝 Default Login Credentials:');
        console.log('   Username: admin');
        console.log('   Password: admin\n');

        return true;
    } catch (error) {
        console.error('❌ Seeding failed:', error.message);
        throw error;
    }
}

// Run seeding if called directly
if (require.main === module) {
    require('dotenv').config();
    seedDatabase().then(() => {
        console.log('✨ Done!');
        process.exit(0);
    }).catch(err => {
        console.error(err);
        process.exit(1);
    });
}

module.exports = { seedDatabase };

