const { query } = require('../config/database');
const fs = require('fs');
const path = require('path');

async function setupDepositTable() {
    try {
        console.log('🚀 Setting up deposit transaction table...');
        
        // Create deposit transactions table
        const createTableSQL = `
            CREATE TABLE IF NOT EXISTS tbl_deposit_transaction (
                id INT AUTO_INCREMENT PRIMARY KEY,
                patient_id INT NOT NULL,
                transaction_type ENUM('deposit', 'withdrawal', 'refund', 'adjustment') NOT NULL,
                amount DECIMAL(10,2) NOT NULL,
                balance_after DECIMAL(10,2) NOT NULL,
                description TEXT,
                reference_no VARCHAR(50),
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                status ENUM('active', 'cancelled') DEFAULT 'active',
                
                INDEX idx_patient_id (patient_id),
                INDEX idx_transaction_type (transaction_type),
                INDEX idx_created_at (created_at),
                INDEX idx_reference_no (reference_no),
                
                FOREIGN KEY (patient_id) REFERENCES tbl_patient(id) ON DELETE CASCADE,
                FOREIGN KEY (created_by) REFERENCES tbl_user(id) ON DELETE SET NULL
            )
        `;
        
        await query(createTableSQL);
        console.log('✅ Deposit transaction table created successfully');
        
        // Add deposit_balance column to patient table if it doesn't exist
        try {
            await query(`
                ALTER TABLE tbl_patient 
                ADD COLUMN deposit_balance DECIMAL(10,2) DEFAULT 0.00 AFTER deposit
            `);
            console.log('✅ Added deposit_balance column to patient table');
        } catch (error) {
            if (error.message.includes('Duplicate column name')) {
                console.log('ℹ️  deposit_balance column already exists');
            } else {
                throw error;
            }
        }
        
        // Update existing patients to have deposit_balance equal to deposit
        const updateResult = await query(`
            UPDATE tbl_patient 
            SET deposit_balance = COALESCE(deposit, 0) 
            WHERE deposit_balance IS NULL OR deposit_balance = 0
        `);
        console.log(`✅ Updated ${updateResult.affectedRows} patients with deposit balance`);
        
        console.log('🎉 Deposit table setup completed successfully!');
        
    } catch (error) {
        console.error('❌ Error setting up deposit table:', error);
        throw error;
    }
}

// Run the setup if this file is executed directly
if (require.main === module) {
    setupDepositTable()
        .then(() => {
            console.log('Setup completed');
            process.exit(0);
        })
        .catch((error) => {
            console.error('Setup failed:', error);
            process.exit(1);
        });
}

module.exports = setupDepositTable;
