/**
 * Stock Utility Functions
 * Handles expiry date alerts, price changes, and profit calculations
 */

const { query } = require('../config/database');

/**
 * Calculate margin percentage
 * @param {number} costPrice - Cost price
 * @param {number} salePrice - Sale price
 * @returns {number} Margin percentage
 */
function calculateMargin(costPrice, salePrice) {
    if (!costPrice || costPrice === 0) return 0;
    return ((salePrice - costPrice) / costPrice) * 100;
}

/**
 * Calculate profit amount
 * @param {number} costPrice - Cost price
 * @param {number} salePrice - Sale price
 * @param {number} quantity - Quantity sold
 * @param {number} discount - Discount amount
 * @returns {number} Profit amount
 */
function calculateProfit(costPrice, salePrice, quantity = 1, discount = 0) {
    const revenue = salePrice * quantity;
    const cost = costPrice * quantity;
    return revenue - cost - discount;
}

/**
 * Calculate alert date based on expiry date and alert months
 * @param {Date|string} expireDate - Expiry date
 * @param {number} alertMonths - Number of months before expiry to alert
 * @returns {Date|null} Alert date
 */
function calculateAlertDate(expireDate, alertMonths) {
    if (!expireDate) return null;
    
    const expiry = new Date(expireDate);
    if (isNaN(expiry.getTime())) return null;
    
    if (!alertMonths || alertMonths === 0) return null;
    
    const alertDate = new Date(expiry);
    alertDate.setMonth(alertDate.getMonth() - alertMonths);
    
    return alertDate;
}

/**
 * Check if item is expired or expiring soon
 * @param {Date|string} expireDate - Expiry date
 * @param {Date|string} alertDate - Alert date
 * @returns {object} Status object with isExpired, isExpiringSoon, daysUntilExpiry
 */
function checkExpiryStatus(expireDate, alertDate) {
    if (!expireDate) {
        return {
            isExpired: false,
            isExpiringSoon: false,
            daysUntilExpiry: null,
            status: 'no_expiry'
        };
    }
    
    const expiry = new Date(expireDate);
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    expiry.setHours(0, 0, 0, 0);
    
    const daysUntilExpiry = Math.ceil((expiry - today) / (1000 * 60 * 60 * 24));
    
    const isExpired = daysUntilExpiry < 0;
    const isExpiringSoon = alertDate ? new Date(alertDate) <= today : false;
    
    let status = 'valid';
    if (isExpired) {
        status = 'expired';
    } else if (isExpiringSoon) {
        status = 'expiring_soon';
    }
    
    return {
        isExpired,
        isExpiringSoon,
        daysUntilExpiry,
        status
    };
}

/**
 * Get expiry alerts for stock items
 * @param {number} daysBeforeAlert - Days before expiry to show alert (default: 30)
 * @returns {Promise<Array>} Array of items with expiry alerts
 */
async function getExpiryAlerts(daysBeforeAlert = 30) {
    try {
        const today = new Date();
        const alertThreshold = new Date(today);
        alertThreshold.setDate(alertThreshold.getDate() + daysBeforeAlert);
        
        const sql = `
            SELECT 
                s.*,
                s.expire_date,
                s.alert_date,
                DATEDIFF(s.expire_date, CURDATE()) as days_until_expiry,
                CASE 
                    WHEN s.expire_date < CURDATE() THEN 'expired'
                    WHEN s.expire_date <= ? THEN 'expiring_soon'
                    WHEN s.alert_date IS NOT NULL AND s.alert_date <= CURDATE() THEN 'alert_active'
                    ELSE 'valid'
                END as expiry_status
            FROM tbl_stock s
            WHERE s.expire_date IS NOT NULL
            AND (
                s.expire_date < CURDATE() 
                OR s.expire_date <= ?
                OR (s.alert_date IS NOT NULL AND s.alert_date <= CURDATE())
            )
            AND s.qty > 0
            ORDER BY s.expire_date ASC
        `;
        
        const alerts = await query(sql, [alertThreshold, alertThreshold]);
        
        return alerts.map(item => {
            const expiryStatus = checkExpiryStatus(item.expire_date, item.alert_date);
            return {
                ...item,
                ...expiryStatus
            };
        });
    } catch (error) {
        console.error('Error getting expiry alerts:', error);
        return [];
    }
}

/**
 * Update stock price and recalculate margin
 * @param {number} stockId - Stock ID
 * @param {number} newCostPrice - New cost price
 * @param {number} newSalePrice - New sale price
 * @returns {Promise<object>} Updated stock with calculated margin
 */
async function updateStockPrice(stockId, newCostPrice, newSalePrice) {
    try {
        const margin = calculateMargin(newCostPrice, newSalePrice);
        
        const sql = `
            UPDATE tbl_stock 
            SET cost_price = ?, 
                sale_price_latli = ?,
                margin = ?
            WHERE id = ?
        `;
        
        await query(sql, [newCostPrice, newSalePrice, margin, stockId]);
        
        // Get updated stock
        const updatedStock = await query('SELECT * FROM tbl_stock WHERE id = ?', [stockId]);
        
        return {
            ...updatedStock[0],
            margin: margin
        };
    } catch (error) {
        console.error('Error updating stock price:', error);
        throw error;
    }
}

/**
 * Track price change history
 * @param {number} stockId - Stock ID
 * @param {number} oldCostPrice - Old cost price
 * @param {number} newCostPrice - New cost price
 * @param {number} oldSalePrice - Old sale price
 * @param {number} newSalePrice - New sale price
 * @param {string} changedBy - User who made the change
 * @returns {Promise<void>}
 */
async function trackPriceChange(stockId, oldCostPrice, newCostPrice, oldSalePrice, newSalePrice, changedBy) {
    try {
        // Check if price_history table exists, if not create it
        const createTableSql = `
            CREATE TABLE IF NOT EXISTS tbl_price_history (
                id INT AUTO_INCREMENT PRIMARY KEY,
                stock_id INT NOT NULL,
                old_cost_price DECIMAL(10, 2),
                new_cost_price DECIMAL(10, 2),
                old_sale_price DECIMAL(10, 2),
                new_sale_price DECIMAL(10, 2),
                old_margin DECIMAL(10, 2),
                new_margin DECIMAL(10, 2),
                changed_by VARCHAR(100),
                changed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_stock_id (stock_id),
                INDEX idx_changed_at (changed_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        `;
        
        await query(createTableSql);
        
        const oldMargin = calculateMargin(oldCostPrice, oldSalePrice);
        const newMargin = calculateMargin(newCostPrice, newSalePrice);
        
        const insertSql = `
            INSERT INTO tbl_price_history 
            (stock_id, old_cost_price, new_cost_price, old_sale_price, new_sale_price, old_margin, new_margin, changed_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        `;
        
        await query(insertSql, [
            stockId,
            oldCostPrice,
            newCostPrice,
            oldSalePrice,
            newSalePrice,
            oldMargin,
            newMargin,
            changedBy
        ]);
    } catch (error) {
        console.error('Error tracking price change:', error);
        // Don't throw error, just log it
    }
}

/**
 * Get price history for a stock item
 * @param {number} stockId - Stock ID
 * @param {number} limit - Number of records to return
 * @returns {Promise<Array>} Price history records
 */
async function getPriceHistory(stockId, limit = 50) {
    try {
        const sql = `
            SELECT * FROM tbl_price_history
            WHERE stock_id = ?
            ORDER BY changed_at DESC
            LIMIT ?
        `;
        
        return await query(sql, [stockId, limit]);
    } catch (error) {
        console.error('Error getting price history:', error);
        return [];
    }
}

/**
 * Calculate profit for a sale transaction
 * @param {Array} saleItems - Array of sale items with cost_price, sale_price, quantity, discount
 * @returns {object} Profit summary
 */
function calculateSaleProfit(saleItems) {
    let totalRevenue = 0;
    let totalCost = 0;
    let totalDiscount = 0;
    let totalProfit = 0;
    
    saleItems.forEach(item => {
        const revenue = (item.sale_price || 0) * (item.quantity || 0);
        const cost = (item.cost_price || 0) * (item.quantity || 0);
        const discount = item.discount || 0;
        const profit = revenue - cost - discount;
        
        totalRevenue += revenue;
        totalCost += cost;
        totalDiscount += discount;
        totalProfit += profit;
    });
    
    const profitMargin = totalRevenue > 0 ? (totalProfit / totalRevenue) * 100 : 0;
    
    return {
        totalRevenue,
        totalCost,
        totalDiscount,
        totalProfit,
        profitMargin: parseFloat(profitMargin.toFixed(2))
    };
}

module.exports = {
    calculateMargin,
    calculateProfit,
    calculateAlertDate,
    checkExpiryStatus,
    getExpiryAlerts,
    updateStockPrice,
    trackPriceChange,
    getPriceHistory,
    calculateSaleProfit
};

