/**
 * Test script to verify warehouse stock deduction
 * Run with: node backend/utils/test-warehouse-deduction.js
 */

const { query, transaction } = require('../config/database');

async function testWarehouseDeduction() {
    try {
        console.log('🧪 Testing Warehouse Stock Deduction...\n');
        
        // Get a warehouse and stock to test with
        const warehouses = await query('SELECT * FROM tbl_warehouse LIMIT 1');
        if (warehouses.length === 0) {
            console.error('❌ No warehouses found');
            return;
        }
        const warehouseId = warehouses[0].id;
        console.log(`📦 Using warehouse: ${warehouses[0].name} (ID: ${warehouseId})`);
        
        // Get a stock item that exists in this warehouse
        const warehouseStock = await query(
            'SELECT ws.*, s.name as stock_name FROM tbl_warehouse_stock ws JOIN tbl_stock s ON ws.stock_id = s.id WHERE ws.warehouse_id = ? LIMIT 1',
            [warehouseId]
        );
        
        if (warehouseStock.length === 0) {
            console.error('❌ No stock found in warehouse');
            return;
        }
        
        const stockId = warehouseStock[0].stock_id;
        const stockName = warehouseStock[0].stock_name;
        const initialQty = warehouseStock[0].quantity;
        
        console.log(`📊 Testing with stock: ${stockName} (ID: ${stockId})`);
        console.log(`📊 Initial quantity: ${initialQty}\n`);
        
        // Test 1: Create an OUT movement and verify trigger updates stock
        console.log('Test 1: Creating OUT movement...');
        const testQty = 1;
        
        const result = await transaction(async (conn) => {
            // Create stock movement
            const [movementResult] = await conn.query(
                `INSERT INTO tbl_stock_movement
                 (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id,
                  movement_date, notes, created_by)
                 VALUES ('OUT', ?, ?, ?, 'SALE', ?, NOW(), ?, ?)`,
                [
                    warehouseId,
                    stockId,
                    testQty,
                    99999, // Test voucher number
                    `Test Sale: ${stockName}`,
                    1
                ]
            );
            
            console.log(`✅ Movement created with ID: ${movementResult.insertId}`);
            
            // Check if trigger updated the stock
            const [updatedStock] = await conn.query(
                'SELECT quantity FROM tbl_warehouse_stock WHERE warehouse_id = ? AND stock_id = ?',
                [warehouseId, stockId]
            );
            
            if (updatedStock.length > 0) {
                const newQty = updatedStock[0].quantity;
                const expectedQty = initialQty - testQty;
                console.log(`📊 Stock after movement: ${newQty} (expected: ${expectedQty})`);
                
                if (newQty === expectedQty) {
                    console.log('✅ SUCCESS: Trigger correctly deducted stock!\n');
                } else {
                    console.error(`❌ FAILED: Stock not deducted correctly! Expected ${expectedQty}, got ${newQty}\n`);
                }
                
                return { success: newQty === expectedQty, newQty, expectedQty };
            } else {
                console.error('❌ FAILED: Warehouse stock record not found after movement!\n');
                return { success: false };
            }
        });
        
        // Verify final state
        const finalStock = await query(
            'SELECT quantity FROM tbl_warehouse_stock WHERE warehouse_id = ? AND stock_id = ?',
            [warehouseId, stockId]
        );
        
        console.log(`📊 Final warehouse stock: ${finalStock[0].quantity}`);
        
        // Check recent movements
        const recentMovements = await query(
            'SELECT * FROM tbl_stock_movement WHERE warehouse_id = ? AND stock_id = ? ORDER BY id DESC LIMIT 3',
            [warehouseId, stockId]
        );
        
        console.log('\n📋 Recent movements for this stock:');
        recentMovements.forEach(m => {
            console.log(`  - ${m.movement_type}: ${m.quantity} units (ID: ${m.id})`);
        });
        
        process.exit(result.success ? 0 : 1);
        
    } catch (error) {
        console.error('❌ Error:', error.message);
        console.error(error.stack);
        process.exit(1);
    }
}

testWarehouseDeduction();
