/**
 * Verify and create tbl_inpatient_usage_settled table if it doesn't exist
 */

const { query, testConnection } = require('../config/database');

async function verifyAndCreateTable() {
    try {
        console.log('🔍 Checking if tbl_inpatient_usage_settled table exists...\n');
        
        // Test connection
        const connected = await testConnection();
        if (!connected) {
            throw new Error('Database connection failed');
        }
        
        // Check if table exists
        try {
            await query('SELECT 1 FROM tbl_inpatient_usage_settled LIMIT 1');
            console.log('✅ Table tbl_inpatient_usage_settled already exists!');
            return;
        } catch (error) {
            if (error.code === 'ER_NO_SUCH_TABLE') {
                console.log('⚠️  Table does not exist. Creating it now...\n');
                
                // Create the table
                const createTableSQL = `
                    CREATE TABLE \`tbl_inpatient_usage_settled\` (
                        \`id\` INT NOT NULL AUTO_INCREMENT,
                        \`original_usage_id\` INT NULL COMMENT 'Reference to original usage record (if available)',
                        \`inpatient_id\` INT NOT NULL,
                        \`item_type\` ENUM('product', 'service') NOT NULL DEFAULT 'product',
                        \`item_id\` INT NOT NULL COMMENT 'ID of product or service',
                        \`product_name\` VARCHAR(255) NOT NULL COMMENT 'Product/service name at time of settlement (for archival)',
                        \`quantity\` DECIMAL(10,2) NOT NULL DEFAULT 1.00,
                        \`price\` DECIMAL(10,2) NOT NULL,
                        \`notes\` TEXT NULL,
                        \`usage_date\` DATE NOT NULL,
                        \`settlement_deposit_id\` INT NULL COMMENT 'Reference to deposit record that settled this usage',
                        \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'When this record was archived/settled',
                        PRIMARY KEY (\`id\`),
                        INDEX \`idx_original_usage_id\` (\`original_usage_id\`),
                        INDEX \`idx_inpatient_id\` (\`inpatient_id\`),
                        INDEX \`idx_item_type\` (\`item_type\`),
                        INDEX \`idx_item_id\` (\`item_id\`),
                        INDEX \`idx_settlement_deposit_id\` (\`settlement_deposit_id\`),
                        INDEX \`idx_usage_date\` (\`usage_date\`),
                        INDEX \`idx_created_at\` (\`created_at\`),
                        FOREIGN KEY (\`inpatient_id\`) REFERENCES \`tbl_inpatient\`(\`id\`) ON DELETE CASCADE,
                        FOREIGN KEY (\`settlement_deposit_id\`) REFERENCES \`tbl_inpatient_deposits\`(\`id\`) ON DELETE SET NULL
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                    COMMENT = 'Archived inpatient usage records after settlement. Preserves historical data when usage is cleared from main table.';
                `;
                
                await query(createTableSQL);
                console.log('✅ Table tbl_inpatient_usage_settled created successfully!');
            } else {
                throw error;
            }
        }
        
        process.exit(0);
    } catch (error) {
        console.error('\n❌ Error:', error.message);
        console.error('Error details:', error);
        process.exit(1);
    }
}

// Run if called directly
if (require.main === module) {
    verifyAndCreateTable();
}

module.exports = { verifyAndCreateTable };

