/**
 * Comprehensive Database Verification Script
 * Checks all 46 tables, stored procedures, and seeded data
 */

require('dotenv').config();
const { query, testConnection } = require('./config/database');

async function verifyAllTables() {
    console.log('🔍 Comprehensive Database Verification\n');
    console.log('═══════════════════════════════════════════════════════\n');

    try {
        // Test connection first
        console.log('1️⃣  Testing Database Connection...');
        const connected = await testConnection();
        if (!connected) {
            console.log('❌ FAILED: Cannot connect to database\n');
            process.exit(1);
        }
        console.log('✅ PASSED: Database connection successful\n');

        // List all expected tables (46 tables)
        const expectedTables = [
            'tbl_alert',
            'tbl_appointment',
            'tbl_case',
            'tbl_cat_expenses',
            'tbl_catalog',
            'tbl_catalog_2',
            'tbl_customer',
            'tbl_customer_debt',
            'tbl_doctor',
            'tbl_expenses',
            'tbl_form',
            'tbl_images',
            'tbl_inpatient',
            'tbl_inpatient_sale',
            'tbl_inpatient_saledebit',
            'tbl_labour',
            'tbl_labour_balance',
            'tbl_labour_level',
            'tbl_loss',
            'tbl_patient',
            'tbl_patient_sale',
            'tbl_payment_method',
            'tbl_purchase',
            'tbl_purchase_return',
            'tbl_purchase_voucher',
            'tbl_role',
            'tbl_sale',
            'tbl_sale_archive',
            'tbl_sale_draft',
            'tbl_sale_draft_no',
            'tbl_sale_package',
            'tbl_sale_package_no',
            'tbl_sale_return',
            'tbl_sale_return_item',
            'tbl_service',
            'tbl_service_catalog',
            'tbl_shopinfo',
            'tbl_stock',
            'tbl_stock_expense',
            'tbl_supplier',
            'tbl_supplier_debt',
            'tbl_treatment',
            'tbl_unit',
            'tbl_user',
            'tbl_user_log',
            'tbl_voucher',
            'tbl_voucher_archive',
            'tbl_voucher_return'
        ];

        console.log(`2️⃣  Checking Tables (Expected: ${expectedTables.length})...`);
        
        // Get all tables in database
        const tables = await query('SHOW TABLES');
        const existingTableNames = tables.map(t => Object.values(t)[0]);
        
        console.log(`   Found: ${existingTableNames.length} tables\n`);

        // Check each table
        const missingTables = [];
        const existingTables = [];
        
        for (const tableName of expectedTables) {
            if (existingTableNames.includes(tableName)) {
                existingTables.push(tableName);
                console.log(`   ✅ ${tableName}`);
            } else {
                missingTables.push(tableName);
                console.log(`   ❌ ${tableName} - MISSING`);
            }
        }

        console.log(`\n   Summary: ${existingTables.length}/${expectedTables.length} tables exist`);
        
        if (missingTables.length > 0) {
            console.log(`   ⚠️  Missing Tables: ${missingTables.join(', ')}\n`);
        } else {
            console.log(`   ✅ All tables exist!\n`);
        }

        // Check stored procedures
        console.log('3️⃣  Checking Stored Procedures...');
        const procedures = await query(
            'SHOW PROCEDURE STATUS WHERE Db = ?',
            [process.env.DB_NAME || 'clinic_pro_db']
        );
        
        const expectedProcs = [
            'auto_purchase_id',
            'AutoGeneratevoucherNo_purchase',
            'AutoGeneratevoucherNo_Sale'
        ];
        
        const existingProcs = procedures.map(p => p.Name);
        expectedProcs.forEach(proc => {
            if (existingProcs.includes(proc)) {
                console.log(`   ✅ ${proc}`);
            } else {
                console.log(`   ❌ ${proc} - MISSING`);
            }
        });
        console.log('');

        // Check seeded data
        console.log('4️⃣  Checking Seeded Data...');
        
        const checks = [
            { table: 'tbl_user', expected: 1, desc: 'Users' },
            { table: 'tbl_role', expected: 1, desc: 'Roles' },
            { table: 'tbl_payment_method', expected: 5, desc: 'Payment Methods' },
            { table: 'tbl_catalog', expected: 1, desc: 'Categories' },
            { table: 'tbl_service_catalog', expected: 1, desc: 'Service Categories' },
            { table: 'tbl_cat_expenses', expected: 1, desc: 'Expense Categories' },
            { table: 'tbl_unit', expected: 1, desc: 'Units' },
            { table: 'tbl_doctor', expected: 1, desc: 'Doctors' },
            { table: 'tbl_case', expected: 1, desc: 'Medical Cases' },
            { table: 'tbl_shopinfo', expected: 1, desc: 'Shop Info' },
            { table: 'tbl_stock', expected: 1, desc: 'Sample Stock' }
        ];

        for (const check of checks) {
            const result = await query(`SELECT COUNT(*) as count FROM ${check.table}`);
            const count = result[0].count;
            if (count >= check.expected) {
                console.log(`   ✅ ${check.desc}: ${count} records`);
            } else {
                console.log(`   ⚠️  ${check.desc}: ${count} records (expected at least ${check.expected})`);
            }
        }
        console.log('');

        // Check indexes
        console.log('5️⃣  Checking Indexes...');
        const indexes = await query('SHOW INDEX FROM tbl_sale');
        console.log(`   ✅ Found ${indexes.length} indexes on tbl_sale`);
        
        const stockIndexes = await query('SHOW INDEX FROM tbl_stock');
        console.log(`   ✅ Found ${stockIndexes.length} indexes on tbl_stock`);
        console.log('');

        // Check character set
        console.log('6️⃣  Checking Character Set...');
        const charset = await query(
            'SELECT DEFAULT_CHARACTER_SET_NAME, DEFAULT_COLLATION_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = ?',
            [process.env.DB_NAME || 'clinic_pro_db']
        );
        
        if (charset.length > 0) {
            console.log(`   ✅ Character Set: ${charset[0].DEFAULT_CHARACTER_SET_NAME}`);
            console.log(`   ✅ Collation: ${charset[0].DEFAULT_COLLATION_NAME}`);
        }
        console.log('');

        // Test basic CRUD operations
        console.log('7️⃣  Testing CRUD Operations...');
        
        // Create
        const insertResult = await query(
            'INSERT INTO tbl_catalog (name) VALUES (?)',
            ['VERIFICATION_TEST']
        );
        const testId = insertResult.insertId;
        console.log(`   ✅ CREATE: Inserted record with ID ${testId}`);
        
        // Read
        const readResult = await query('SELECT * FROM tbl_catalog WHERE id = ?', [testId]);
        console.log(`   ✅ READ: Retrieved ${readResult.length} record(s)`);
        
        // Update
        await query('UPDATE tbl_catalog SET name = ? WHERE id = ?', ['UPDATED_TEST', testId]);
        console.log(`   ✅ UPDATE: Record updated`);
        
        // Delete
        await query('DELETE FROM tbl_catalog WHERE id = ?', [testId]);
        console.log(`   ✅ DELETE: Record deleted`);
        console.log('');

        // Test stored procedures
        console.log('8️⃣  Testing Stored Procedures...');
        try {
            const saleVoucherResult = await query('CALL AutoGeneratevoucherNo_Sale()');
            const voucherNo = saleVoucherResult[0][0].voucher;
            console.log(`   ✅ AutoGeneratevoucherNo_Sale: Next voucher = ${voucherNo}`);
        } catch (error) {
            console.log(`   ❌ AutoGeneratevoucherNo_Sale: ${error.message}`);
        }
        console.log('');

        // Final Summary
        console.log('═══════════════════════════════════════════════════════');
        console.log('📊 VERIFICATION SUMMARY');
        console.log('═══════════════════════════════════════════════════════');
        console.log(`✅ Tables: ${existingTables.length}/${expectedTables.length}`);
        console.log(`✅ Stored Procedures: ${existingProcs.length}/${expectedProcs.length}`);
        console.log(`✅ Character Set: UTF-8`);
        console.log(`✅ CRUD Operations: Working`);
        console.log('═══════════════════════════════════════════════════════\n');

        if (missingTables.length === 0 && existingProcs.length === expectedProcs.length) {
            console.log('🎉 SUCCESS: Database is fully configured and ready!\n');
            process.exit(0);
        } else {
            console.log('⚠️  WARNING: Some components are missing. Review the log above.\n');
            process.exit(0);
        }

    } catch (error) {
        console.error('❌ VERIFICATION FAILED:', error.message);
        console.error(error);
        process.exit(1);
    }
}

// Run verification
verifyAllTables();

