-- ============================================
-- LABORATORY & DIAGNOSTIC INTEGRATION
-- Professional Hospital Lab Management System
-- ============================================

USE clinic_pro_db;

-- ============================================
-- Table: tbl_lab_test_catalog
-- Master list of all available lab tests
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_lab_test_catalog` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `test_code` VARCHAR(20) NOT NULL UNIQUE,
    `test_name` VARCHAR(100) NOT NULL,
    `category` VARCHAR(50) NULL COMMENT 'Hematology, Biochemistry, Microbiology, etc.',
    `specimen_type` VARCHAR(50) NULL COMMENT 'Blood, Urine, Stool, etc.',
    `normal_range_min` DECIMAL(10,2) NULL,
    `normal_range_max` DECIMAL(10,2) NULL,
    `unit` VARCHAR(20) NULL COMMENT 'mg/dL, mmol/L, etc.',
    `turnaround_time` INT NULL COMMENT 'Expected hours for results',
    `price` DECIMAL(10,2) NOT NULL DEFAULT 0,
    `critical_low` DECIMAL(10,2) NULL COMMENT 'Alert if below this value',
    `critical_high` DECIMAL(10,2) NULL COMMENT 'Alert if above this value',
    `preparation_required` TEXT NULL COMMENT 'Patient preparation instructions',
    `is_active` TINYINT(1) DEFAULT 1,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_test_code` (`test_code`),
    INDEX `idx_category` (`category`),
    INDEX `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_lab_orders
-- Lab test orders from doctors
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_lab_orders` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `order_number` VARCHAR(50) NOT NULL UNIQUE,
    `patient_id` VARCHAR(50) NOT NULL,
    `doctor_id` INT NULL,
    `order_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `priority` ENUM('Routine', 'Urgent', 'STAT', 'Emergency') DEFAULT 'Routine',
    `clinical_info` TEXT NULL COMMENT 'Clinical indication/reason for test',
    `status` ENUM('Ordered', 'Specimen_Collected', 'In_Progress', 'Completed', 'Cancelled') DEFAULT 'Ordered',
    `collection_date` DATETIME NULL,
    `collected_by` INT NULL COMMENT 'User ID of person who collected specimen',
    `notes` TEXT NULL,
    `total_amount` DECIMAL(10,2) DEFAULT 0,
    `created_by` INT NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_order_number` (`order_number`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_doctor` (`doctor_id`),
    INDEX `idx_order_date` (`order_date`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`doctor_id`) REFERENCES `tbl_doctor`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_lab_order_details
-- Individual tests within an order
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_lab_order_details` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `order_id` INT NOT NULL,
    `test_id` INT NOT NULL,
    `test_code` VARCHAR(20) NOT NULL,
    `test_name` VARCHAR(100) NOT NULL,
    `price` DECIMAL(10,2) NOT NULL,
    `specimen_id` VARCHAR(50) NULL COMMENT 'Barcode/ID of specimen',
    `status` ENUM('Pending', 'In_Progress', 'Completed', 'Rejected') DEFAULT 'Pending',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_order` (`order_id`),
    INDEX `idx_test` (`test_id`),
    INDEX `idx_specimen` (`specimen_id`),
    FOREIGN KEY (`order_id`) REFERENCES `tbl_lab_orders`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`test_id`) REFERENCES `tbl_lab_test_catalog`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_lab_results
-- Lab test results with quality control
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_lab_results` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `order_detail_id` INT NOT NULL,
    `order_id` INT NOT NULL,
    `patient_id` VARCHAR(50) NOT NULL,
    `test_id` INT NOT NULL,
    `result_value` VARCHAR(500) NULL,
    `result_numeric` DECIMAL(10,4) NULL COMMENT 'For numeric values',
    `result_text` TEXT NULL COMMENT 'For descriptive results',
    `unit` VARCHAR(20) NULL,
    `normal_range` VARCHAR(100) NULL,
    `flag` ENUM('Normal', 'Low', 'High', 'Critical_Low', 'Critical_High', 'Abnormal') DEFAULT 'Normal',
    `is_critical` TINYINT(1) DEFAULT 0,
    `notes` TEXT NULL,
    `performed_by` INT NOT NULL COMMENT 'Lab technician user ID',
    `performed_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `verified_by` INT NULL COMMENT 'Supervisor/pathologist user ID',
    `verified_date` DATETIME NULL,
    `status` ENUM('Preliminary', 'Final', 'Corrected', 'Cancelled') DEFAULT 'Preliminary',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_order_detail` (`order_detail_id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_test` (`test_id`),
    INDEX `idx_performed_date` (`performed_date`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`order_detail_id`) REFERENCES `tbl_lab_order_details`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`order_id`) REFERENCES `tbl_lab_orders`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`test_id`) REFERENCES `tbl_lab_test_catalog`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_lab_quality_control
-- Quality control tracking
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_lab_quality_control` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `test_id` INT NOT NULL,
    `control_type` ENUM('Normal', 'Abnormal_Low', 'Abnormal_High') NOT NULL,
    `lot_number` VARCHAR(50) NULL,
    `expected_value` DECIMAL(10,4) NULL,
    `measured_value` DECIMAL(10,4) NOT NULL,
    `acceptable_range_min` DECIMAL(10,4) NULL,
    `acceptable_range_max` DECIMAL(10,4) NULL,
    `is_within_range` TINYINT(1) DEFAULT 1,
    `performed_by` INT NOT NULL,
    `performed_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_test` (`test_id`),
    INDEX `idx_performed_date` (`performed_date`),
    FOREIGN KEY (`test_id`) REFERENCES `tbl_lab_test_catalog`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_lab_critical_alerts
-- Critical value notifications
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_lab_critical_alerts` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `result_id` INT NOT NULL,
    `patient_id` VARCHAR(50) NOT NULL,
    `test_name` VARCHAR(100) NOT NULL,
    `critical_value` VARCHAR(100) NOT NULL,
    `alert_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `notified_doctor_id` INT NULL,
    `notification_method` VARCHAR(50) NULL COMMENT 'Phone, SMS, Email, System Alert',
    `acknowledged_by` INT NULL,
    `acknowledged_date` DATETIME NULL,
    `action_taken` TEXT NULL,
    `status` ENUM('Pending', 'Notified', 'Acknowledged', 'Resolved') DEFAULT 'Pending',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_result` (`result_id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_status` (`status`),
    INDEX `idx_alert_date` (`alert_date`),
    FOREIGN KEY (`result_id`) REFERENCES `tbl_lab_results`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Insert Sample Lab Test Catalog
-- ============================================
INSERT INTO `tbl_lab_test_catalog` (`test_code`, `test_name`, `category`, `specimen_type`, `normal_range_min`, `normal_range_max`, `unit`, `turnaround_time`, `price`, `critical_low`, `critical_high`) VALUES
-- Hematology
('CBC', 'Complete Blood Count', 'Hematology', 'Blood', NULL, NULL, NULL, 2, 15000, NULL, NULL),
('HGB', 'Hemoglobin', 'Hematology', 'Blood', 12.0, 16.0, 'g/dL', 2, 5000, 7.0, 20.0),
('WBC', 'White Blood Cell Count', 'Hematology', 'Blood', 4.0, 11.0, '10^3/µL', 2, 5000, 2.0, 30.0),
('PLT', 'Platelet Count', 'Hematology', 'Blood', 150, 400, '10^3/µL', 2, 5000, 50, 1000),
('ESR', 'Erythrocyte Sedimentation Rate', 'Hematology', 'Blood', 0, 20, 'mm/hr', 3, 3000, NULL, NULL),

-- Biochemistry
('GLU', 'Glucose (Fasting)', 'Biochemistry', 'Blood', 70, 100, 'mg/dL', 1, 3000, 40, 400),
('CREAT', 'Creatinine', 'Biochemistry', 'Blood', 0.6, 1.2, 'mg/dL', 2, 5000, 0.3, 10.0),
('BUN', 'Blood Urea Nitrogen', 'Biochemistry', 'Blood', 7, 20, 'mg/dL', 2, 5000, NULL, 100),
('ALT', 'Alanine Aminotransferase', 'Biochemistry', 'Blood', 7, 56, 'U/L', 3, 6000, NULL, 300),
('AST', 'Aspartate Aminotransferase', 'Biochemistry', 'Blood', 10, 40, 'U/L', 3, 6000, NULL, 300),
('CHOL', 'Total Cholesterol', 'Biochemistry', 'Blood', 0, 200, 'mg/dL', 3, 7000, NULL, NULL),
('TG', 'Triglycerides', 'Biochemistry', 'Blood', 0, 150, 'mg/dL', 3, 7000, NULL, NULL),
('HDL', 'HDL Cholesterol', 'Biochemistry', 'Blood', 40, 60, 'mg/dL', 3, 7000, NULL, NULL),
('LDL', 'LDL Cholesterol', 'Biochemistry', 'Blood', 0, 100, 'mg/dL', 3, 7000, NULL, NULL),
('NA', 'Sodium', 'Biochemistry', 'Blood', 136, 145, 'mmol/L', 2, 4000, 120, 160),
('K', 'Potassium', 'Biochemistry', 'Blood', 3.5, 5.0, 'mmol/L', 2, 4000, 2.5, 6.5),

-- Liver Function
('TBIL', 'Total Bilirubin', 'Biochemistry', 'Blood', 0.1, 1.2, 'mg/dL', 3, 5000, NULL, 15),
('DBIL', 'Direct Bilirubin', 'Biochemistry', 'Blood', 0, 0.3, 'mg/dL', 3, 5000, NULL, NULL),
('ALP', 'Alkaline Phosphatase', 'Biochemistry', 'Blood', 30, 120, 'U/L', 3, 5000, NULL, NULL),
('TP', 'Total Protein', 'Biochemistry', 'Blood', 6.0, 8.3, 'g/dL', 3, 4000, NULL, NULL),
('ALB', 'Albumin', 'Biochemistry', 'Blood', 3.5, 5.5, 'g/dL', 3, 4000, NULL, NULL),

-- Thyroid Function
('TSH', 'Thyroid Stimulating Hormone', 'Endocrinology', 'Blood', 0.4, 4.0, 'mIU/L', 24, 12000, NULL, NULL),
('T3', 'Triiodothyronine (T3)', 'Endocrinology', 'Blood', 80, 200, 'ng/dL', 24, 10000, NULL, NULL),
('T4', 'Thyroxine (T4)', 'Endocrinology', 'Blood', 5.0, 12.0, 'µg/dL', 24, 10000, NULL, NULL),

-- Urine Tests
('URINE', 'Urinalysis Complete', 'Urinalysis', 'Urine', NULL, NULL, NULL, 1, 5000, NULL, NULL),
('UCG', 'Urine Culture', 'Microbiology', 'Urine', NULL, NULL, NULL, 48, 15000, NULL, NULL),

-- Microbiology
('BLOODC', 'Blood Culture', 'Microbiology', 'Blood', NULL, NULL, NULL, 72, 25000, NULL, NULL),
('SPUTUMC', 'Sputum Culture', 'Microbiology', 'Sputum', NULL, NULL, NULL, 48, 15000, NULL, NULL),

-- Serology
('HBsAg', 'Hepatitis B Surface Antigen', 'Serology', 'Blood', NULL, NULL, NULL, 24, 12000, NULL, NULL),
('HCV', 'Hepatitis C Antibody', 'Serology', 'Blood', NULL, NULL, NULL, 24, 15000, NULL, NULL),
('HIV', 'HIV Antibody Test', 'Serology', 'Blood', NULL, NULL, NULL, 24, 10000, NULL, NULL);

-- ============================================
-- Create Auto-increment Sequence for Order Numbers
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_lab_sequence` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `last_order_number` INT DEFAULT 0,
    `year` INT DEFAULT YEAR(CURDATE()),
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `tbl_lab_sequence` (`last_order_number`, `year`) VALUES (0, YEAR(CURDATE()))
ON DUPLICATE KEY UPDATE `last_order_number` = `last_order_number`;

