-- ============================================
-- ADVANCED HOSPITAL OPERATIONS
-- Bed Management, OR Scheduling, Emergency Department
-- ============================================

USE clinic_pro_db;

-- ============================================
-- Table: tbl_departments
-- Hospital departments/units
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_departments` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `dept_code` VARCHAR(20) NOT NULL UNIQUE,
    `dept_name` VARCHAR(100) NOT NULL,
    `dept_type` ENUM('Inpatient', 'Outpatient', 'Emergency', 'ICU', 'Surgery', 'Diagnostic', 'Support') NOT NULL,
    `head_doctor_id` INT NULL,
    `location` VARCHAR(100) NULL,
    `phone` VARCHAR(50) NULL,
    `total_beds` INT DEFAULT 0,
    `is_active` TINYINT(1) DEFAULT 1,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_dept_code` (`dept_code`),
    INDEX `idx_dept_type` (`dept_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_beds
-- Hospital bed inventory and management
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_beds` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `bed_code` VARCHAR(20) NOT NULL UNIQUE,
    `department_id` INT NOT NULL,
    `room_number` VARCHAR(20) NOT NULL,
    `bed_number` VARCHAR(10) NOT NULL,
    `bed_type` ENUM('ICU', 'General', 'Emergency', 'Surgery', 'Private', 'Semi-Private', 'Isolation') NOT NULL,
    `status` ENUM('Available', 'Occupied', 'Maintenance', 'Reserved', 'Cleaning', 'Out_of_Service') DEFAULT 'Available',
    `current_patient_id` VARCHAR(50) NULL,
    `daily_rate` DECIMAL(10,2) DEFAULT 0,
    `features` TEXT NULL COMMENT 'Oxygen, Monitor, etc.',
    `last_maintenance` DATE NULL,
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_bed_code` (`bed_code`),
    INDEX `idx_department` (`department_id`),
    INDEX `idx_status` (`status`),
    INDEX `idx_bed_type` (`bed_type`),
    INDEX `idx_current_patient` (`current_patient_id`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE RESTRICT,
    UNIQUE KEY `unique_bed` (`room_number`, `bed_number`, `department_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_bed_assignments
-- Track bed assignment history
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_bed_assignments` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `bed_id` INT NOT NULL,
    `patient_id` VARCHAR(50) NOT NULL,
    `admission_id` INT NULL,
    `assigned_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `discharged_date` DATETIME NULL,
    `assigned_by` INT NOT NULL COMMENT 'User ID',
    `reason` VARCHAR(255) NULL,
    `status` ENUM('Active', 'Completed', 'Transferred') DEFAULT 'Active',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_bed` (`bed_id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_assigned_date` (`assigned_date`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`bed_id`) REFERENCES `tbl_beds`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_operating_rooms
-- Operating theater management
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_operating_rooms` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `or_code` VARCHAR(20) NOT NULL UNIQUE,
    `or_name` VARCHAR(100) NOT NULL,
    `department_id` INT NULL,
    `or_type` ENUM('General', 'Cardiac', 'Orthopedic', 'Neurosurgery', 'Minor', 'Emergency') NOT NULL,
    `status` ENUM('Available', 'In_Use', 'Maintenance', 'Cleaning', 'Reserved') DEFAULT 'Available',
    `equipment` TEXT NULL COMMENT 'Available equipment',
    `capacity` INT DEFAULT 1 COMMENT 'Simultaneous procedures',
    `is_active` TINYINT(1) DEFAULT 1,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_or_code` (`or_code`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_surgery_schedules
-- Operating room scheduling
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_surgery_schedules` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `schedule_number` VARCHAR(50) NOT NULL UNIQUE,
    `or_id` INT NOT NULL,
    `patient_id` VARCHAR(50) NOT NULL,
    `surgeon_id` INT NOT NULL,
    `anesthesiologist_id` INT NULL,
    `procedure_name` VARCHAR(255) NOT NULL,
    `procedure_code` VARCHAR(50) NULL,
    `scheduled_date` DATE NOT NULL,
    `scheduled_start_time` TIME NOT NULL,
    `scheduled_end_time` TIME NULL,
    `actual_start_time` DATETIME NULL,
    `actual_end_time` DATETIME NULL,
    `priority` ENUM('Elective', 'Urgent', 'Emergency') DEFAULT 'Elective',
    `anesthesia_type` ENUM('General', 'Spinal', 'Epidural', 'Local', 'Sedation') NULL,
    `estimated_duration` INT NULL COMMENT 'Minutes',
    `pre_op_diagnosis` TEXT NULL,
    `status` ENUM('Scheduled', 'Confirmed', 'In_Progress', 'Completed', 'Cancelled', 'Postponed') DEFAULT 'Scheduled',
    `special_requirements` TEXT NULL,
    `notes` TEXT NULL,
    `created_by` INT NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_schedule_number` (`schedule_number`),
    INDEX `idx_or` (`or_id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_surgeon` (`surgeon_id`),
    INDEX `idx_scheduled_date` (`scheduled_date`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`or_id`) REFERENCES `tbl_operating_rooms`(`id`) ON DELETE RESTRICT,
    FOREIGN KEY (`surgeon_id`) REFERENCES `tbl_doctor`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_surgery_team
-- Surgical team members for each procedure
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_surgery_team` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `surgery_id` INT NOT NULL,
    `staff_id` INT NOT NULL COMMENT 'User ID',
    `role` ENUM('Surgeon', 'Assistant_Surgeon', 'Anesthesiologist', 'Nurse', 'Technician', 'Other') NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_surgery` (`surgery_id`),
    INDEX `idx_staff` (`staff_id`),
    FOREIGN KEY (`surgery_id`) REFERENCES `tbl_surgery_schedules`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_surgery_notes
-- Operative notes and documentation
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_surgery_notes` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `surgery_id` INT NOT NULL,
    `pre_op_diagnosis` TEXT NULL,
    `post_op_diagnosis` TEXT NULL,
    `procedure_performed` TEXT NULL,
    `findings` TEXT NULL,
    `complications` TEXT NULL,
    `specimens_sent` TEXT NULL,
    `blood_loss` VARCHAR(50) NULL,
    `fluids_given` TEXT NULL,
    `drains_placed` TEXT NULL,
    `condition_on_leaving_or` VARCHAR(100) NULL,
    `surgeon_notes` TEXT NULL,
    `anesthesia_notes` TEXT NULL,
    `created_by` INT NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_surgery` (`surgery_id`),
    FOREIGN KEY (`surgery_id`) REFERENCES `tbl_surgery_schedules`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_emergency_admissions
-- Emergency Department patient tracking
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_emergency_admissions` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `er_number` VARCHAR(50) NOT NULL UNIQUE,
    `patient_id` VARCHAR(50) NULL,
    `arrival_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `arrival_mode` ENUM('Walk-in', 'Ambulance', 'Police', 'Referral', 'Other') NOT NULL,
    `chief_complaint` TEXT NOT NULL,
    `triage_level` ENUM('1-Resuscitation', '2-Emergent', '3-Urgent', '4-Less_Urgent', '5-Non_Urgent') NOT NULL,
    `triage_time` DATETIME NULL,
    `triaged_by` INT NULL COMMENT 'User ID',
    `vital_signs` TEXT NULL COMMENT 'JSON of vital signs',
    `assigned_doctor_id` INT NULL,
    `assigned_time` DATETIME NULL,
    `treatment_start_time` DATETIME NULL,
    `disposition` ENUM('Admitted', 'Discharged', 'Transferred', 'Left_AMA', 'Deceased', 'Observation') NULL,
    `disposition_time` DATETIME NULL,
    `status` ENUM('Waiting', 'In_Treatment', 'Observation', 'Ready_for_Admission', 'Completed') DEFAULT 'Waiting',
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_er_number` (`er_number`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_arrival_date` (`arrival_date`),
    INDEX `idx_triage_level` (`triage_level`),
    INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_ward_rounds
-- Doctor ward rounds documentation
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_ward_rounds` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `patient_id` VARCHAR(50) NOT NULL,
    `doctor_id` INT NOT NULL,
    `round_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `subjective` TEXT NULL COMMENT 'Patient complaints',
    `objective` TEXT NULL COMMENT 'Physical exam findings',
    `assessment` TEXT NULL COMMENT 'Diagnosis/impression',
    `plan` TEXT NULL COMMENT 'Treatment plan',
    `vital_signs` TEXT NULL COMMENT 'JSON of vital signs',
    `orders` TEXT NULL COMMENT 'New orders',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_doctor` (`doctor_id`),
    INDEX `idx_round_date` (`round_date`),
    FOREIGN KEY (`doctor_id`) REFERENCES `tbl_doctor`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Insert Sample Departments
-- ============================================
INSERT INTO `tbl_departments` (`dept_code`, `dept_name`, `dept_type`, `total_beds`) VALUES
('MED', 'Medicine', 'Inpatient', 30),
('SURG', 'Surgery', 'Inpatient', 25),
('ICU', 'Intensive Care Unit', 'ICU', 10),
('ER', 'Emergency Room', 'Emergency', 15),
('OB-GYN', 'Obstetrics & Gynecology', 'Inpatient', 20),
('PEDS', 'Pediatrics', 'Inpatient', 20),
('ORTHO', 'Orthopedics', 'Inpatient', 15),
('CARDIO', 'Cardiology', 'Inpatient', 12),
('NEURO', 'Neurology', 'Inpatient', 10),
('OPD', 'Outpatient Department', 'Outpatient', 0),
('LAB', 'Laboratory', 'Diagnostic', 0),
('RAD', 'Radiology', 'Diagnostic', 0),
('PHARM', 'Pharmacy', 'Support', 0);

-- ============================================
-- Insert Sample Operating Rooms
-- ============================================
INSERT INTO `tbl_operating_rooms` (`or_code`, `or_name`, `or_type`) VALUES
('OR-1', 'Operating Room 1', 'General'),
('OR-2', 'Operating Room 2', 'General'),
('OR-3', 'Operating Room 3', 'Cardiac'),
('OR-4', 'Operating Room 4', 'Orthopedic'),
('OR-5', 'Minor Surgery Room', 'Minor');

-- ============================================
-- Create sequence tables
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_surgery_sequence` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `last_schedule_number` INT DEFAULT 0,
    `year` INT DEFAULT YEAR(CURDATE()),
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `tbl_surgery_sequence` (`last_schedule_number`, `year`) VALUES (0, YEAR(CURDATE()))
ON DUPLICATE KEY UPDATE `last_schedule_number` = `last_schedule_number`;

CREATE TABLE IF NOT EXISTS `tbl_er_sequence` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `last_er_number` INT DEFAULT 0,
    `year` INT DEFAULT YEAR(CURDATE()),
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `tbl_er_sequence` (`last_er_number`, `year`) VALUES (0, YEAR(CURDATE()))
ON DUPLICATE KEY UPDATE `last_er_number` = `last_er_number`;

