-- ============================================
-- ADVANCED PHARMACY FEATURES
-- Drug Formulary, Interaction Checking, Narcotic Tracking
-- ============================================

USE clinic_pro_db;

-- ============================================
-- Table: tbl_drug_formulary
-- Hospital drug formulary management
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_drug_formulary` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `drug_code` VARCHAR(50) NOT NULL UNIQUE,
    `generic_name` VARCHAR(255) NOT NULL,
    `brand_name` VARCHAR(255) NULL,
    `drug_class` VARCHAR(100) NULL COMMENT 'Antibiotic, Analgesic, etc.',
    `therapeutic_category` VARCHAR(100) NULL,
    `form` VARCHAR(50) NULL COMMENT 'Tablet, Capsule, Injection, Syrup',
    `strength` VARCHAR(50) NULL,
    `route` VARCHAR(50) NULL COMMENT 'Oral, IV, IM, Topical',
    `unit_of_measure` VARCHAR(20) NULL,
    `formulary_status` ENUM('Preferred', 'Non-Preferred', 'Restricted', 'Non-Formulary') DEFAULT 'Preferred',
    `is_controlled_substance` TINYINT(1) DEFAULT 0,
    `dea_schedule` ENUM('Schedule_I', 'Schedule_II', 'Schedule_III', 'Schedule_IV', 'Schedule_V', 'None') DEFAULT 'None',
    `requires_authorization` TINYINT(1) DEFAULT 0,
    `max_daily_dose` VARCHAR(50) NULL,
    `typical_dosage` TEXT NULL,
    `contraindications` TEXT NULL,
    `side_effects` TEXT NULL,
    `storage_requirements` VARCHAR(255) NULL,
    `pregnancy_category` CHAR(1) NULL COMMENT 'A, B, C, D, X',
    `manufacturer` VARCHAR(255) NULL,
    `is_active` TINYINT(1) DEFAULT 1,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_drug_code` (`drug_code`),
    INDEX `idx_generic_name` (`generic_name`),
    INDEX `idx_drug_class` (`drug_class`),
    INDEX `idx_controlled` (`is_controlled_substance`),
    INDEX `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_drug_interactions
-- Drug-drug interaction database
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_drug_interactions` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `drug1_id` INT NOT NULL,
    `drug2_id` INT NOT NULL,
    `interaction_severity` ENUM('Contraindicated', 'Major', 'Moderate', 'Minor') NOT NULL,
    `interaction_type` VARCHAR(100) NULL COMMENT 'Pharmacodynamic, Pharmacokinetic',
    `description` TEXT NULL,
    `clinical_management` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_drug1` (`drug1_id`),
    INDEX `idx_drug2` (`drug2_id`),
    INDEX `idx_severity` (`interaction_severity`),
    FOREIGN KEY (`drug1_id`) REFERENCES `tbl_drug_formulary`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`drug2_id`) REFERENCES `tbl_drug_formulary`(`id`) ON DELETE CASCADE,
    UNIQUE KEY `unique_interaction` (`drug1_id`, `drug2_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_prescription_orders
-- Medication orders from doctors
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_prescription_orders` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `order_number` VARCHAR(50) NOT NULL UNIQUE,
    `patient_id` VARCHAR(50) NOT NULL,
    `doctor_id` INT NOT NULL,
    `order_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `encounter_type` ENUM('Outpatient', 'Inpatient', 'Emergency') DEFAULT 'Outpatient',
    `diagnosis` TEXT NULL,
    `status` ENUM('Active', 'Completed', 'Cancelled', 'On_Hold') DEFAULT 'Active',
    `pharmacy_notes` TEXT NULL,
    `total_amount` DECIMAL(10,2) DEFAULT 0,
    `created_by` INT NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_order_number` (`order_number`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_doctor` (`doctor_id`),
    INDEX `idx_order_date` (`order_date`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`doctor_id`) REFERENCES `tbl_doctor`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_prescription_details
-- Individual medications in prescription
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_prescription_details` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `prescription_id` INT NOT NULL,
    `drug_id` INT NULL,
    `drug_name` VARCHAR(255) NOT NULL,
    `dosage` VARCHAR(100) NOT NULL,
    `frequency` VARCHAR(100) NOT NULL COMMENT 'BID, TID, QID, etc.',
    `route` VARCHAR(50) NOT NULL,
    `duration` VARCHAR(50) NULL COMMENT '7 days, 2 weeks, etc.',
    `quantity` DECIMAL(10,2) NOT NULL,
    `refills` INT DEFAULT 0,
    `instructions` TEXT NULL COMMENT 'Take with food, etc.',
    `indication` VARCHAR(255) NULL,
    `start_date` DATE NULL,
    `end_date` DATE NULL,
    `is_prn` TINYINT(1) DEFAULT 0 COMMENT 'PRN (as needed)',
    `price` DECIMAL(10,2) DEFAULT 0,
    `status` ENUM('Pending', 'Dispensed', 'Cancelled') DEFAULT 'Pending',
    `dispensed_date` DATETIME NULL,
    `dispensed_by` INT NULL COMMENT 'Pharmacist user ID',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_prescription` (`prescription_id`),
    INDEX `idx_drug` (`drug_id`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`prescription_id`) REFERENCES `tbl_prescription_orders`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`drug_id`) REFERENCES `tbl_drug_formulary`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_medication_administration
-- Medication administration record (MAR)
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_medication_administration` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `prescription_detail_id` INT NOT NULL,
    `patient_id` VARCHAR(50) NOT NULL,
    `drug_name` VARCHAR(255) NOT NULL,
    `scheduled_time` DATETIME NOT NULL,
    `actual_time` DATETIME NULL,
    `dose_given` VARCHAR(100) NULL,
    `route` VARCHAR(50) NULL,
    `site` VARCHAR(50) NULL COMMENT 'For injections',
    `administered_by` INT NULL COMMENT 'Nurse user ID',
    `status` ENUM('Scheduled', 'Given', 'Omitted', 'Refused', 'Held') DEFAULT 'Scheduled',
    `reason_not_given` TEXT NULL,
    `patient_response` TEXT NULL,
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_prescription_detail` (`prescription_detail_id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_scheduled_time` (`scheduled_time`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`prescription_detail_id`) REFERENCES `tbl_prescription_details`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_controlled_substance_log
-- Narcotic and controlled substance tracking
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_controlled_substance_log` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `drug_id` INT NOT NULL,
    `drug_name` VARCHAR(255) NOT NULL,
    `dea_schedule` VARCHAR(20) NOT NULL,
    `transaction_type` ENUM('Received', 'Dispensed', 'Wasted', 'Returned', 'Destroyed', 'Adjustment') NOT NULL,
    `transaction_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `quantity` DECIMAL(10,3) NOT NULL,
    `balance_before` DECIMAL(10,3) NOT NULL,
    `balance_after` DECIMAL(10,3) NOT NULL,
    `patient_id` VARCHAR(50) NULL,
    `prescription_id` INT NULL,
    `witnessed_by` INT NULL COMMENT 'Second person witness user ID',
    `performed_by` INT NOT NULL COMMENT 'Pharmacist/Nurse user ID',
    `reason` VARCHAR(255) NULL,
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_drug` (`drug_id`),
    INDEX `idx_transaction_date` (`transaction_date`),
    INDEX `idx_transaction_type` (`transaction_type`),
    INDEX `idx_patient` (`patient_id`),
    FOREIGN KEY (`drug_id`) REFERENCES `tbl_drug_formulary`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_adverse_drug_events
-- Pharmacovigilance and adverse event reporting
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_adverse_drug_events` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `report_number` VARCHAR(50) NOT NULL UNIQUE,
    `patient_id` VARCHAR(50) NOT NULL,
    `drug_id` INT NULL,
    `drug_name` VARCHAR(255) NOT NULL,
    `event_date` DATE NOT NULL,
    `reported_date` DATE NOT NULL DEFAULT (CURDATE()),
    `severity` ENUM('Mild', 'Moderate', 'Severe', 'Life_Threatening', 'Fatal') NOT NULL,
    `event_description` TEXT NOT NULL,
    `outcome` ENUM('Recovered', 'Recovering', 'Not_Recovered', 'Recovered_with_Sequelae', 'Fatal', 'Unknown') NULL,
    `action_taken` TEXT NULL,
    `rechallenge` TINYINT(1) DEFAULT 0 COMMENT 'Drug given again?',
    `rechallenge_outcome` VARCHAR(255) NULL,
    `causality_assessment` ENUM('Certain', 'Probable', 'Possible', 'Unlikely', 'Unrelated') NULL,
    `reported_by` INT NOT NULL,
    `reported_to_authority` TINYINT(1) DEFAULT 0,
    `authority_report_date` DATE NULL,
    `status` ENUM('Open', 'Under_Investigation', 'Closed') DEFAULT 'Open',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_report_number` (`report_number`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_drug` (`drug_id`),
    INDEX `idx_event_date` (`event_date`),
    INDEX `idx_severity` (`severity`),
    FOREIGN KEY (`drug_id`) REFERENCES `tbl_drug_formulary`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_medication_reconciliation
-- Track medications across care transitions
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_medication_reconciliation` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `patient_id` VARCHAR(50) NOT NULL,
    `reconciliation_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `transition_point` ENUM('Admission', 'Transfer', 'Discharge') NOT NULL,
    `performed_by` INT NOT NULL COMMENT 'Pharmacist/Nurse user ID',
    `home_medications` TEXT NULL COMMENT 'JSON array of home medications',
    `hospital_medications` TEXT NULL COMMENT 'JSON array of hospital medications',
    `discrepancies_found` TINYINT(1) DEFAULT 0,
    `discrepancies_description` TEXT NULL,
    `discrepancies_resolved` TINYINT(1) DEFAULT 0,
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_reconciliation_date` (`reconciliation_date`),
    INDEX `idx_transition_point` (`transition_point`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_pharmacy_inventory
-- Enhanced pharmacy stock management
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_pharmacy_inventory` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `drug_id` INT NOT NULL,
    `batch_number` VARCHAR(50) NOT NULL,
    `expiry_date` DATE NOT NULL,
    `quantity_on_hand` DECIMAL(10,2) NOT NULL DEFAULT 0,
    `reorder_level` DECIMAL(10,2) DEFAULT 0,
    `max_stock_level` DECIMAL(10,2) DEFAULT 0,
    `unit_cost` DECIMAL(10,2) DEFAULT 0,
    `location` VARCHAR(100) NULL COMMENT 'Shelf/bin location',
    `supplier_id` INT NULL,
    `received_date` DATE NULL,
    `status` ENUM('Active', 'Expired', 'Quarantine', 'Recalled') DEFAULT 'Active',
    `last_counted_date` DATE NULL,
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_drug` (`drug_id`),
    INDEX `idx_batch` (`batch_number`),
    INDEX `idx_expiry` (`expiry_date`),
    INDEX `idx_status` (`status`),
    FOREIGN KEY (`drug_id`) REFERENCES `tbl_drug_formulary`(`id`) ON DELETE RESTRICT,
    FOREIGN KEY (`supplier_id`) REFERENCES `tbl_supplier`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Insert Sample Drug Formulary Data
-- ============================================
INSERT INTO `tbl_drug_formulary` (`drug_code`, `generic_name`, `brand_name`, `drug_class`, `form`, `strength`, `route`, `formulary_status`, `is_controlled_substance`, `dea_schedule`) VALUES
-- Antibiotics
('AMX500', 'Amoxicillin', 'Amoxil', 'Antibiotic', 'Capsule', '500mg', 'Oral', 'Preferred', 0, 'None'),
('AZI250', 'Azithromycin', 'Zithromax', 'Antibiotic', 'Tablet', '250mg', 'Oral', 'Preferred', 0, 'None'),
('CIPR500', 'Ciprofloxacin', 'Cipro', 'Antibiotic', 'Tablet', '500mg', 'Oral', 'Preferred', 0, 'None'),
('CEFT1G', 'Ceftriaxone', 'Rocephin', 'Antibiotic', 'Injection', '1g', 'IV/IM', 'Preferred', 0, 'None'),

-- Analgesics
('PARA500', 'Paracetamol', 'Tylenol', 'Analgesic', 'Tablet', '500mg', 'Oral', 'Preferred', 0, 'None'),
('IBU400', 'Ibuprofen', 'Advil', 'NSAID', 'Tablet', '400mg', 'Oral', 'Preferred', 0, 'None'),
('TRAM50', 'Tramadol', 'Ultram', 'Opioid Analgesic', 'Tablet', '50mg', 'Oral', 'Restricted', 1, 'Schedule_IV'),
('MORPH10', 'Morphine', 'MS Contin', 'Opioid Analgesic', 'Injection', '10mg/mL', 'IV/IM', 'Restricted', 1, 'Schedule_II'),

-- Cardiovascular
('ATEN50', 'Atenolol', 'Tenormin', 'Beta Blocker', 'Tablet', '50mg', 'Oral', 'Preferred', 0, 'None'),
('AMLO5', 'Amlodipine', 'Norvasc', 'Calcium Channel Blocker', 'Tablet', '5mg', 'Oral', 'Preferred', 0, 'None'),
('ENAL5', 'Enalapril', 'Vasotec', 'ACE Inhibitor', 'Tablet', '5mg', 'Oral', 'Preferred', 0, 'None'),
('FUROS40', 'Furosemide', 'Lasix', 'Diuretic', 'Tablet', '40mg', 'Oral', 'Preferred', 0, 'None'),

-- Diabetes
('METF500', 'Metformin', 'Glucophage', 'Antidiabetic', 'Tablet', '500mg', 'Oral', 'Preferred', 0, 'None'),
('GLIB5', 'Glibenclamide', 'Diabeta', 'Antidiabetic', 'Tablet', '5mg', 'Oral', 'Preferred', 0, 'None'),
('INSREG', 'Insulin Regular', 'Humulin R', 'Insulin', 'Injection', '100units/mL', 'SC', 'Preferred', 0, 'None'),

-- Gastrointestinal
('OMEP20', 'Omeprazole', 'Prilosec', 'PPI', 'Capsule', '20mg', 'Oral', 'Preferred', 0, 'None'),
('RANI150', 'Ranitidine', 'Zantac', 'H2 Blocker', 'Tablet', '150mg', 'Oral', 'Preferred', 0, 'None'),
('METRO400', 'Metronidazole', 'Flagyl', 'Antibiotic', 'Tablet', '400mg', 'Oral', 'Preferred', 0, 'None'),

-- Respiratory
('SALB100', 'Salbutamol', 'Ventolin', 'Bronchodilator', 'Inhaler', '100mcg', 'Inhalation', 'Preferred', 0, 'None'),
('PRED5', 'Prednisolone', 'Deltasone', 'Corticosteroid', 'Tablet', '5mg', 'Oral', 'Preferred', 0, 'None');

-- ============================================
-- Create Prescription Sequence
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_prescription_sequence` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `last_order_number` INT DEFAULT 0,
    `year` INT DEFAULT YEAR(CURDATE()),
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `tbl_prescription_sequence` (`last_order_number`, `year`) VALUES (0, YEAR(CURDATE()))
ON DUPLICATE KEY UPDATE `last_order_number` = `last_order_number`;

