-- ============================================
-- ENHANCED REPORTING & ANALYTICS
-- Quality Metrics, Clinical Indicators, Hospital KPIs
-- ============================================

USE clinic_pro_db;

-- ============================================
-- Table: tbl_quality_metrics
-- Hospital quality indicators tracking
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_quality_metrics` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `metric_code` VARCHAR(50) NOT NULL,
    `metric_name` VARCHAR(255) NOT NULL,
    `metric_category` ENUM('Clinical_Quality', 'Patient_Safety', 'Operational', 'Financial', 'Patient_Satisfaction') NOT NULL,
    `department_id` INT NULL,
    `measurement_period` ENUM('Daily', 'Weekly', 'Monthly', 'Quarterly', 'Yearly') NOT NULL,
    `period_start` DATE NOT NULL,
    `period_end` DATE NOT NULL,
    `numerator` DECIMAL(10,2) NULL COMMENT 'Cases meeting criteria',
    `denominator` DECIMAL(10,2) NULL COMMENT 'Total eligible cases',
    `actual_value` DECIMAL(10,2) NOT NULL,
    `target_value` DECIMAL(10,2) NULL,
    `unit` VARCHAR(20) NULL COMMENT 'Percentage, Number, Rate',
    `meets_target` TINYINT(1) DEFAULT 0,
    `variance` DECIMAL(10,2) NULL COMMENT 'Difference from target',
    `trend` ENUM('Improving', 'Stable', 'Declining') NULL,
    `notes` TEXT NULL,
    `recorded_by` INT NOT NULL,
    `recorded_date` DATE NOT NULL DEFAULT (CURDATE()),
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_metric_code` (`metric_code`),
    INDEX `idx_category` (`metric_category`),
    INDEX `idx_department` (`department_id`),
    INDEX `idx_period` (`period_start`, `period_end`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_clinical_indicators
-- Key clinical performance indicators
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_clinical_indicators` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `indicator_name` VARCHAR(255) NOT NULL,
    `indicator_type` ENUM('Infection_Rate', 'Readmission_Rate', 'Mortality_Rate', 'Complication_Rate', 'Length_of_Stay', 'Other') NOT NULL,
    `department_id` INT NULL,
    `month` INT NOT NULL,
    `year` INT NOT NULL,
    `total_cases` INT NOT NULL,
    `affected_cases` INT NOT NULL,
    `rate` DECIMAL(5,2) NOT NULL COMMENT 'Percentage or average',
    `benchmark` DECIMAL(5,2) NULL COMMENT 'Industry standard',
    `is_within_benchmark` TINYINT(1) DEFAULT 1,
    `action_plan` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_type` (`indicator_type`),
    INDEX `idx_department` (`department_id`),
    INDEX `idx_period` (`year`, `month`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_infection_surveillance
-- Hospital-acquired infection tracking
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_infection_surveillance` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `patient_id` VARCHAR(50) NOT NULL,
    `infection_type` ENUM('UTI', 'Surgical_Site', 'Bloodstream', 'Pneumonia', 'C_Difficile', 'MRSA', 'Other') NOT NULL,
    `infection_site` VARCHAR(100) NULL,
    `onset_date` DATE NOT NULL,
    `detected_date` DATE NOT NULL,
    `days_since_admission` INT NULL,
    `risk_factors` TEXT NULL,
    `causative_organism` VARCHAR(100) NULL,
    `department_id` INT NULL,
    `associated_procedure` VARCHAR(255) NULL,
    `device_related` TINYINT(1) DEFAULT 0 COMMENT 'Catheter, ventilator, etc.',
    `device_type` VARCHAR(100) NULL,
    `treatment_started` DATE NULL,
    `outcome` ENUM('Resolved', 'Ongoing', 'Transferred', 'Deceased') NULL,
    `reported_by` INT NOT NULL,
    `investigated` TINYINT(1) DEFAULT 0,
    `preventable` TINYINT(1) NULL,
    `corrective_actions` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_type` (`infection_type`),
    INDEX `idx_onset_date` (`onset_date`),
    INDEX `idx_department` (`department_id`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_readmissions
-- 30-day readmission tracking
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_readmissions` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `patient_id` VARCHAR(50) NOT NULL,
    `original_admission_date` DATE NOT NULL,
    `original_discharge_date` DATE NOT NULL,
    `original_diagnosis` TEXT NULL,
    `readmission_date` DATE NOT NULL,
    `days_since_discharge` INT NOT NULL,
    `readmission_diagnosis` TEXT NULL,
    `is_related` TINYINT(1) DEFAULT 1 COMMENT 'Related to original admission',
    `is_planned` TINYINT(1) DEFAULT 0,
    `department_id` INT NULL,
    `readmission_reason` TEXT NULL,
    `preventability_assessment` ENUM('Preventable', 'Potentially_Preventable', 'Not_Preventable', 'Under_Review') NULL,
    `root_cause` TEXT NULL,
    `action_plan` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_original_discharge` (`original_discharge_date`),
    INDEX `idx_readmission_date` (`readmission_date`),
    INDEX `idx_department` (`department_id`),
    INDEX `idx_days_since` (`days_since_discharge`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_patient_satisfaction
-- Patient satisfaction surveys and scores
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_patient_satisfaction` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `patient_id` VARCHAR(50) NOT NULL,
    `survey_date` DATE NOT NULL,
    `encounter_type` ENUM('Outpatient', 'Inpatient', 'Emergency', 'Surgery') NOT NULL,
    `department_id` INT NULL,
    `overall_rating` INT NOT NULL COMMENT '1-5 scale',
    `staff_courtesy` INT NULL COMMENT '1-5 scale',
    `communication` INT NULL COMMENT '1-5 scale',
    `cleanliness` INT NULL COMMENT '1-5 scale',
    `wait_time` INT NULL COMMENT '1-5 scale',
    `pain_management` INT NULL COMMENT '1-5 scale',
    `would_recommend` TINYINT(1) NULL,
    `complaints` TEXT NULL,
    `compliments` TEXT NULL,
    `suggestions` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_survey_date` (`survey_date`),
    INDEX `idx_department` (`department_id`),
    INDEX `idx_overall_rating` (`overall_rating`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_financial_metrics
-- Financial performance tracking
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_financial_metrics` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `metric_type` ENUM('Revenue', 'Cost', 'Profit', 'Utilization', 'Collections', 'Bad_Debt') NOT NULL,
    `department_id` INT NULL,
    `period_type` ENUM('Daily', 'Weekly', 'Monthly', 'Quarterly', 'Yearly') NOT NULL,
    `period_start` DATE NOT NULL,
    `period_end` DATE NOT NULL,
    `amount` DECIMAL(15,2) NOT NULL,
    `budget` DECIMAL(15,2) NULL,
    `variance` DECIMAL(15,2) NULL,
    `variance_percentage` DECIMAL(5,2) NULL,
    `patient_days` INT NULL COMMENT 'For calculating per-patient metrics',
    `cost_per_patient_day` DECIMAL(10,2) NULL,
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_metric_type` (`metric_type`),
    INDEX `idx_department` (`department_id`),
    INDEX `idx_period` (`period_start`, `period_end`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_operational_metrics
-- Operational efficiency metrics
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_operational_metrics` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `metric_date` DATE NOT NULL,
    `department_id` INT NULL,
    `bed_occupancy_rate` DECIMAL(5,2) NULL COMMENT 'Percentage',
    `average_length_of_stay` DECIMAL(5,2) NULL COMMENT 'Days',
    `er_wait_time` INT NULL COMMENT 'Minutes',
    `or_utilization_rate` DECIMAL(5,2) NULL COMMENT 'Percentage',
    `surgery_cancellation_rate` DECIMAL(5,2) NULL COMMENT 'Percentage',
    `staff_to_patient_ratio` DECIMAL(5,2) NULL,
    `emergency_admissions` INT NULL,
    `elective_admissions` INT NULL,
    `total_discharges` INT NULL,
    `bed_turnover_rate` DECIMAL(5,2) NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_metric_date` (`metric_date`),
    INDEX `idx_department` (`department_id`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_staff_productivity
-- Staff performance and productivity tracking
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_staff_productivity` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `user_id` INT NOT NULL,
    `staff_type` ENUM('Doctor', 'Nurse', 'Pharmacist', 'Lab_Tech', 'Other') NOT NULL,
    `department_id` INT NULL,
    `metric_date` DATE NOT NULL,
    `patients_seen` INT NULL,
    `procedures_performed` INT NULL,
    `prescriptions_processed` INT NULL,
    `tests_conducted` INT NULL,
    `hours_worked` DECIMAL(5,2) NULL,
    `overtime_hours` DECIMAL(5,2) NULL,
    `revenue_generated` DECIMAL(10,2) NULL,
    `patient_satisfaction_score` DECIMAL(3,2) NULL COMMENT '1-5 scale',
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_user` (`user_id`),
    INDEX `idx_metric_date` (`metric_date`),
    INDEX `idx_department` (`department_id`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_incident_reports
-- Safety incidents and near-misses
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_incident_reports` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `incident_number` VARCHAR(50) NOT NULL UNIQUE,
    `incident_date` DATETIME NOT NULL,
    `reported_date` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `incident_type` ENUM('Medication_Error', 'Fall', 'Equipment_Failure', 'Adverse_Event', 'Near_Miss', 'Other') NOT NULL,
    `severity` ENUM('No_Harm', 'Minor', 'Moderate', 'Severe', 'Death') NOT NULL,
    `patient_id` VARCHAR(50) NULL,
    `location` VARCHAR(100) NULL,
    `department_id` INT NULL,
    `description` TEXT NOT NULL,
    `immediate_action` TEXT NULL,
    `reported_by` INT NOT NULL,
    `investigated_by` INT NULL,
    `investigation_status` ENUM('Pending', 'In_Progress', 'Completed') DEFAULT 'Pending',
    `root_cause` TEXT NULL,
    `contributing_factors` TEXT NULL,
    `corrective_actions` TEXT NULL,
    `preventive_measures` TEXT NULL,
    `follow_up_required` TINYINT(1) DEFAULT 0,
    `follow_up_date` DATE NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_incident_number` (`incident_number`),
    INDEX `idx_incident_date` (`incident_date`),
    INDEX `idx_type` (`incident_type`),
    INDEX `idx_severity` (`severity`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_department` (`department_id`),
    FOREIGN KEY (`department_id`) REFERENCES `tbl_departments`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Insert Sample Quality Metrics Definitions
-- ============================================
INSERT INTO `tbl_quality_metrics` (`metric_code`, `metric_name`, `metric_category`, `measurement_period`, `period_start`, `period_end`, `actual_value`, `target_value`, `unit`, `meets_target`, `recorded_by`, `recorded_date`) VALUES
('HAI-01', 'Hospital Acquired Infection Rate', 'Patient_Safety', 'Monthly', '2025-01-01', '2025-01-31', 2.5, 2.0, 'Percentage', 0, 1, '2025-01-31'),
('READ-30', '30-Day Readmission Rate', 'Clinical_Quality', 'Monthly', '2025-01-01', '2025-01-31', 8.5, 10.0, 'Percentage', 1, 1, '2025-01-31'),
('BED-OCC', 'Bed Occupancy Rate', 'Operational', 'Monthly', '2025-01-01', '2025-01-31', 85.0, 80.0, 'Percentage', 1, 1, '2025-01-31'),
('ER-WAIT', 'Emergency Room Wait Time', 'Operational', 'Monthly', '2025-01-01', '2025-01-31', 35.0, 30.0, 'Minutes', 0, 1, '2025-01-31'),
('PSAT-01', 'Patient Satisfaction Score', 'Patient_Satisfaction', 'Monthly', '2025-01-01', '2025-01-31', 4.2, 4.5, 'Score', 0, 1, '2025-01-31');

-- ============================================
-- Create Incident Report Sequence
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_incident_sequence` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `last_incident_number` INT DEFAULT 0,
    `year` INT DEFAULT YEAR(CURDATE()),
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `tbl_incident_sequence` (`last_incident_number`, `year`) VALUES (0, YEAR(CURDATE()))
ON DUPLICATE KEY UPDATE `last_incident_number` = `last_incident_number`;

-- ============================================
-- Create Audit Log Table for Compliance
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_audit_log` (
    `id` BIGINT NOT NULL AUTO_INCREMENT,
    `user_id` INT NULL,
    `username` VARCHAR(50) NULL,
    `action` VARCHAR(100) NOT NULL COMMENT 'CREATE, READ, UPDATE, DELETE',
    `table_name` VARCHAR(100) NOT NULL,
    `record_id` VARCHAR(100) NULL,
    `patient_id` VARCHAR(50) NULL COMMENT 'For patient data access tracking',
    `old_values` TEXT NULL COMMENT 'JSON of old values',
    `new_values` TEXT NULL COMMENT 'JSON of new values',
    `ip_address` VARCHAR(45) NULL,
    `user_agent` VARCHAR(255) NULL,
    `timestamp` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `session_id` VARCHAR(100) NULL,
    PRIMARY KEY (`id`),
    INDEX `idx_user` (`user_id`),
    INDEX `idx_timestamp` (`timestamp`),
    INDEX `idx_table` (`table_name`),
    INDEX `idx_patient` (`patient_id`),
    INDEX `idx_action` (`action`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

