# Database Changelog

All notable database changes for Clinic Pro project are documented in this file.

## [1.0.0] - 2024-10-16

### Added

#### Doctor System Enhancements
- **Photo Upload Support** (`tbl_doctor.photo`)
  - Stores doctor photo file path
  - Supports image upload via multipart/form-data
  - Used in doctor profiles and displays

- **Extended Doctor Information**
  - `email` - Doctor email address
  - `address` - Doctor physical address
  - `license_number` - Medical license number
  - `experience_years` - Years of medical experience
  - `consultation_fee` - Consultation fee amount
  - `is_active` - Active status (1=active, 0=inactive)
  - `created_at` - Record creation timestamp
  - `updated_at` - Last update timestamp

- **Performance Indexes**
  - `idx_doctor_active` - For filtering active/inactive doctors
  - `idx_doctor_specialist` - For specialist type lookups

#### Patient Deposit System
- **Deposit Balance Tracking** (`tbl_patient.deposit_balance`)
  - Stores current patient deposit balance
  - Decimal(10,2) for accurate currency handling
  - Automatically migrated from existing `deposit` column

- **Deposit Transaction Table** (`tbl_deposit_transaction`)
  - Complete transaction history
  - Supports deposits and withdrawals
  - Tracks balance before/after each transaction
  - Links to vouchers and users
  - Audit trail for financial compliance

**Transaction Fields:**
| Field | Type | Description |
|-------|------|-------------|
| id | INT | Primary key |
| patient_id | INT | Reference to patient |
| transaction_type | ENUM | 'deposit' or 'withdrawal' |
| amount | DECIMAL(10,2) | Transaction amount |
| balance_before | DECIMAL(10,2) | Balance before transaction |
| balance_after | DECIMAL(10,2) | Balance after transaction |
| description | TEXT | Transaction description |
| reference_no | VARCHAR(50) | Voucher/receipt reference |
| created_by | INT | User who created transaction |
| created_at | TIMESTAMP | Creation time |
| updated_at | TIMESTAMP | Last update time |

**Indexes:**
- `idx_patient_id` - Patient lookup
- `idx_transaction_type` - Filter by type
- `idx_created_at` - Date range queries
- `idx_reference_no` - Reference lookup

#### Inpatient Bed Integration
- **Bed Assignment Fields** (`tbl_inpatient`)
  - `bed_id` - Foreign key to `tbl_beds.id`
  - `bed_number` - Bed number from assignment
  - `bed_type` - Type (General, ICU, Private, etc.)
  - `daily_rate` - Daily bed rate
  - `dr_id` - Foreign key to `tbl_doctor.id`

- **Foreign Key Constraints**
  - `fk_inpatient_bed` - Links to bed records
  - `fk_inpatient_doctor` - Links to doctor records
  - Both use `ON DELETE SET NULL` for safety

- **Performance Indexes**
  - `idx_inpatient_bed_id` - Bed assignment lookups
  - `idx_inpatient_dr_id` - Doctor assignment lookups
  - `idx_inpatient_end_date` - Active patient queries

### Fixed

#### Bed Data Consistency
- **Issue**: Beds marked as 'Occupied' with no patient assigned
- **Fix**: Automated cleanup of inconsistent bed statuses
  - Sets status to 'Available' when no patient
  - Clears `current_patient_id`
  - Clears `admission_date`
- **Impact**: Freed 3 beds for new admissions

### Changed

#### Data Migration
- **Deposit Balance**: Migrated existing `tbl_patient.deposit` values to `deposit_balance`
- **Default Values**: Set `bed_type = 'General'` and `daily_rate = 0.00` for existing inpatient records

## Migration Details

### Migration Files

| File | Description | Tables Affected |
|------|-------------|-----------------|
| 001_add_doctor_photo.sql | Doctor enhancements | tbl_doctor |
| 002_add_deposit_system.sql | Deposit tracking | tbl_patient, tbl_deposit_transaction |
| 003_add_inpatient_bed_fields.sql | Bed integration | tbl_inpatient |
| 004_fix_bed_inconsistencies.sql | Data cleanup | tbl_beds |

### Seed Files

| File | Description | Records Created |
|------|-------------|-----------------|
| 001_seed_doctors.sql | Sample doctors | 5 doctors |
| 002_seed_patients.sql | Sample patients | 10 patients |

## Database Statistics

### Before Migrations
- **Tables**: 50+
- **Doctor Fields**: 5
- **Patient Tracking**: Basic
- **Inpatient System**: Standalone

### After Migrations
- **Tables**: 52 (added tbl_deposit_transaction, tbl_migrations)
- **Doctor Fields**: 14
- **Patient Tracking**: Advanced with deposits
- **Inpatient System**: Integrated with beds and doctors

## Compatibility

### Backward Compatibility
✅ **Fully backward compatible**
- All new fields are nullable or have defaults
- Existing code continues to work
- No breaking changes

### Database Requirements
- MySQL 5.7+ or MariaDB 10.2+
- InnoDB storage engine
- UTF8MB4 character set

## Testing

### Automated Tests
- ✅ Migration execution
- ✅ Seed execution
- ✅ Foreign key constraints
- ✅ Index creation
- ✅ Data consistency

### Manual Testing Required
- [ ] Doctor photo upload
- [ ] Deposit transactions
- [ ] Inpatient bed assignment
- [ ] Report generation with new fields

## Rollback Instructions

### To Rollback Doctor Changes
```sql
ALTER TABLE tbl_doctor 
DROP COLUMN IF EXISTS photo,
DROP COLUMN IF EXISTS email,
DROP COLUMN IF EXISTS address,
DROP COLUMN IF EXISTS license_number,
DROP COLUMN IF EXISTS experience_years,
DROP COLUMN IF EXISTS consultation_fee,
DROP COLUMN IF EXISTS is_active,
DROP COLUMN IF EXISTS created_at,
DROP COLUMN IF EXISTS updated_at;

DROP INDEX IF EXISTS idx_doctor_active ON tbl_doctor;
DROP INDEX IF EXISTS idx_doctor_specialist ON tbl_doctor;
```

### To Rollback Deposit System
```sql
ALTER TABLE tbl_patient DROP COLUMN IF EXISTS deposit_balance;
DROP TABLE IF EXISTS tbl_deposit_transaction;
```

### To Rollback Inpatient Changes
```sql
ALTER TABLE tbl_inpatient 
DROP FOREIGN KEY IF EXISTS fk_inpatient_bed,
DROP FOREIGN KEY IF EXISTS fk_inpatient_doctor,
DROP COLUMN IF EXISTS bed_id,
DROP COLUMN IF EXISTS bed_number,
DROP COLUMN IF EXISTS bed_type,
DROP COLUMN IF EXISTS daily_rate,
DROP COLUMN IF EXISTS dr_id;

DROP INDEX IF EXISTS idx_inpatient_bed_id ON tbl_inpatient;
DROP INDEX IF EXISTS idx_inpatient_dr_id ON tbl_inpatient;
DROP INDEX IF EXISTS idx_inpatient_end_date ON tbl_inpatient;
```

## Performance Impact

### Query Performance
- ✅ **Improved**: Indexed lookups for doctors and inpatients
- ✅ **Improved**: Deposit transaction queries
- ⚠️ **Monitor**: Large deposit transaction tables over time

### Storage Impact
- **Doctor Table**: +~500 bytes per record
- **Patient Table**: +8 bytes per record
- **New Transaction Table**: ~150 bytes per transaction
- **Estimated Growth**: 5-10% database size increase

## Security Considerations

### Data Privacy
- ✅ Doctor email and address protected
- ✅ Deposit transactions audited
- ✅ User tracking for financial transactions
- ⚠️ Ensure photo uploads are validated

### Access Control
- Deposit transactions require authentication
- Photo uploads require authorization
- Audit trail prevents unauthorized changes

## Future Enhancements

### Planned for v1.1.0
- [ ] Doctor availability schedule
- [ ] Deposit transaction reports
- [ ] Bed allocation analytics
- [ ] Doctor performance metrics

### Under Consideration
- [ ] Multi-currency deposit support
- [ ] Automated bed assignment
- [ ] Doctor rating system
- [ ] Deposit interest calculation

## References

### Related Documentation
- [Migration Guide](MIGRATION_GUIDE.md)
- [Database Connection Fix](../DATABASE_CONNECTION_FIX.md)
- [Invoice System](../INVOICE_SYSTEM_COMPLETE.md)
- [Deposit System](../DEPOSIT_SYSTEM_COMPLETE.md)

### API Endpoints Affected
- `POST /api/doctors` - Now supports photo upload
- `PUT /api/doctors/:id` - Now supports photo update
- `POST /api/deposits` - New endpoint
- `GET /api/deposits/patient/:id` - New endpoint
- `POST /api/inpatients` - Now requires bed_id

---

**Maintained by**: Clinic Pro Development Team  
**Last Updated**: October 16, 2024  
**Version**: 1.0.0
