# Database Migration & Seeding Guide

## Overview
This guide explains how to use the migration and seeding system for Clinic Pro database changes.

## Directory Structure

```
database/
├── migrations/                  # Database schema changes
│   ├── 001_add_doctor_photo.sql
│   ├── 002_add_deposit_system.sql
│   ├── 003_add_inpatient_bed_fields.sql
│   ├── 004_fix_bed_inconsistencies.sql
│   └── run-all-migrations.js
├── seeds/                       # Test data
│   ├── 001_seed_doctors.sql
│   ├── 002_seed_patients.sql
│   └── run-all-seeds.js
└── setup-database.js            # Complete setup script
```

## Quick Start

### Option 1: Run Complete Setup (Recommended for new installations)
```bash
cd Clinic-Pro-Web/database
node setup-database.js
```

This will:
1. Run all migrations in order
2. Run all seeds in order
3. Display comprehensive status

### Option 2: Run Migrations Only
```bash
cd Clinic-Pro-Web/database/migrations
node run-all-migrations.js
```

### Option 3: Run Seeds Only
```bash
cd Clinic-Pro-Web/database/seeds
node run-all-seeds.js
```

## Migration System

### How It Works

1. **Migration Tracking**: Creates `tbl_migrations` table to track executed migrations
2. **Idempotent**: Safe to run multiple times (skips already executed migrations)
3. **Sequential**: Migrations run in filename order (001, 002, 003, etc.)
4. **Error Handling**: Stops on errors, shows clear messages

### Creating New Migrations

1. Create a new SQL file in `database/migrations/` with format: `XXX_description.sql`
   - Example: `005_add_prescription_fields.sql`

2. Write SQL with proper safety checks:
```sql
-- Add column safely
ALTER TABLE tbl_patient 
ADD COLUMN IF NOT EXISTS new_field VARCHAR(255) NULL;

-- Create index safely
CREATE INDEX IF NOT EXISTS idx_new_field ON tbl_patient(new_field);
```

3. Run migrations:
```bash
node run-all-migrations.js
```

## Seeding System

### How It Works

1. **Order-based**: Seeds run in filename order (001, 002, 003, etc.)
2. **Duplicate-safe**: Uses `INSERT IGNORE` to prevent duplicate entries
3. **Replayable**: Safe to run multiple times

### Creating New Seeds

1. Create a new SQL file in `database/seeds/` with format: `XXX_description.sql`
   - Example: `003_seed_appointments.sql`

2. Write SQL with `INSERT IGNORE`:
```sql
INSERT IGNORE INTO tbl_appointment (patient_id, doctor_id, date, time, status) VALUES 
('P000001', 1, '2024-10-20', '10:00:00', 'Scheduled'),
('P000002', 2, '2024-10-20', '11:00:00', 'Scheduled');
```

3. Run seeds:
```bash
node run-all-seeds.js
```

## Migration Files Details

### 001_add_doctor_photo.sql
**Purpose**: Add photo and enhanced fields to doctor table

**Changes**:
- `photo` - Doctor photo path
- `email` - Doctor email
- `address` - Doctor address
- `license_number` - Medical license number
- `experience_years` - Years of experience
- `consultation_fee` - Consultation fee amount
- `is_active` - Active status flag
- `created_at` / `updated_at` - Timestamps

**Indexes**:
- `idx_doctor_active` - For filtering active doctors
- `idx_doctor_specialist` - For specialist lookups

### 002_add_deposit_system.sql
**Purpose**: Implement patient deposit management

**Changes**:
- `tbl_patient.deposit_balance` - Current deposit balance
- `tbl_deposit_transaction` - New table for deposit transactions

**Features**:
- Track deposits and withdrawals
- Maintain balance history
- Reference transactions to vouchers
- Audit trail with user tracking

### 003_add_inpatient_bed_fields.sql
**Purpose**: Integrate bed management with inpatient system

**Changes**:
- `bed_id` - Foreign key to tbl_beds
- `bed_number` - Bed number from assignment
- `bed_type` - Type of bed (General, ICU, etc.)
- `daily_rate` - Daily rate for bed
- `dr_id` - Foreign key to tbl_doctor

**Foreign Keys**:
- Links to `tbl_beds` for bed information
- Links to `tbl_doctor` for assigned doctor

### 004_fix_bed_inconsistencies.sql
**Purpose**: Clean up inconsistent bed data

**Changes**:
- Sets beds to 'Available' if marked 'Occupied' but no patient assigned
- Clears `current_patient_id` and `admission_date` for fixed beds

## Seed Files Details

### 001_seed_doctors.sql
Creates 5 sample doctors with different specializations:
- General Physician
- Pediatrician
- Orthopedic Surgeon
- Cardiologist
- Neurologist

### 002_seed_patients.sql
Creates 10 sample patients with:
- Diverse ages and conditions
- Different deposit amounts
- Assigned to various doctors
- Realistic patient data

## Best Practices

### For Migrations

1. **Always use IF NOT EXISTS**
   ```sql
   ALTER TABLE tbl_name ADD COLUMN IF NOT EXISTS col_name VARCHAR(255);
   ```

2. **Add indexes separately**
   ```sql
   CREATE INDEX IF NOT EXISTS idx_name ON tbl_name(col_name);
   ```

3. **Include comments**
   ```sql
   ALTER TABLE tbl_patient 
   ADD COLUMN deposit_balance DECIMAL(10,2) COMMENT 'Current deposit balance';
   ```

4. **Test migrations locally first**

5. **Never delete data in migrations** - Create separate cleanup scripts if needed

### For Seeds

1. **Use INSERT IGNORE**
   ```sql
   INSERT IGNORE INTO tbl_doctor (name, phone) VALUES ('Dr. Test', '09-123456');
   ```

2. **Use realistic data** - Helps with testing

3. **Keep seeds minimal** - Only essential test data

4. **Document seed data** - Add comments explaining purpose

## Troubleshooting

### Migration Already Executed
**Issue**: Migration shows as "already executed"

**Solution**: Check `tbl_migrations` table:
```sql
SELECT * FROM tbl_migrations ORDER BY executed_at DESC;
```

To re-run a migration:
```sql
DELETE FROM tbl_migrations WHERE migration_name = '001_add_doctor_photo.sql';
```

### Foreign Key Errors
**Issue**: Foreign key constraint fails

**Solution**: Ensure referenced tables exist first:
1. Check table existence
2. Run migrations in correct order
3. Verify column types match

### Duplicate Entry Errors in Seeds
**Issue**: Seed throws duplicate entry error

**Solution**: 
- Use `INSERT IGNORE` instead of `INSERT`
- Or use `ON DUPLICATE KEY UPDATE`

## Rollback Strategy

**Note**: This system doesn't auto-rollback. For rollbacks:

1. Create manual rollback scripts:
```sql
-- Rollback 001_add_doctor_photo.sql
ALTER TABLE tbl_doctor DROP COLUMN IF EXISTS photo;
ALTER TABLE tbl_doctor DROP COLUMN IF EXISTS email;
-- ... etc
```

2. Test rollback scripts in development first

3. Backup database before major migrations:
```bash
mysqldump -u root -p clinic_pro_db > backup_$(date +%Y%m%d).sql
```

## Maintenance

### Check Migration Status
```sql
SELECT * FROM tbl_migrations ORDER BY id;
```

### View All Tables
```sql
SHOW TABLES;
```

### Check Table Structure
```sql
DESCRIBE tbl_patient;
DESCRIBE tbl_doctor;
DESCRIBE tbl_inpatient;
DESCRIBE tbl_deposit_transaction;
```

## Package.json Scripts (Optional)

Add to `backend/package.json`:
```json
{
  "scripts": {
    "db:setup": "node ../database/setup-database.js",
    "db:migrate": "node ../database/migrations/run-all-migrations.js",
    "db:seed": "node ../database/seeds/run-all-seeds.js"
  }
}
```

Then run:
```bash
npm run db:setup     # Complete setup
npm run db:migrate   # Migrations only
npm run db:seed      # Seeds only
```

## Support

For issues or questions:
1. Check error messages in console output
2. Verify database connection in `.env`
3. Check MySQL error logs
4. Review migration/seed file SQL syntax

## Version History

| Version | Date       | Changes                           |
|---------|------------|-----------------------------------|
| 1.0.0   | 2024-10-16 | Initial migration system          |
|         |            | - Doctor photo fields             |
|         |            | - Deposit system                  |
|         |            | - Inpatient bed integration       |
|         |            | - Bed data fixes                  |

---

**Last Updated**: October 16, 2024  
**System Version**: 1.0.0
