# Clinic Pro - Database Management

Complete database migration and seeding system for Clinic Pro.

## 🚀 Quick Start

```bash
# Complete setup (migrations + seeds)
cd Clinic-Pro-Web/database
node setup-database.js

# Or run individually
node migrations/run-all-migrations.js  # Migrations only
node seeds/run-all-seeds.js            # Seeds only
```

## 📁 Structure

```asdasdasdasd            ffff
database/
├── migrations/          # Schema changes (DDL)
├── seeds/              # Test data (DML)
├── setup-database.js   # Master setup script
├── MIGRATION_GUIDE.md  # Detailed guide
├── CHANGELOG.md        # Change history
└── README.md          # This file
```

## 📦 What's Included

### Migrations (Schema Changes)
1. **Doctor Enhancements** - Photo upload & extended fields
2. **Deposit System** - Patient deposit tracking
3. **Inpatient-Bed Integration** - Bed assignment system
4. **Data Fixes** - Cleanup inconsistent bed data

### Seeds (Test Data)
1. **Sample Doctors** - 5 doctors with specializations
2. **Sample Patients** - 10 patients with deposits

## ✨ Features

- ✅ **Idempotent** - Safe to run multiple times
- ✅ **Tracked** - Knows which migrations ran
- ✅ **Sequential** - Runs in order
- ✅ **Documented** - Comprehensive guides
- ✅ **Error Handling** - Clear error messages

## 📚 Documentation

- **[Migration Guide](MIGRATION_GUIDE.md)** - Complete usage guide
- **[Changelog](CHANGELOG.md)** - All database changes
- **Migration Files** - Each file has inline comments

## 🔧 Requirements

- Node.js 14+
- MySQL 5.7+ or MariaDB 10.2+
- Existing Clinic Pro database

## 💡 Common Tasks

### Run Complete Setup
```bash
node setup-database.js
```

### Check What Migrations Ran
```sql
SELECT * FROM tbl_migrations ORDER BY executed_at;
```

### Re-run a Migration
```sql
-- Delete from tracking table
DELETE FROM tbl_migrations WHERE migration_name = '001_add_doctor_photo.sql';

-- Run migrations again
node migrations/run-all-migrations.js
```

### Backup Before Migrating
```bash
mysqldump -u root -p clinic_pro_db > backup_$(date +%Y%m%d).sql
```

## 🐛 Troubleshooting

### "Migration already executed"
- Check `tbl_migrations` table
- Delete entry to re-run

### "Foreign key constraint fails"
- Ensure referenced tables exist
- Run migrations in order

### "Connection refused"
- Check MySQL is running
- Verify `.env` configuration

## 📝 Creating New Migrations

1. Create file: `XXX_description.sql`
2. Use safe SQL:
```sql
ALTER TABLE tbl_name 
ADD COLUMN IF NOT EXISTS col_name VARCHAR(255);
```
3. Run: `node migrations/run-all-migrations.js`

## 🌱 Creating New Seeds

1. Create file: `XXX_description.sql`
2. Use INSERT IGNORE:
```sql
INSERT IGNORE INTO tbl_name (col1, col2) 
VALUES ('val1', 'val2');
```
3. Run: `node seeds/run-all-seeds.js`

## 🔐 Security Notes

- Migrations are tracked in database
- Seeds use INSERT IGNORE (safe for duplicates)
- Backup before major changes
- Test in development first

## 📊 Database Changes Summary

### Tables Modified
- `tbl_doctor` - Added 9 new fields
- `tbl_patient` - Added deposit_balance
- `tbl_inpatient` - Added 5 bed-related fields

### Tables Created
- `tbl_deposit_transaction` - Deposit history
- `tbl_migrations` - Migration tracking

### Indexes Added
- 8 new indexes for performance

## 🎯 Next Steps

After running migrations:

1. ✅ Test doctor photo upload
2. ✅ Test deposit transactions
3. ✅ Test inpatient bed assignment
4. ✅ Update frontend if needed
5. ✅ Test all features end-to-end

## 📞 Support

For issues:
1. Check error messages
2. Review [Migration Guide](MIGRATION_GUIDE.md)
3. Check [Changelog](CHANGELOG.md)
4. Verify database connection

## 🏆 Best Practices

- ✅ Always backup before migrations
- ✅ Test in development first
- ✅ Run complete setup on new installs
- ✅ Document custom migrations
- ✅ Use semantic versioning

## 📅 Version History

| Version | Date       | Changes                    |
|---------|------------|----------------------------|
| 1.0.0   | 2024-10-16 | Initial migration system   |

---

**Last Updated**: October 16, 2024  
**Maintained by**: Clinic Pro Team
