/**
 * Run Today's Database Updates
 * October 20, 2025
 * 
 * This script updates the database for:
 * - Sale Returns System
 * - Doctor Performance Reports
 * - POS Sales Doctor Tracking
 */

const { query } = require('../backend/config/database');
const fs = require('fs');
const path = require('path');

console.log('═══════════════════════════════════════════════════════════');
console.log('🔄 Database Update - October 20, 2025');
console.log('═══════════════════════════════════════════════════════════\n');

async function runUpdate() {
    try {
        console.log('📋 Updates to apply:');
        console.log('   1. ✅ Doctor tracking in vouchers');
        console.log('   2. ✅ Sale returns support');
        console.log('   3. ✅ Performance indexes');
        console.log('   4. ✅ Sample data for testing\n');
        
        // Step 1: Run migration 006
        console.log('🔄 Step 1/3: Running migration 006...');
        const migrationPath = path.join(__dirname, 'migrations', '006_update_for_today_features.sql');
        const migrationSql = fs.readFileSync(migrationPath, 'utf8');
        
        const migrationStatements = migrationSql
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0 && !s.startsWith('--') && !s.startsWith('SELECT'));
        
        let migrationCount = 0;
        for (const statement of migrationStatements) {
            if (statement.trim()) {
                try {
                    await query(statement);
                    migrationCount++;
                } catch (error) {
                    if (!['ER_DUP_FIELDNAME', 'ER_DUP_KEYNAME', 'ER_CANT_DROP_FIELD_OR_KEY'].includes(error.code)) {
                        console.warn(`   ⚠️  Warning:`, error.message);
                    }
                }
            }
        }
        console.log(`   ✅ Migration completed (${migrationCount} statements)\n`);
        
        // Step 2: Run seed 003
        console.log('🔄 Step 2/3: Running seed 003 (sample data)...');
        const seedPath = path.join(__dirname, 'seeds', '003_seed_doctor_performance_data.sql');
        const seedSql = fs.readFileSync(seedPath, 'utf8');
        
        const seedStatements = seedSql
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0 && !s.startsWith('--') && !s.startsWith('SELECT'));
        
        let seedCount = 0;
        for (const statement of seedStatements) {
            if (statement.trim()) {
                try {
                    await query(statement);
                    seedCount++;
                } catch (error) {
                    if (error.code !== 'ER_DUP_ENTRY') {
                        console.warn(`   ⚠️  Warning:`, error.message);
                    }
                }
            }
        }
        console.log(`   ✅ Seeding completed (${seedCount} statements)\n`);
        
        // Step 3: Verify updates
        console.log('🔄 Step 3/3: Verifying updates...');
        
        // Check if doctor columns exist
        const voucherDesc = await query('DESCRIBE tbl_voucher');
        const hasDoctorId = voucherDesc.some(col => col.Field === 'doctor_id');
        const hasDoctorName = voucherDesc.some(col => col.Field === 'doctor_name');
        
        if (hasDoctorId && hasDoctorName) {
            console.log('   ✅ Doctor fields added to tbl_voucher');
        } else {
            console.log('   ⚠️  Doctor fields may not be added properly');
        }
        
        // Check vouchers with doctors
        const voucherCount = await query('SELECT COUNT(*) as count FROM tbl_voucher WHERE doctor_id IS NOT NULL');
        console.log(`   ✅ ${voucherCount[0].count} vouchers have doctor assignments`);
        
        // Check returns
        const returnCount = await query('SELECT COUNT(*) as count FROM tbl_voucher_return');
        console.log(`   ✅ ${returnCount[0].count} return records in database`);
        
        // Check indexes
        const indexes = await query('SHOW INDEX FROM tbl_voucher WHERE Key_name LIKE "idx_%"');
        console.log(`   ✅ ${indexes.length} indexes on tbl_voucher`);
        
        console.log('\n═══════════════════════════════════════════════════════════');
        console.log('🎉 Database Update Completed Successfully!');
        console.log('═══════════════════════════════════════════════════════════\n');
        
        console.log('📊 Summary:');
        console.log(`   ✅ Migrations executed`);
        console.log(`   ✅ Sample data seeded`);
        console.log(`   ✅ Indexes created`);
        console.log(`   ✅ Verification passed`);
        
        console.log('\n🚀 Next Steps:');
        console.log('   1. Restart your backend server');
        console.log('   2. Refresh browser pages');
        console.log('   3. Test Doctor Performance Report');
        console.log('   4. Test Sale Returns');
        console.log('   5. Create POS sale with doctor');
        console.log('   6. Verify doctor appears in report\n');
        
        console.log('✅ Your database is now updated and ready!\n');
        
    } catch (error) {
        console.error('\n❌ Update failed:', error);
        console.error('Message:', error.message);
        console.error('\n💡 Troubleshooting:');
        console.error('   1. Check MySQL is running');
        console.error('   2. Verify database connection in .env');
        console.error('   3. Ensure user has ALTER/INDEX permissions');
        console.error('   4. Try running migrations manually\n');
        throw error;
    }
};

// Run the update
if (require.main === module) {
    runUpdate()
        .then(() => {
            console.log('✅ All done! Database is ready for new features.');
            process.exit(0);
        })
        .catch((error) => {
            console.error('❌ Update process failed');
            process.exit(1);
        });
}

module.exports = { runUpdate };

