-- ============================================
-- INPATIENT SALES MANAGEMENT TABLES
-- Daily usage tracking, deposits, and discharge settlements
-- ============================================

USE clinic_pro_db;

-- ============================================
-- Table: tbl_inpatient_usage
-- Tracks daily usage of products/services by inpatients
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_inpatient_usage` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `inpatient_id` INT NOT NULL,
    `product_id` INT NOT NULL,
    `quantity` DECIMAL(10,2) NOT NULL DEFAULT 1.00,
    `price` DECIMAL(10,2) NOT NULL,
    `notes` TEXT NULL,
    `usage_date` DATE NOT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `created_by` INT NULL,
    PRIMARY KEY (`id`),
    INDEX `idx_inpatient_id` (`inpatient_id`),
    INDEX `idx_product_id` (`product_id`),
    INDEX `idx_usage_date` (`usage_date`),
    INDEX `idx_created_at` (`created_at`),
    FOREIGN KEY (`inpatient_id`) REFERENCES `tbl_inpatient`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`product_id`) REFERENCES `tbl_stock`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`created_by`) REFERENCES `tbl_user`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_inpatient_deposits
-- Tracks deposits collected from inpatients
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_inpatient_deposits` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `inpatient_id` INT NOT NULL,
    `amount` DECIMAL(10,2) NOT NULL,
    `payment_method` ENUM('cash', 'card', 'bank_transfer', 'check', 'other') NOT NULL DEFAULT 'cash',
    `notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `created_by` INT NULL,
    PRIMARY KEY (`id`),
    INDEX `idx_inpatient_id` (`inpatient_id`),
    INDEX `idx_created_at` (`created_at`),
    FOREIGN KEY (`inpatient_id`) REFERENCES `tbl_inpatient`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`created_by`) REFERENCES `tbl_user`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_inpatient_discharge
-- Tracks discharge settlements and final bills
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_inpatient_discharge` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `inpatient_id` INT NOT NULL,
    `discharge_date` DATE NOT NULL,
    `total_usage` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `bed_charges` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `total_charges` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `total_deposits` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `outstanding_balance` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `final_payment` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    `payment_method` ENUM('cash', 'card', 'bank_transfer', 'check', 'other') NULL,
    `discharge_notes` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `created_by` INT NULL,
    PRIMARY KEY (`id`),
    UNIQUE KEY `unique_inpatient_discharge` (`inpatient_id`),
    INDEX `idx_discharge_date` (`discharge_date`),
    INDEX `idx_created_at` (`created_at`),
    FOREIGN KEY (`inpatient_id`) REFERENCES `tbl_inpatient`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`created_by`) REFERENCES `tbl_user`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Add discharge-related columns to tbl_inpatient
-- ============================================
ALTER TABLE `tbl_inpatient` 
ADD COLUMN IF NOT EXISTS `final_bill` DECIMAL(10,2) NULL DEFAULT NULL COMMENT 'Final bill amount on discharge',
ADD COLUMN IF NOT EXISTS `discharge_notes` TEXT NULL DEFAULT NULL COMMENT 'Notes on discharge',
ADD COLUMN IF NOT EXISTS `discharge_date` DATE NULL DEFAULT NULL COMMENT 'Date of discharge';

-- ============================================
-- Create indexes for better performance
-- ============================================
CREATE INDEX IF NOT EXISTS `idx_inpatient_end_date` ON `tbl_inpatient`(`end_date`);
CREATE INDEX IF NOT EXISTS `idx_inpatient_discharge_date` ON `tbl_inpatient`(`discharge_date`);

-- ============================================
-- Insert sample data (optional)
-- ============================================
-- Note: This is optional and should only be run in development
-- INSERT INTO `tbl_inpatient_usage` (`inpatient_id`, `product_id`, `quantity`, `price`, `usage_date`, `notes`) VALUES
-- (1, 1, 2.00, 10.50, CURDATE(), 'Morning medication'),
-- (1, 2, 1.00, 25.00, CURDATE(), 'Blood test');

-- INSERT INTO `tbl_inpatient_deposits` (`inpatient_id`, `amount`, `payment_method`, `notes`) VALUES
-- (1, 100.00, 'cash', 'Initial deposit'),
-- (1, 50.00, 'card', 'Additional deposit');

-- ============================================
-- Create views for easier reporting
-- ============================================

-- View: Inpatient daily usage summary
CREATE OR REPLACE VIEW `vw_inpatient_daily_usage` AS
SELECT 
    u.inpatient_id,
    i.patient_id,
    i.name as patient_name,
    u.usage_date,
    COUNT(u.id) as usage_count,
    SUM(u.quantity * u.price) as daily_total
FROM tbl_inpatient_usage u
JOIN tbl_inpatient i ON u.inpatient_id = i.id
GROUP BY u.inpatient_id, u.usage_date
ORDER BY u.usage_date DESC;

-- View: Inpatient outstanding balances
CREATE OR REPLACE VIEW `vw_inpatient_outstanding` AS
SELECT 
    i.id as inpatient_id,
    i.patient_id,
    i.name as patient_name,
    i.bed_number,
    i.room_no,
    COALESCE(usage_totals.total_usage, 0) as total_usage,
    COALESCE(bed_charges.bed_charges, 0) as bed_charges,
    COALESCE(usage_totals.total_usage, 0) + COALESCE(bed_charges.bed_charges, 0) as total_charges,
    COALESCE(deposit_totals.total_deposits, 0) as total_deposits,
    (COALESCE(usage_totals.total_usage, 0) + COALESCE(bed_charges.bed_charges, 0)) - COALESCE(deposit_totals.total_deposits, 0) as outstanding_balance
FROM tbl_inpatient i
LEFT JOIN (
    SELECT 
        inpatient_id,
        SUM(quantity * price) as total_usage
    FROM tbl_inpatient_usage
    GROUP BY inpatient_id
) usage_totals ON i.id = usage_totals.inpatient_id
LEFT JOIN (
    SELECT 
        id as inpatient_id,
        daily_rate * GREATEST(1, DATEDIFF(CURDATE(), start_date)) as bed_charges
    FROM tbl_inpatient
    WHERE end_date IS NULL
) bed_charges ON i.id = bed_charges.inpatient_id
LEFT JOIN (
    SELECT 
        inpatient_id,
        SUM(amount) as total_deposits
    FROM tbl_inpatient_deposits
    GROUP BY inpatient_id
) deposit_totals ON i.id = deposit_totals.inpatient_id
WHERE i.end_date IS NULL
ORDER BY outstanding_balance DESC;

-- ============================================
-- Create stored procedures for common operations
-- ============================================

DELIMITER //

-- Procedure: Calculate inpatient outstanding balance
CREATE PROCEDURE IF NOT EXISTS `sp_calculate_inpatient_balance`(
    IN p_inpatient_id INT,
    OUT p_total_usage DECIMAL(10,2),
    OUT p_bed_charges DECIMAL(10,2),
    OUT p_total_charges DECIMAL(10,2),
    OUT p_total_deposits DECIMAL(10,2),
    OUT p_outstanding_balance DECIMAL(10,2)
)
BEGIN
    DECLARE v_daily_rate DECIMAL(10,2) DEFAULT 0;
    DECLARE v_days_stayed INT DEFAULT 0;
    
    -- Get total usage
    SELECT COALESCE(SUM(quantity * price), 0) INTO p_total_usage
    FROM tbl_inpatient_usage
    WHERE inpatient_id = p_inpatient_id;
    
    -- Get bed charges
    SELECT daily_rate, DATEDIFF(CURDATE(), start_date) INTO v_daily_rate, v_days_stayed
    FROM tbl_inpatient
    WHERE id = p_inpatient_id AND end_date IS NULL;
    
    SET p_bed_charges = v_daily_rate * GREATEST(1, v_days_stayed);
    SET p_total_charges = p_total_usage + p_bed_charges;
    
    -- Get total deposits
    SELECT COALESCE(SUM(amount), 0) INTO p_total_deposits
    FROM tbl_inpatient_deposits
    WHERE inpatient_id = p_inpatient_id;
    
    -- Calculate outstanding balance
    SET p_outstanding_balance = p_total_charges - p_total_deposits;
END //

-- Procedure: Generate discharge summary
CREATE PROCEDURE IF NOT EXISTS `sp_generate_discharge_summary`(
    IN p_inpatient_id INT,
    OUT p_discharge_summary JSON
)
BEGIN
    DECLARE v_total_usage DECIMAL(10,2);
    DECLARE v_bed_charges DECIMAL(10,2);
    DECLARE v_total_charges DECIMAL(10,2);
    DECLARE v_total_deposits DECIMAL(10,2);
    DECLARE v_outstanding_balance DECIMAL(10,2);
    
    -- Calculate balances
    CALL sp_calculate_inpatient_balance(
        p_inpatient_id,
        v_total_usage,
        v_bed_charges,
        v_total_charges,
        v_total_deposits,
        v_outstanding_balance
    );
    
    -- Create JSON summary
    SET p_discharge_summary = JSON_OBJECT(
        'inpatient_id', p_inpatient_id,
        'total_usage', v_total_usage,
        'bed_charges', v_bed_charges,
        'total_charges', v_total_charges,
        'total_deposits', v_total_deposits,
        'outstanding_balance', v_outstanding_balance,
        'generated_at', NOW()
    );
END //

DELIMITER ;

-- ============================================
-- Create triggers for data integrity
-- ============================================

-- Trigger: Update inpatient deposit when new deposit is added
DELIMITER //
CREATE TRIGGER IF NOT EXISTS `tr_update_inpatient_deposit_after_insert`
AFTER INSERT ON `tbl_inpatient_deposits`
FOR EACH ROW
BEGIN
    UPDATE tbl_inpatient 
    SET deposit = COALESCE(deposit, 0) + NEW.amount
    WHERE id = NEW.inpatient_id;
END //

-- Trigger: Update inpatient deposit when deposit is deleted
CREATE TRIGGER IF NOT EXISTS `tr_update_inpatient_deposit_after_delete`
AFTER DELETE ON `tbl_inpatient_deposits`
FOR EACH ROW
BEGIN
    UPDATE tbl_inpatient 
    SET deposit = GREATEST(0, COALESCE(deposit, 0) - OLD.amount)
    WHERE id = OLD.inpatient_id;
END //
DELIMITER ;

-- ============================================
-- Grant permissions (adjust as needed)
-- ============================================
-- GRANT SELECT, INSERT, UPDATE, DELETE ON clinic_pro_db.tbl_inpatient_usage TO 'clinic_user'@'localhost';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON clinic_pro_db.tbl_inpatient_deposits TO 'clinic_user'@'localhost';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON clinic_pro_db.tbl_inpatient_discharge TO 'clinic_user'@'localhost';
-- GRANT EXECUTE ON PROCEDURE clinic_pro_db.sp_calculate_inpatient_balance TO 'clinic_user'@'localhost';
-- GRANT EXECUTE ON PROCEDURE clinic_pro_db.sp_generate_discharge_summary TO 'clinic_user'@'localhost';
