-- ============================================
-- REFERRING DOCTOR SYSTEM - Complete Implementation
-- Migration 005: Create referring doctor and commission tracking
-- ============================================

USE clinic_pro_db;

-- ============================================
-- Table: tbl_referring_doctor
-- Purpose: Store referring doctors (internal and external)
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_referring_doctor` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `name` VARCHAR(100) NOT NULL,
    `specialty` VARCHAR(100) NULL COMMENT 'Medical specialty',
    `qualification` VARCHAR(200) NULL COMMENT 'Degrees and certifications',
    `clinic_name` VARCHAR(200) NULL COMMENT 'Clinic or hospital name',
    `phone` VARCHAR(50) NULL,
    `email` VARCHAR(100) NULL,
    `address` TEXT NULL,
    `city` VARCHAR(100) NULL,
    `state` VARCHAR(100) NULL,
    
    -- Commission Settings
    `commission_type` ENUM('percentage', 'fixed', 'none') DEFAULT 'none' COMMENT 'Type of commission',
    `commission_value` DECIMAL(10,2) DEFAULT 0.00 COMMENT 'Commission percentage or fixed amount',
    `commission_applies_to` ENUM('all', 'products_only', 'services_only', 'lab_only') DEFAULT 'all',
    
    -- Relationship
    `is_internal` TINYINT(1) DEFAULT 0 COMMENT '1 = Internal doctor from tbl_doctor, 0 = External',
    `internal_doctor_id` INT NULL COMMENT 'Link to tbl_doctor.id if internal',
    
    -- Additional Info
    `registration_number` VARCHAR(100) NULL COMMENT 'Medical council registration',
    `notes` TEXT NULL,
    `photo_path` VARCHAR(255) NULL,
    
    -- Status and Tracking
    `active` TINYINT(1) DEFAULT 1 COMMENT '1 = Active, 0 = Inactive',
    `total_referrals` INT DEFAULT 0 COMMENT 'Total number of referrals',
    `total_revenue` DECIMAL(12,2) DEFAULT 0.00 COMMENT 'Total revenue from referrals',
    `total_commission_earned` DECIMAL(12,2) DEFAULT 0.00 COMMENT 'Total commission earned',
    `total_commission_paid` DECIMAL(12,2) DEFAULT 0.00 COMMENT 'Total commission paid',
    
    -- Audit
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `created_by` INT NULL,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `updated_by` INT NULL,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_email` (`email`),
    INDEX `idx_name` (`name`),
    INDEX `idx_active` (`active`),
    INDEX `idx_is_internal` (`is_internal`),
    INDEX `idx_internal_doctor` (`internal_doctor_id`),
    INDEX `idx_commission_type` (`commission_type`),
    FOREIGN KEY (`internal_doctor_id`) REFERENCES `tbl_doctor`(`id`) ON DELETE SET NULL,
    FOREIGN KEY (`created_by`) REFERENCES `tbl_user`(`id`) ON DELETE SET NULL,
    FOREIGN KEY (`updated_by`) REFERENCES `tbl_user`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: tbl_referral_commission
-- Purpose: Track commissions for each referral
-- ============================================
CREATE TABLE IF NOT EXISTS `tbl_referral_commission` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `referring_doctor_id` INT NOT NULL,
    
    -- Transaction Link
    `transaction_type` ENUM('sale', 'lab_order', 'service', 'appointment') NOT NULL,
    `sale_id` INT NULL COMMENT 'Link to tbl_sale if POS transaction',
    `lab_order_id` INT NULL COMMENT 'Link to tbl_lab_order if lab test',
    `voucher_no` VARCHAR(50) NULL COMMENT 'Reference voucher number',
    
    -- Patient Info
    `patient_id` VARCHAR(250) NULL,
    `patient_name` VARCHAR(100) NULL,
    
    -- Transaction Details
    `transaction_date` DATE NOT NULL,
    `transaction_amount` DECIMAL(10,2) NOT NULL COMMENT 'Total sale/service amount',
    
    -- Commission Calculation
    `commission_type` ENUM('percentage', 'fixed', 'none') NOT NULL,
    `commission_rate` DECIMAL(5,2) NULL COMMENT 'Percentage rate if applicable',
    `commission_amount` DECIMAL(10,2) NOT NULL DEFAULT 0.00 COMMENT 'Calculated commission',
    
    -- Payment Tracking
    `payment_status` ENUM('pending', 'approved', 'paid', 'cancelled') DEFAULT 'pending',
    `payment_date` DATE NULL,
    `payment_method` VARCHAR(50) NULL,
    `payment_reference` VARCHAR(100) NULL,
    `payment_notes` TEXT NULL,
    
    -- Additional Info
    `notes` TEXT NULL,
    
    -- Audit
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `created_by` INT NULL,
    `approved_at` TIMESTAMP NULL,
    `approved_by` INT NULL,
    `paid_at` TIMESTAMP NULL,
    `paid_by` INT NULL,
    
    PRIMARY KEY (`id`),
    INDEX `idx_referring_doctor` (`referring_doctor_id`),
    INDEX `idx_transaction_type` (`transaction_type`),
    INDEX `idx_sale` (`sale_id`),
    INDEX `idx_lab_order` (`lab_order_id`),
    INDEX `idx_voucher` (`voucher_no`),
    INDEX `idx_transaction_date` (`transaction_date`),
    INDEX `idx_payment_status` (`payment_status`),
    INDEX `idx_patient` (`patient_id`),
    FOREIGN KEY (`referring_doctor_id`) REFERENCES `tbl_referring_doctor`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`created_by`) REFERENCES `tbl_user`(`id`) ON DELETE SET NULL,
    FOREIGN KEY (`approved_by`) REFERENCES `tbl_user`(`id`) ON DELETE SET NULL,
    FOREIGN KEY (`paid_by`) REFERENCES `tbl_user`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Update Existing Tables: Add referring_doctor_id
-- ============================================

-- Add to tbl_sale (POS transactions)
-- Note: Migration script will handle duplicate column/key/table errors gracefully
ALTER TABLE `tbl_sale` 
ADD COLUMN `referring_doctor_id` INT NULL COMMENT 'ID from tbl_referring_doctor';

ALTER TABLE `tbl_sale` 
ADD COLUMN `referring_doctor_name` VARCHAR(100) NULL COMMENT 'Cached name for performance';

ALTER TABLE `tbl_sale` 
ADD INDEX `idx_referring_doctor` (`referring_doctor_id`);

-- Add to tbl_lab_order (Laboratory orders)
-- Note: Will be skipped if table doesn't exist (migration script handles this)
ALTER TABLE `tbl_lab_order` 
ADD COLUMN `referring_doctor_id` INT NULL COMMENT 'ID from tbl_referring_doctor';

ALTER TABLE `tbl_lab_order` 
ADD COLUMN `referring_doctor_name` VARCHAR(100) NULL COMMENT 'Cached name for performance';

ALTER TABLE `tbl_lab_order` 
ADD INDEX `idx_referring_doctor` (`referring_doctor_id`);

-- ============================================
-- Create Views for Easy Reporting
-- ============================================

-- View: Commission Summary by Referring Doctor
CREATE OR REPLACE VIEW `vw_commission_summary` AS
SELECT 
    rd.id,
    rd.name as doctor_name,
    rd.specialty,
    rd.clinic_name,
    rd.commission_type,
    rd.commission_value,
    rd.total_referrals,
    rd.total_revenue,
    rd.total_commission_earned,
    rd.total_commission_paid,
    (rd.total_commission_earned - rd.total_commission_paid) as commission_pending,
    COUNT(rc.id) as total_transactions,
    SUM(CASE WHEN rc.payment_status = 'pending' THEN rc.commission_amount ELSE 0 END) as pending_commission,
    SUM(CASE WHEN rc.payment_status = 'approved' THEN rc.commission_amount ELSE 0 END) as approved_commission,
    SUM(CASE WHEN rc.payment_status = 'paid' THEN rc.commission_amount ELSE 0 END) as paid_commission
FROM tbl_referring_doctor rd
LEFT JOIN tbl_referral_commission rc ON rd.id = rc.referring_doctor_id
WHERE rd.active = 1
GROUP BY rd.id
ORDER BY rd.total_revenue DESC;

-- View: Recent Referrals with Commission
CREATE OR REPLACE VIEW `vw_recent_referrals` AS
SELECT 
    rc.id as commission_id,
    rc.transaction_date,
    rc.transaction_type,
    rc.voucher_no,
    rc.patient_name,
    rd.name as referring_doctor,
    rd.specialty,
    rc.transaction_amount,
    rc.commission_type,
    rc.commission_rate,
    rc.commission_amount,
    rc.payment_status,
    rc.payment_date
FROM tbl_referral_commission rc
JOIN tbl_referring_doctor rd ON rc.referring_doctor_id = rd.id
ORDER BY rc.transaction_date DESC, rc.created_at DESC
LIMIT 100;

-- View: Monthly Commission Summary
CREATE OR REPLACE VIEW `vw_monthly_commission` AS
SELECT 
    DATE_FORMAT(rc.transaction_date, '%Y-%m') as month,
    rd.id as referring_doctor_id,
    rd.name as doctor_name,
    COUNT(rc.id) as total_referrals,
    SUM(rc.transaction_amount) as total_revenue,
    SUM(rc.commission_amount) as total_commission,
    SUM(CASE WHEN rc.payment_status = 'paid' THEN rc.commission_amount ELSE 0 END) as paid_commission,
    SUM(CASE WHEN rc.payment_status IN ('pending', 'approved') THEN rc.commission_amount ELSE 0 END) as pending_commission
FROM tbl_referral_commission rc
JOIN tbl_referring_doctor rd ON rc.referring_doctor_id = rd.id
GROUP BY DATE_FORMAT(rc.transaction_date, '%Y-%m'), rd.id
ORDER BY month DESC, total_commission DESC;

-- ============================================
-- Insert Sample Data (Optional - for testing)
-- ============================================

-- Sample internal referring doctor
INSERT INTO `tbl_referring_doctor` 
(`name`, `specialty`, `clinic_name`, `phone`, `email`, `commission_type`, `commission_value`, `is_internal`, `active`, `notes`) 
VALUES 
('Dr. John Referrer', 'General Medicine', 'City Medical Center', '+1-234-567-8900', 'john.referrer@medical.com', 'percentage', 10.00, 0, 1, 'External referring doctor - 10% commission on all sales'),
('Dr. Sarah External', 'Pediatrics', 'Children Hospital', '+1-234-567-8901', 'sarah.external@hospital.com', 'fixed', 15.00, 0, 1, 'Fixed commission of $15 per referral'),
('Walk-in / No Referral', 'N/A', 'N/A', NULL, NULL, 'none', 0.00, 0, 1, 'Default option for walk-in patients with no referring doctor');

-- ============================================
-- Stored Procedures
-- Note: Stored procedures with DELIMITER are commented out
-- They can be created manually or via a separate migration script
-- ============================================

-- Stored procedures are temporarily commented out due to DELIMITER handling issues
-- in the migration script. They can be created manually if needed.
--
-- To create the procedures manually, run the SQL commands from a MySQL client
-- that supports DELIMITER (like mysql command line or MySQL Workbench)
--
-- See: 005_referring_doctor_procedures.sql (if created separately)

-- ============================================
-- Success Message
-- ============================================
SELECT 'Migration 005: Referring Doctor System created successfully!' as status,
       'Tables: tbl_referring_doctor, tbl_referral_commission' as tables_created,
       'Views: vw_commission_summary, vw_recent_referrals, vw_monthly_commission' as views_created,
       'Procedures: (commented out - create manually if needed)' as procedures_created;

