-- ============================================
-- Migration: Updates for Sale Returns & Doctor Reports
-- Date: October 20, 2025
-- Purpose: Ensure all tables support new features implemented today
-- ============================================

-- ============================================
-- PART 1: Sale Returns Enhancement
-- ============================================

-- Ensure tbl_voucher_return has all needed fields
ALTER TABLE tbl_voucher_return 
ADD COLUMN IF NOT EXISTS original_voucher_no INT NULL COMMENT 'Reference to original sale voucher';

-- Add index for faster return lookups
-- Note: Remove IF NOT EXISTS for MySQL compatibility
CREATE INDEX idx_return_date ON tbl_voucher_return(Date);
CREATE INDEX idx_return_user ON tbl_voucher_return(user_name);

-- Ensure tbl_sale_return_item has proper structure
ALTER TABLE tbl_sale_return_item
ADD COLUMN IF NOT EXISTS user_name VARCHAR(50) NULL COMMENT 'Who processed the return';

CREATE INDEX idx_return_item_vno ON tbl_sale_return_item(vno);
CREATE INDEX idx_return_item_barcode ON tbl_sale_return_item(barcode);
CREATE INDEX idx_return_item_date ON tbl_sale_return_item(date);

-- ============================================
-- PART 2: Doctor Performance Report Support
-- ============================================

-- Ensure all voucher tables have doctor tracking fields
-- (Covers tbl_voucher)
ALTER TABLE tbl_voucher 
ADD COLUMN IF NOT EXISTS doctor_id INT NULL COMMENT 'Doctor who prescribed/referred',
ADD COLUMN IF NOT EXISTS doctor_name VARCHAR(100) NULL COMMENT 'Doctor name for quick reference',
ADD COLUMN IF NOT EXISTS patient_id VARCHAR(50) NULL COMMENT 'Patient ID for tracking';

-- (Covers tbl_voucher_return)
ALTER TABLE tbl_voucher_return 
ADD COLUMN IF NOT EXISTS doctor_id INT NULL,
ADD COLUMN IF NOT EXISTS doctor_name VARCHAR(100) NULL,
ADD COLUMN IF NOT EXISTS patient_id VARCHAR(50) NULL;

-- (Covers tbl_voucher_archive)
ALTER TABLE tbl_voucher_archive 
ADD COLUMN IF NOT EXISTS doctor_id INT NULL,
ADD COLUMN IF NOT EXISTS doctor_name VARCHAR(100) NULL,
ADD COLUMN IF NOT EXISTS patient_id VARCHAR(50) NULL;

-- Add performance-critical indexes
-- Note: Remove IF NOT EXISTS for MySQL compatibility
CREATE INDEX idx_voucher_doctor_id ON tbl_voucher(doctor_id);
CREATE INDEX idx_voucher_patient_id ON tbl_voucher(patient_id);
CREATE INDEX idx_voucher_doctor_name ON tbl_voucher(doctor_name);
CREATE INDEX idx_voucher_date ON tbl_voucher(Date);

-- Treatment table indexes for better join performance
CREATE INDEX idx_treatment_doctor ON tbl_treatment(doctor_id);
CREATE INDEX idx_treatment_patient ON tbl_treatment(patient_id);
CREATE INDEX idx_treatment_voucher ON tbl_treatment(Voucher_id);
CREATE INDEX idx_treatment_date ON tbl_treatment(Date);

-- ============================================
-- PART 3: Performance Optimization
-- ============================================

-- Sale table indexes for report generation
CREATE INDEX idx_sale_vno ON tbl_sale(VNo);
CREATE INDEX idx_sale_date ON tbl_sale(Date);
CREATE INDEX idx_sale_barcode ON tbl_sale(Barcode);

-- Stock table for return processing
CREATE INDEX idx_stock_barcode ON tbl_stock(barcode);

-- Alert table for low stock management
CREATE INDEX idx_alert_barcode ON tbl_alert(barcode);

-- ============================================
-- PART 4: Data Integrity
-- ============================================

-- Ensure Date columns are properly formatted
-- (MySQL-safe updates, won't fail if already correct)

-- Update any NULL dates in voucher_return to current timestamp
UPDATE tbl_voucher_return 
SET Date = NOW() 
WHERE Date IS NULL;

-- Update any NULL status to default (1 = completed)
UPDATE tbl_voucher_return 
SET status = 1 
WHERE status IS NULL;

-- ============================================
-- Verification Queries (for logging)
-- ============================================

-- Count vouchers with doctors
SELECT COUNT(*) as vouchers_with_doctors 
FROM tbl_voucher 
WHERE doctor_id IS NOT NULL;

-- Count returns
SELECT COUNT(*) as total_returns 
FROM tbl_voucher_return;

-- ============================================
-- Migration Complete
-- ============================================
-- This migration ensures:
-- 1. ✅ Sale returns system fully supported
-- 2. ✅ Doctor performance tracking enabled
-- 3. ✅ POS sales with doctors show in reports
-- 4. ✅ Proper indexes for performance
-- 5. ✅ Data integrity maintained
-- ============================================

