-- Migration: Create Lab Orders from Sales
-- Description: Creates lab order system that links to sales when laboratory items are sold
-- Date: 2024
-- Author: System

-- Create tbl_lab_order table (if not exists)
CREATE TABLE IF NOT EXISTS tbl_lab_order (
    id INT NOT NULL AUTO_INCREMENT,
    order_number VARCHAR(50) NOT NULL UNIQUE,
    sale_id INT NULL COMMENT 'Reference to tbl_sale when lab item was sold',
    sale_vno INT NULL COMMENT 'Voucher number from sale',
    lab_item_id INT NULL COMMENT 'Reference to tbl_laboratory_item',
    lab_item_name VARCHAR(255) NOT NULL COMMENT 'Lab item name at time of sale',
    patient_id VARCHAR(50) NOT NULL COMMENT 'Patient ID from sale',
    patient_name VARCHAR(255) NULL COMMENT 'Patient name',
    doctor_id INT NULL COMMENT 'Doctor ID from sale',
    doctor_name VARCHAR(255) NULL COMMENT 'Doctor name',
    referring_doctor_id INT NULL COMMENT 'Referring doctor ID',
    referring_doctor_name VARCHAR(255) NULL COMMENT 'Referring doctor name',
    order_date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'When lab item was sold/ordered',
    priority ENUM('Routine', 'Urgent', 'STAT', 'Emergency') DEFAULT 'Routine',
    clinical_info TEXT NULL COMMENT 'Clinical indication/reason for test',
    status ENUM('Ordered', 'Specimen_Collected', 'In_Progress', 'Completed', 'Cancelled') DEFAULT 'Ordered',
    collection_date DATETIME NULL,
    collected_by INT NULL COMMENT 'User ID of person who collected specimen',
    notes TEXT NULL,
    total_amount DECIMAL(10,2) DEFAULT 0.00,
    created_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    INDEX idx_order_number (order_number),
    INDEX idx_sale_id (sale_id),
    INDEX idx_sale_vno (sale_vno),
    INDEX idx_lab_item_id (lab_item_id),
    INDEX idx_patient_id (patient_id),
    INDEX idx_doctor_id (doctor_id),
    INDEX idx_status (status),
    INDEX idx_order_date (order_date),
    FOREIGN KEY (lab_item_id) REFERENCES tbl_laboratory_item(id) ON DELETE SET NULL,
    FOREIGN KEY (doctor_id) REFERENCES tbl_doctor(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Add doctor signature field to tbl_lab_report (if not exists)
-- Note: Migration script will handle duplicate column/key errors gracefully
ALTER TABLE `tbl_lab_report` 
ADD COLUMN `doctor_signature` VARCHAR(255) NULL COMMENT 'Path to doctor digital signature image';

-- Add order_id field to tbl_lab_report (if not exists)
ALTER TABLE `tbl_lab_report` 
ADD COLUMN `order_id` INT NULL COMMENT 'Reference to tbl_lab_order';

-- Add index for order_id (if not exists)
ALTER TABLE `tbl_lab_report` 
ADD INDEX `idx_order_id` (`order_id`);

-- Add reference range fields to tbl_lab_report_item (if not exists)
ALTER TABLE `tbl_lab_report_item` 
ADD COLUMN `reference_min` DECIMAL(10,2) NULL COMMENT 'Minimum reference value';

ALTER TABLE `tbl_lab_report_item` 
ADD COLUMN `reference_max` DECIMAL(10,2) NULL COMMENT 'Maximum reference value';

-- Add comment to table
ALTER TABLE tbl_lab_order 
COMMENT = 'Lab orders created automatically when laboratory items are sold in POS/IPD Sales';

