/**
 * Migration Runner
 * Runs all migration files in order
 */

const { query } = require('../../backend/config/database');
const fs = require('fs').promises;
const path = require('path');

// Create migrations tracking table
const createMigrationsTable = async () => {
    await query(`
        CREATE TABLE IF NOT EXISTS tbl_migrations (
            id INT NOT NULL AUTO_INCREMENT,
            migration_name VARCHAR(255) NOT NULL UNIQUE,
            executed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);
};

// Check if migration has been run
const isMigrationExecuted = async (migrationName) => {
    const result = await query(
        'SELECT id FROM tbl_migrations WHERE migration_name = ?',
        [migrationName]
    );
    return result.length > 0;
};

// Mark migration as executed
const markMigrationExecuted = async (migrationName) => {
    await query(
        'INSERT INTO tbl_migrations (migration_name) VALUES (?)',
        [migrationName]
    );
};

// Execute SQL file
const executeSqlFile = async (filePath) => {
    const sql = await fs.readFile(filePath, 'utf8');
    
    // Split by semicolons and filter out empty statements
    const statements = sql
        .split(';')
        .map(s => s.trim())
        .filter(s => s.length > 0 && !s.startsWith('--'));
    
    for (const statement of statements) {
        if (statement.trim()) {
            try {
                await query(statement);
            } catch (error) {
                // Ignore duplicate column/key errors
                if (!['ER_DUP_FIELDNAME', 'ER_DUP_KEYNAME', 'ER_CANT_DROP_FIELD_OR_KEY'].includes(error.code)) {
                    throw error;
                }
            }
        }
    }
};

// Run all migrations
const runMigrations = async () => {
    try {
        console.log('🚀 Starting database migrations...\n');
        
        // Create migrations tracking table
        await createMigrationsTable();
        console.log('✅ Migrations tracking table ready\n');
        
        // Get all migration files
        const migrationsDir = __dirname;
        const files = await fs.readdir(migrationsDir);
        const migrationFiles = files
            .filter(f => f.endsWith('.sql'))
            .sort(); // Sort to ensure order
        
        console.log(`📋 Found ${migrationFiles.length} migration files\n`);
        
        let executedCount = 0;
        let skippedCount = 0;
        
        // Execute each migration
        for (const file of migrationFiles) {
            const migrationName = file;
            const filePath = path.join(migrationsDir, file);
            
            // Check if already executed
            if (await isMigrationExecuted(migrationName)) {
                console.log(`⏭️  Skipping ${migrationName} (already executed)`);
                skippedCount++;
                continue;
            }
            
            console.log(`🔄 Running ${migrationName}...`);
            
            try {
                await executeSqlFile(filePath);
                await markMigrationExecuted(migrationName);
                console.log(`✅ Completed ${migrationName}\n`);
                executedCount++;
            } catch (error) {
                console.error(`❌ Error in ${migrationName}:`, error.message);
                throw error;
            }
        }
        
        console.log('\n📊 Migration Summary:');
        console.log(`   ✅ Executed: ${executedCount}`);
        console.log(`   ⏭️  Skipped: ${skippedCount}`);
        console.log(`   📁 Total: ${migrationFiles.length}`);
        console.log('\n🎉 All migrations completed successfully!');
        
    } catch (error) {
        console.error('\n❌ Migration failed:', error);
        throw error;
    }
};

// Run migrations if called directly
if (require.main === module) {
    runMigrations()
        .then(() => {
            console.log('\n✅ Migration process completed');
            process.exit(0);
        })
        .catch((error) => {
            console.error('\n❌ Migration process failed:', error);
            process.exit(1);
        });
}

module.exports = { runMigrations };
