-- ============================================
-- Warehouse Management System
-- ============================================

-- Table: tbl_warehouse
-- Stores warehouse/location information
CREATE TABLE IF NOT EXISTS `tbl_warehouse` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `warehouse_code` VARCHAR(50) NOT NULL UNIQUE,
    `name` VARCHAR(255) NOT NULL,
    `description` TEXT NULL,
    `address` VARCHAR(500) NULL,
    `phone` VARCHAR(50) NULL,
    `email` VARCHAR(255) NULL,
    `manager_name` VARCHAR(255) NULL,
    `manager_phone` VARCHAR(50) NULL,
    `is_active` TINYINT(1) DEFAULT 1,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `created_by` INT NULL,
    PRIMARY KEY (`id`),
    INDEX `idx_warehouse_code` (`warehouse_code`),
    INDEX `idx_is_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table: tbl_warehouse_stock
-- Stores stock quantities per warehouse
CREATE TABLE IF NOT EXISTS `tbl_warehouse_stock` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `warehouse_id` INT NOT NULL,
    `stock_id` INT NOT NULL,
    `quantity` INT NOT NULL DEFAULT 0,
    `reserved_quantity` INT NOT NULL DEFAULT 0 COMMENT 'Quantity reserved for transfers/orders',
    `min_stock_level` INT DEFAULT 0,
    `max_stock_level` INT DEFAULT 0,
    `last_updated` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `updated_by` INT NULL,
    PRIMARY KEY (`id`),
    UNIQUE KEY `unique_warehouse_stock` (`warehouse_id`, `stock_id`),
    INDEX `idx_warehouse_id` (`warehouse_id`),
    INDEX `idx_stock_id` (`stock_id`),
    INDEX `idx_quantity` (`quantity`),
    FOREIGN KEY (`warehouse_id`) REFERENCES `tbl_warehouse`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`stock_id`) REFERENCES `tbl_stock`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table: tbl_stock_transfer
-- Records stock transfers between warehouses
CREATE TABLE IF NOT EXISTS `tbl_stock_transfer` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `transfer_number` VARCHAR(50) NOT NULL UNIQUE,
    `from_warehouse_id` INT NOT NULL,
    `to_warehouse_id` INT NOT NULL,
    `transfer_date` DATE NOT NULL,
    `status` ENUM('Pending', 'In Transit', 'Completed', 'Cancelled') DEFAULT 'Pending',
    `notes` TEXT NULL,
    `created_by` INT NULL,
    `approved_by` INT NULL,
    `received_by` INT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `completed_at` TIMESTAMP NULL,
    PRIMARY KEY (`id`),
    INDEX `idx_transfer_number` (`transfer_number`),
    INDEX `idx_from_warehouse` (`from_warehouse_id`),
    INDEX `idx_to_warehouse` (`to_warehouse_id`),
    INDEX `idx_status` (`status`),
    INDEX `idx_transfer_date` (`transfer_date`),
    FOREIGN KEY (`from_warehouse_id`) REFERENCES `tbl_warehouse`(`id`) ON DELETE RESTRICT,
    FOREIGN KEY (`to_warehouse_id`) REFERENCES `tbl_warehouse`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table: tbl_stock_transfer_item
-- Items in a stock transfer
CREATE TABLE IF NOT EXISTS `tbl_stock_transfer_item` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `transfer_id` INT NOT NULL,
    `stock_id` INT NOT NULL,
    `quantity` INT NOT NULL,
    `received_quantity` INT DEFAULT 0,
    `notes` TEXT NULL,
    PRIMARY KEY (`id`),
    INDEX `idx_transfer_id` (`transfer_id`),
    INDEX `idx_stock_id` (`stock_id`),
    FOREIGN KEY (`transfer_id`) REFERENCES `tbl_stock_transfer`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`stock_id`) REFERENCES `tbl_stock`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table: tbl_stock_movement
-- Tracks all stock movements (in/out) with assigned person
CREATE TABLE IF NOT EXISTS `tbl_stock_movement` (
    `id` INT NOT NULL AUTO_INCREMENT,
    `movement_type` ENUM('IN', 'OUT', 'TRANSFER_IN', 'TRANSFER_OUT', 'ADJUSTMENT') NOT NULL,
    `warehouse_id` INT NOT NULL,
    `stock_id` INT NOT NULL,
    `quantity` INT NOT NULL,
    `reference_type` ENUM('PURCHASE', 'SALE', 'TRANSFER', 'RETURN', 'ADJUSTMENT', 'ASSIGNMENT') NULL,
    `reference_id` INT NULL COMMENT 'ID of purchase, sale, transfer, etc.',
    `movement_date` DATETIME NOT NULL,
    `assigned_person` VARCHAR(255) NULL COMMENT 'Person responsible for this movement',
    `assigned_person_id` INT NULL COMMENT 'User ID if available',
    `notes` TEXT NULL,
    `created_by` INT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_warehouse_id` (`warehouse_id`),
    INDEX `idx_stock_id` (`stock_id`),
    INDEX `idx_movement_type` (`movement_type`),
    INDEX `idx_movement_date` (`movement_date`),
    INDEX `idx_assigned_person` (`assigned_person`),
    INDEX `idx_reference` (`reference_type`, `reference_id`),
    FOREIGN KEY (`warehouse_id`) REFERENCES `tbl_warehouse`(`id`) ON DELETE RESTRICT,
    FOREIGN KEY (`stock_id`) REFERENCES `tbl_stock`(`id`) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Trigger to update warehouse stock on movement
DELIMITER $$
CREATE TRIGGER `trg_update_warehouse_stock_on_movement`
AFTER INSERT ON `tbl_stock_movement`
FOR EACH ROW
BEGIN
    IF NEW.movement_type IN ('IN', 'TRANSFER_IN') THEN
        INSERT INTO `tbl_warehouse_stock` (`warehouse_id`, `stock_id`, `quantity`, `updated_by`)
        VALUES (NEW.warehouse_id, NEW.stock_id, NEW.quantity, NEW.created_by)
        ON DUPLICATE KEY UPDATE 
            `quantity` = `quantity` + NEW.quantity,
            `updated_by` = NEW.created_by,
            `last_updated` = CURRENT_TIMESTAMP;
    ELSEIF NEW.movement_type IN ('OUT', 'TRANSFER_OUT') THEN
        UPDATE `tbl_warehouse_stock`
        SET `quantity` = GREATEST(0, `quantity` - NEW.quantity),
            `updated_by` = NEW.created_by,
            `last_updated` = CURRENT_TIMESTAMP
        WHERE `warehouse_id` = NEW.warehouse_id AND `stock_id` = NEW.stock_id;
    END IF;
END$$
DELIMITER ;

-- Insert default warehouse (Main Warehouse)
-- Only insert if no warehouses exist
INSERT INTO `tbl_warehouse` (`warehouse_code`, `name`, `description`, `is_active`) 
SELECT 'WH001', 'Main Warehouse', 'Primary storage location', 1
WHERE NOT EXISTS (SELECT 1 FROM `tbl_warehouse` WHERE `warehouse_code` = 'WH001');

