/**
 * Seeder Runner
 * Runs all seed files in order
 */

const { query } = require('../../backend/config/database');
const fs = require('fs').promises;
const path = require('path');

// Execute SQL file
const executeSqlFile = async (filePath) => {
    const sql = await fs.readFile(filePath, 'utf8');
    
    // Split by semicolons and filter out empty statements
    const statements = sql
        .split(';')
        .map(s => s.trim())
        .filter(s => s.length > 0 && !s.startsWith('--'));
    
    for (const statement of statements) {
        if (statement.trim()) {
            try {
                await query(statement);
            } catch (error) {
                // Ignore duplicate entry errors for seeds
                if (error.code !== 'ER_DUP_ENTRY') {
                    console.warn(`⚠️  Warning in statement:`, error.message);
                }
            }
        }
    }
};

// Run all seeds
const runSeeds = async () => {
    try {
        console.log('🌱 Starting database seeding...\n');
        
        // Get all seed files
        const seedsDir = __dirname;
        const files = await fs.readdir(seedsDir);
        const seedFiles = files
            .filter(f => f.endsWith('.sql'))
            .sort(); // Sort to ensure order
        
        console.log(`📋 Found ${seedFiles.length} seed files\n`);
        
        // Execute each seed
        for (const file of seedFiles) {
            const filePath = path.join(seedsDir, file);
            
            console.log(`🌱 Seeding ${file}...`);
            
            try {
                await executeSqlFile(filePath);
                console.log(`✅ Completed ${file}\n`);
            } catch (error) {
                console.error(`❌ Error in ${file}:`, error.message);
                throw error;
            }
        }
        
        console.log(`🎉 All ${seedFiles.length} seed files executed successfully!`);
        
    } catch (error) {
        console.error('\n❌ Seeding failed:', error);
        throw error;
    }
};

// Run seeds if called directly
if (require.main === module) {
    runSeeds()
        .then(() => {
            console.log('\n✅ Seeding process completed');
            process.exit(0);
        })
        .catch((error) => {
            console.error('\n❌ Seeding process failed:', error);
            process.exit(1);
        });
}

module.exports = { runSeeds };
