/**
 * Download vendor assets (Bootstrap, Bootstrap Icons, Flag Icons) for offline use
 * Run with: node frontend/download-vendor-assets.js
 */

const https = require('https');
const http = require('http');
const fs = require('fs');
const path = require('path');

const VENDOR_DIR = path.join(__dirname, 'vendor');
const BOOTSTRAP_VERSION = '5.3.2';
const BOOTSTRAP_ICONS_VERSION = '1.11.1';
const FLAG_ICONS_VERSION = '6.6.6';

// Create vendor directory structure
function createDirectories() {
    const dirs = [
        VENDOR_DIR,
        path.join(VENDOR_DIR, 'bootstrap', 'css'),
        path.join(VENDOR_DIR, 'bootstrap', 'js'),
        path.join(VENDOR_DIR, 'bootstrap-icons', 'css'),
        path.join(VENDOR_DIR, 'bootstrap-icons', 'fonts'),
        path.join(VENDOR_DIR, 'flag-icons', 'css'),
        path.join(VENDOR_DIR, 'flag-icons', 'flags')
    ];
    
    dirs.forEach(dir => {
        if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
            console.log(`✅ Created directory: ${dir}`);
        }
    });
}

// Download file from URL
function downloadFile(url, dest) {
    return new Promise((resolve, reject) => {
        const protocol = url.startsWith('https') ? https : http;
        const file = fs.createWriteStream(dest);
        
        console.log(`📥 Downloading: ${url}`);
        
        protocol.get(url, (response) => {
            if (response.statusCode === 301 || response.statusCode === 302) {
                // Handle redirect
                return downloadFile(response.headers.location, dest).then(resolve).catch(reject);
            }
            
            if (response.statusCode !== 200) {
                file.close();
                fs.unlinkSync(dest);
                reject(new Error(`Failed to download: ${response.statusCode}`));
                return;
            }
            
            response.pipe(file);
            
            file.on('finish', () => {
                file.close();
                console.log(`✅ Downloaded: ${path.basename(dest)}`);
                resolve();
            });
        }).on('error', (err) => {
            file.close();
            if (fs.existsSync(dest)) {
                fs.unlinkSync(dest);
            }
            reject(err);
        });
    });
}

// Download Bootstrap files
async function downloadBootstrap() {
    console.log('\n📦 Downloading Bootstrap...');
    
    const files = [
        {
            url: `https://cdn.jsdelivr.net/npm/bootstrap@${BOOTSTRAP_VERSION}/dist/css/bootstrap.min.css`,
            dest: path.join(VENDOR_DIR, 'bootstrap', 'css', 'bootstrap.min.css')
        },
        {
            url: `https://cdn.jsdelivr.net/npm/bootstrap@${BOOTSTRAP_VERSION}/dist/js/bootstrap.bundle.min.js`,
            dest: path.join(VENDOR_DIR, 'bootstrap', 'js', 'bootstrap.bundle.min.js')
        }
    ];
    
    for (const file of files) {
        try {
            await downloadFile(file.url, file.dest);
        } catch (error) {
            console.error(`❌ Error downloading ${file.url}:`, error.message);
        }
    }
}

// Download Bootstrap Icons
async function downloadBootstrapIcons() {
    console.log('\n📦 Downloading Bootstrap Icons...');
    
    // Download CSS
    const cssUrl = `https://cdn.jsdelivr.net/npm/bootstrap-icons@${BOOTSTRAP_ICONS_VERSION}/font/bootstrap-icons.css`;
    const cssDest = path.join(VENDOR_DIR, 'bootstrap-icons', 'css', 'bootstrap-icons.css');
    
    try {
        await downloadFile(cssUrl, cssDest);
        
        // Read CSS and extract font URLs
        const cssContent = fs.readFileSync(cssDest, 'utf8');
        const fontUrlMatch = cssContent.match(/url\(['"]?([^'"]+fonts\/[^'"]+)['"]?\)/);
        
        if (fontUrlMatch) {
            // Download font files (woff2, woff)
            const baseUrl = `https://cdn.jsdelivr.net/npm/bootstrap-icons@${BOOTSTRAP_ICONS_VERSION}/font/fonts/`;
            const fontFiles = ['bootstrap-icons.woff2', 'bootstrap-icons.woff'];
            
            for (const fontFile of fontFiles) {
                try {
                    await downloadFile(baseUrl + fontFile, path.join(VENDOR_DIR, 'bootstrap-icons', 'fonts', fontFile));
                } catch (error) {
                    console.error(`❌ Error downloading font ${fontFile}:`, error.message);
                }
            }
            
            // Update CSS to use local font paths
            const updatedCss = cssContent.replace(
                /url\(['"]?[^'"]*fonts\/([^'"]+)['"]?\)/g,
                "url('../fonts/$1')"
            );
            fs.writeFileSync(cssDest, updatedCss);
            console.log('✅ Updated Bootstrap Icons CSS with local font paths');
        }
    } catch (error) {
        console.error(`❌ Error downloading Bootstrap Icons:`, error.message);
    }
}

// Download Flag Icons
async function downloadFlagIcons() {
    console.log('\n📦 Downloading Flag Icons...');
    
    const cssUrl = `https://cdnjs.cloudflare.com/ajax/libs/flag-icon-css/${FLAG_ICONS_VERSION}/css/flag-icons.min.css`;
    const cssDest = path.join(VENDOR_DIR, 'flag-icons', 'css', 'flag-icons.min.css');
    
    try {
        await downloadFile(cssUrl, cssDest);
        
        // Read CSS and extract flag image URLs
        const cssContent = fs.readFileSync(cssDest, 'utf8');
        const flagUrlMatch = cssContent.match(/url\(['"]?([^'"]+flags\/[^'"]+)['"]?\)/);
        
        if (flagUrlMatch) {
            // Download common flag images (4x3 format)
            const baseUrl = `https://cdnjs.cloudflare.com/ajax/libs/flag-icon-css/${FLAG_ICONS_VERSION}/flags/4x3/`;
            const commonFlags = ['gb.svg', 'mm.svg', 'cn.svg', 'us.svg']; // Add more as needed
            
            for (const flag of commonFlags) {
                try {
                    await downloadFile(baseUrl + flag, path.join(VENDOR_DIR, 'flag-icons', 'flags', flag));
                } catch (error) {
                    console.error(`❌ Error downloading flag ${flag}:`, error.message);
                }
            }
            
            // Update CSS to use local flag paths
            const updatedCss = cssContent.replace(
                /url\(['"]?[^'"]*flags\/4x3\/([^'"]+)['"]?\)/g,
                "url('../flags/$1')"
            );
            fs.writeFileSync(cssDest, updatedCss);
            console.log('✅ Updated Flag Icons CSS with local flag paths');
        }
    } catch (error) {
        console.error(`❌ Error downloading Flag Icons:`, error.message);
    }
}

// Main function
async function main() {
    console.log('🚀 Starting vendor assets download...\n');
    
    try {
        createDirectories();
        await downloadBootstrap();
        await downloadBootstrapIcons();
        await downloadFlagIcons();
        
        console.log('\n✅ All vendor assets downloaded successfully!');
        console.log('\n📝 Next step: Run the update-html-files.js script to update all HTML files.');
    } catch (error) {
        console.error('\n❌ Error:', error);
        process.exit(1);
    }
}

// Run if executed directly
if (require.main === module) {
    main();
}

module.exports = { downloadBootstrap, downloadBootstrapIcons, downloadFlagIcons };
