/**
 * API Helper Functions
 * Handles all HTTP requests to backend
 */

// Use relative URL since frontend is served by same server
const API_URL = '/api';
let currentLanguage = localStorage.getItem('language') || 'en';

// Request throttling to prevent API spam
const requestThrottle = new Map();
const THROTTLE_DELAY = 100; // 100ms between requests to same endpoint

// Throttle requests to prevent API spam
async function throttleRequest(url) {
    const now = Date.now();
    const lastRequest = requestThrottle.get(url);
    
    if (lastRequest && (now - lastRequest) < THROTTLE_DELAY) {
        const delay = THROTTLE_DELAY - (now - lastRequest);
        await new Promise(resolve => setTimeout(resolve, delay));
    }
    
    requestThrottle.set(url, Date.now());
}

// Get auth token
function getAuthToken() {
    return localStorage.getItem('authToken') || localStorage.getItem('token');
}

// Set auth token
function setAuthToken(token) {
    localStorage.setItem('authToken', token);
    localStorage.setItem('token', token); // Keep both for compatibility
}

// Remove auth token
function removeAuthToken() {
    localStorage.removeItem('authToken');
    localStorage.removeItem('token');
    localStorage.removeItem('user');
    sessionStorage.removeItem('authToken');
    sessionStorage.removeItem('token');
    sessionStorage.removeItem('user');
}

// Get current user info
function getCurrentUser() {
    const userStr = localStorage.getItem('user');
    if (userStr) {
        try {
            return JSON.parse(userStr);
        } catch (e) {
            console.error('Error parsing user data:', e);
            return null;
        }
    }
    return null;
}

// Get current user name
function getCurrentUserName() {
    const user = getCurrentUser();
    return user ? (user.name || 'Unknown User') : 'Unknown User';
}

// Request timeout helper
function fetchWithTimeout(url, options = {}, timeout = 30000) {
    return Promise.race([
        fetch(url, options),
        new Promise((_, reject) =>
            setTimeout(() => reject(new Error('Request timeout')), timeout)
        )
    ]);
}

// API request helper with timeout and retry
async function apiRequest(endpoint, options = {}, retries = 2) {
    // Throttle requests to prevent API spam
    await throttleRequest(endpoint);
    
    const token = getAuthToken();
    
    const defaultOptions = {
        headers: {
            'Content-Type': 'application/json',
            'Accept-Language': currentLanguage,
            'Connection': 'keep-alive'
        }
    };

    if (token) {
        defaultOptions.headers['Authorization'] = `Bearer ${token}`;
    }

    const config = {
        ...defaultOptions,
        ...options,
        headers: {
            ...defaultOptions.headers,
            ...options.headers
        }
    };

    // Set timeout: 30 seconds for normal requests, 60 seconds for POST/PUT
    const timeout = (options.method === 'POST' || options.method === 'PUT') ? 60000 : 30000;

    for (let attempt = 0; attempt <= retries; attempt++) {
        try {
            const response = await fetchWithTimeout(`${API_URL}${endpoint}`, config, timeout);
            
            // Check if response is JSON
            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                const data = await response.json();

                if (!response.ok) {
                    // For login/register endpoints, don't redirect on 401 - let the handler show the error
                    if (response.status === 401 && !endpoint.includes('/auth/login') && !endpoint.includes('/auth/register')) {
                        removeAuthToken();
                        // Show notification before redirect
                        if (typeof showError === 'function') {
                            showError('Your session has expired. Please login again.');
                        }
                        setTimeout(() => {
                            window.location.href = '/';
                        }, 1500);
                        return;
                    }
                    
                    // Create user-friendly error messages based on status code
                    let errorMessage = data.message || `Request failed with status ${response.status}`;
                    
                    if (!data.message) {
                        switch (response.status) {
                            case 400:
                                errorMessage = 'Invalid request. Please check your input.';
                                break;
                            case 403:
                                errorMessage = 'You do not have permission to perform this action.';
                                break;
                            case 404:
                                errorMessage = 'The requested resource was not found.';
                                break;
                            case 409:
                                errorMessage = data.message || 'A conflict occurred. The item may already exist.';
                                break;
                            case 422:
                                errorMessage = data.message || 'Validation failed. Please check your input.';
                                break;
                            case 500:
                                errorMessage = 'Server error. Please try again later or contact support.';
                                break;
                            case 503:
                                errorMessage = 'Service temporarily unavailable. Please try again later.';
                                break;
                        }
                    }
                    
                    const error = new Error(errorMessage);
                    error.status = response.status;
                    error.data = data.data || data;
                    throw error;
                }

                return data;
            } else {
                // Handle non-JSON responses
                if (!response.ok) {
                    // For login/register endpoints, don't redirect on 401
                    if (response.status === 401 && !endpoint.includes('/auth/login') && !endpoint.includes('/auth/register')) {
                        removeAuthToken();
                        if (typeof showError === 'function') {
                            showError('Your session has expired. Please login again.');
                        }
                        setTimeout(() => {
                            window.location.href = '/';
                        }, 1500);
                        return;
                    }
                    throw new Error(`Request failed with status ${response.status}`);
                }
                return { success: true };
            }
        } catch (error) {
            console.error(`API Error (attempt ${attempt + 1}/${retries + 1}):`, error);
            
            // Handle network errors that might indicate auth issues
            // But don't redirect for login/register endpoints
            if ((error.message.includes('401') || error.message.includes('Unauthorized')) 
                && !endpoint.includes('/auth/login') && !endpoint.includes('/auth/register')) {
                removeAuthToken();
                if (typeof showError === 'function') {
                    showError('Your session has expired. Please login again.');
                }
                setTimeout(() => {
                    window.location.href = '/';
                }, 1500);
                return;
            }
            
            // If this is the last retry, throw the error
            if (attempt === retries) {
                // Make error message more user-friendly
                if (error.message === 'Request timeout') {
                    throw new Error('Server is taking too long to respond. Please check your connection and try again.');
                } else if (error.message.includes('Failed to fetch')) {
                    throw new Error('Cannot connect to server. Please check your internet connection.');
                } else if (error.message.includes('NetworkError') || error.message.includes('network')) {
                    throw new Error('Network error. Please check your connection and try again.');
                }
                throw error;
            }
            
            // Wait before retrying (exponential backoff)
            await new Promise(resolve => setTimeout(resolve, 1000 * (attempt + 1)));
        }
    }
}

// Auth API
const authAPI = {
    login: async (username, password) => {
        return apiRequest('/auth/login', {
            method: 'POST',
            body: JSON.stringify({ name: username, passwd: password })
        });
    },
    
    logout: async () => {
        return apiRequest('/auth/logout');
    },
    
    getMe: async () => {
        return apiRequest('/auth/me');
    }
};

// POS API
const posAPI = {
    getProducts: async (filters = {}) => {
        const params = new URLSearchParams(filters);
        return apiRequest(`/pos/products?${params}`);
    },
    
    getCategories: async () => {
        return apiRequest('/pos/categories');
    },
    
    getNextVoucher: async () => {
        return apiRequest('/pos/next-voucher');
    },
    
    getPaymentMethods: async () => {
        return apiRequest('/pos/payment-methods');
    },
    
    processSale: async (saleData) => {
        return apiRequest('/pos/sale', {
            method: 'POST',
            body: JSON.stringify(saleData)
        });
    },
    
    saveDraft: async (draftData) => {
        return apiRequest('/pos/draft', {
            method: 'POST',
            body: JSON.stringify(draftData)
        });
    },
    
    getDrafts: async () => {
        return apiRequest('/pos/drafts');
    },
    
    getDraft: async (id) => {
        return apiRequest(`/pos/draft/${id}`);
    },
    
    deleteDraft: async (id) => {
        return apiRequest(`/pos/draft/${id}`, {
            method: 'DELETE'
        });
    },
    
    getVoucherDetails: async (vno) => {
        return apiRequest(`/pos/voucher/${vno}`);
    }
};

// Dashboard API
const dashboardAPI = {
    getStats: async (startDate, endDate) => {
        const params = startDate && endDate ? `?start_date=${startDate}&end_date=${endDate}` : '';
        return apiRequest(`/dashboard/stats${params}`);
    },
    
    getRecent: async (startDate, endDate, limit = 10) => {
        const params = [];
        if (startDate && endDate) params.push(`start_date=${startDate}`, `end_date=${endDate}`);
        if (limit) params.push(`limit=${limit}`);
        const queryString = params.length > 0 ? `?${params.join('&')}` : '';
        return apiRequest(`/dashboard/recent${queryString}`);
    }
};

// Patient API
const patientAPI = {
    getAll: async (params = {}) => {
        const query = new URLSearchParams(params);
        return apiRequest(`/patients?${query}`);
    },
    
    get: async (id) => {
        return apiRequest(`/patients/${id}`);
    },
    
    create: async (data) => {
        return apiRequest('/patients', {
            method: 'POST',
            body: JSON.stringify(data)
        });
    },
    
    update: async (id, data) => {
        return apiRequest(`/patients/${id}`, {
            method: 'PUT',
            body: JSON.stringify(data)
        });
    },
    
    delete: async (id) => {
        return apiRequest(`/patients/${id}`, {
            method: 'DELETE'
        });
    },
    
    search: async (query) => {
        return apiRequest(`/patients/search?q=${query}`);
    }
};

// Stock API
const stockAPI = {
    getAll: async (params = {}) => {
        const query = new URLSearchParams(params);
        return apiRequest(`/stock?${query}`);
    },
    
    getAlerts: async () => {
        return apiRequest('/stock/alerts');
    }
};

// Change language
function setLanguage(lang) {
    currentLanguage = lang;
    localStorage.setItem('language', lang);
    
    // Update global i18n instance if available
    if (window.i18n && typeof window.i18n.changeLanguage === 'function') {
        window.i18n.changeLanguage(lang);
    } else {
        // Fallback: reload page if i18n system not available
        location.reload();
    }
}

