/**
 * Invoice API Service
 * Handles invoice generation and printing functionality
 */

class InvoiceAPI {
    constructor() {
        this.baseURL = '/api/invoice';
    }

    /**
     * Get invoice data for a specific voucher
     * @param {string} voucherNo - Voucher number
     * @returns {Promise<Object>} Invoice data
     */
    async getInvoiceData(voucherNo) {
        try {
            const response = await fetch(`${this.baseURL}/${voucherNo}`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const result = await response.json();
            return result;
        } catch (error) {
            console.error('Error fetching invoice data:', error);
            throw error;
        }
    }

    /**
     * Generate invoice data for printing
     * @param {string} voucherNo - Voucher number
     * @returns {Promise<Object>} Invoice data
     */
    async generateInvoice(voucherNo) {
        try {
            const response = await fetch(`${this.baseURL}/generate`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                },
                body: JSON.stringify({ voucherNo })
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const result = await response.json();
            return result;
        } catch (error) {
            console.error('Error generating invoice:', error);
            throw error;
        }
    }

    /**
     * Open invoice in new window for printing
     * @param {Object} invoiceData - Invoice data to display
     * @param {boolean} autoPrint - Whether to automatically trigger print dialog
     */
    async openInvoiceWindow(invoiceData, autoPrint = true) {
        try {
            // Ensure we have the latest shop info
            if (!invoiceData.shopInfo) {
                invoiceData.shopInfo = await this.getShopInfo();
            }

            // Determine invoice template based on printer type
            const printerType = invoiceData.shopInfo?.printerType || '80mm';
            const vouchersPerPage = invoiceData.shopInfo?.vouchersPerPage || 1;
            
            let templatePath = '/pages/invoice-80mm.html'; // Default
            
            switch (printerType) {
                case '58mm':
                    templatePath = '/pages/invoice-58mm.html';
                    break;
                case 'A4':
                    templatePath = '/pages/invoice-a4.html';
                    break;
                case 'A5':
                    templatePath = '/pages/invoice-a5.html';
                    break;
                default:
                    templatePath = '/pages/invoice-80mm.html';
            }

            // Store data in localStorage for the invoice page to access
            console.log('Storing invoice data in localStorage:', invoiceData);
            localStorage.setItem('invoiceData', JSON.stringify(invoiceData));

            // Open invoice page in new window
            const invoiceWindow = window.open(
                templatePath,
                'invoice',
                'width=400,height=600,scrollbars=yes,resizable=yes'
            );

            // Focus the window
            if (invoiceWindow) {
                invoiceWindow.focus();
                
                // Handle multiple vouchers per page
                if (vouchersPerPage > 1) {
                    invoiceWindow.onload = () => {
                        this.handleMultipleVouchers(invoiceWindow, invoiceData, vouchersPerPage);
                    };
                }
            }

            return invoiceWindow;
        } catch (error) {
            console.error('Error opening invoice window:', error);
            throw error;
        }
    }

    /**
     * Handle multiple vouchers per page
     * @param {Window} invoiceWindow - The invoice window
     * @param {Object} invoiceData - Invoice data
     * @param {number} vouchersPerPage - Number of vouchers per page
     */
    handleMultipleVouchers(invoiceWindow, invoiceData, vouchersPerPage) {
        // Clone the invoice container for multiple vouchers
        const originalContainer = invoiceWindow.document.querySelector('.invoice-container');
        if (!originalContainer) return;

        // Clear the body and create multiple vouchers
        invoiceWindow.document.body.innerHTML = '';
        
        for (let i = 0; i < vouchersPerPage; i++) {
            const clonedContainer = originalContainer.cloneNode(true);
            
            // Add page break for multiple vouchers (except the first one)
            if (i > 0) {
                clonedContainer.style.pageBreakBefore = 'always';
                clonedContainer.style.marginTop = '20px';
            }
            
            invoiceWindow.document.body.appendChild(clonedContainer);
            
            // Populate each voucher with data
            if (invoiceWindow.populateInvoice) {
                invoiceWindow.populateInvoice(invoiceData);
            }
        }
        
        // Print after a short delay
        setTimeout(() => {
            invoiceWindow.print();
        }, 500);
    }

    /**
     * Get shop information from settings
     * @returns {Promise<Object>} Shop information
     */
    async getShopInfo() {
        try {
            const response = await fetch('/api/settings/shop', {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                }
            });
            const result = await response.json();
            
            if (result.success && result.data) {
                return {
                    shopName: result.data.name || 'Clinic Pro',
                    shopType: result.data.type || 'Medical Center',
                    shopAddress: result.data.address || '123 Medical Street, Health City',
                    shopPhone: result.data.ph_no ? `Phone: ${result.data.ph_no}` : 'Phone: +1-234-567-8900',
                    shopEmail: result.data.email || '',
                    logoPath: result.data.photo_path || '',
                    thankYouMessage: result.data.thanks || 'Thank you for your visit!',
                    voucherFooter: result.data.remark || 'Please keep this receipt for your records',
                    printerType: result.data.printer || '80mm',
                    vouchersPerPage: result.data.printer_voucher_count || 1
                };
            } else {
                // Fallback to default values
                return {
                    shopName: 'Clinic Pro',
                    shopType: 'Medical Center',
                    shopAddress: '123 Medical Street, Health City',
                    shopPhone: 'Phone: +1-234-567-8900',
                    shopEmail: '',
                    logoPath: '',
                    thankYouMessage: 'Thank you for your visit!',
                    voucherFooter: 'Please keep this receipt for your records',
                    printerType: '80mm',
                    vouchersPerPage: 1
                };
            }
        } catch (error) {
            console.error('Error fetching shop info:', error);
            // Return fallback values
            return {
                shopName: 'Clinic Pro',
                shopType: 'Medical Center',
                shopAddress: '123 Medical Street, Health City',
                shopPhone: 'Phone: +1-234-567-8900',
                shopEmail: '',
                logoPath: '',
                thankYouMessage: 'Thank you for your visit!',
                voucherFooter: 'Please keep this receipt for your records',
                printerType: '80mm',
                vouchersPerPage: 1
            };
        }
    }

    /**
     * Print invoice for a completed sale
     * @param {string} voucherNo - Voucher number
     * @param {Object} saleData - Sale data (optional, will fetch if not provided)
     */
    async printInvoice(voucherNo, saleData = null) {
        try {
            let invoiceData;

            // Always fetch shop info from database
            const shopInfo = await this.getShopInfo();

            if (saleData) {
                console.log('Sale data received:', saleData);
                console.log('Sale data items:', saleData.items);
                
                // Use provided sale data to create invoice (for immediate printing)
                invoiceData = {
                    voucherNo: voucherNo,
                    date: new Date().toISOString(),
                    cashier: saleData.cashier || 'Current User',
                    customerName: saleData.customer_name || 'Walk-in',
                    doctorName: saleData.doctor_name || '',
                    patientName: saleData.patient_name || '',
                    items: saleData.items || [],
                    subtotal: saleData.sub_total || 0,
                    discount: saleData.discount || 0,
                    tax: saleData.tax || 0,
                    total: saleData.total || 0,
                    paid: saleData.total || 0,
                    paymentMethod: saleData.payment_method || 'Cash',
                    shopInfo: shopInfo
                };
                
                console.log('Invoice data created:', invoiceData);
            } else {
                // Fetch complete invoice data from server (includes shop info)
                const result = await this.getInvoiceData(voucherNo);
                invoiceData = result.data;
                // Ensure shop info is up to date
                invoiceData.shopInfo = shopInfo;
            }

            // Open invoice window
            await this.openInvoiceWindow(invoiceData, true);

        } catch (error) {
            console.error('Error printing invoice:', error);
            throw error;
        }
    }
}

// Create and export instance
const invoiceAPI = new InvoiceAPI();
window.invoiceAPI = invoiceAPI; // Make available globally

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = invoiceAPI;
}
