/**
 * Authentication Check Script
 * Include this in all protected pages to ensure user is logged in
 */

// Check authentication on page load
document.addEventListener('DOMContentLoaded', function() {
    checkAuthentication();
    applyCreatePermissionVisibility();
    applyReadPermissionVisibility();
});

function checkAuthentication() {
    const token = getAuthToken();
    
    if (!token) {
        // No token found, redirect to login
        window.location.href = '/';
        return false;
    }
    
    // Verify token with server
    verifyToken(token);
    return true;
}

async function verifyToken(token) {
    try {
        const response = await fetch('/api/auth/verify', {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
            }
        });
        
        if (response.status === 401 || response.status === 403) {
            // Token is invalid or expired
            removeAuthToken();
            window.location.href = '/';
            return;
        }
        
        if (response.ok) {
            const data = await response.json();
            const userData = data.user || data.data || null;
            const permissionsData = data.permissions || data.data?.permissions || null;

            if (data.success && userData) {
                // Update user info in localStorage
                localStorage.setItem('user', JSON.stringify(userData));
                if (permissionsData) {
                    localStorage.setItem('permissions', JSON.stringify(permissionsData));
                }
                updateUserInfo(userData);
                applyCreatePermissionVisibility();
                applyReadPermissionVisibility();
            }
        }
    } catch (error) {
        console.error('Token verification error:', error);
        // On network error, don't redirect immediately, but show warning
        console.warn('Could not verify authentication token');
    }
}

function updateUserInfo(user) {
    // Update user info in UI if elements exist
    const userNameElement = document.getElementById('userName');
    const userRoleElement = document.getElementById('userRole');
    
    if (userNameElement) {
        userNameElement.textContent = user.name || 'User';
    }
    
    if (userRoleElement) {
        userRoleElement.textContent = user.role || 'User';
    }
}

function getStoredPermissions() {
    try {
        const perms = localStorage.getItem('permissions');
        return perms ? JSON.parse(perms) : {};
    } catch (e) {
        return {};
    }
}

function canPermission(resource, action) {
    const perms = getStoredPermissions();
    return perms?.[resource]?.[action] === true;
}

function getPagePermissionResource() {
    const page = window.location.pathname.split('/').pop().replace('.html', '');
    const map = {
        'dashboard': 'dashboard',
        'purchases': 'purchases',
        'add-purchase': 'purchases',
        'sales': 'sales',
        'products': 'stock',
        'suppliers': 'suppliers',
        'expenses': 'expenses',
        'patients': 'patients',
        'doctors': 'doctors',
        'appointments': 'appointments',
        'inpatients': 'inpatients',
        'warehouse-stock': 'warehouse',
        'warehouse-stock-assignment': 'warehouse',
        'warehouse-transfers': 'warehouse',
        'warehouse-history': 'warehouse',
        'warehouses': 'warehouse',
        'stock-expense': 'stockExpense',
        'reports': 'reports',
        'profit-loss': 'reports',
        'stock-sales-report': 'reports',
        'service-sales-report': 'reports',
        'voucher-sales-report': 'reports',
        'lab-reports': 'reports',
        'settings': 'settings',
        'emr': 'emr',
        'emr-all': 'emr',
        'laboratory': 'laboratory',
        'advanced-pharmacy': 'pharmacy'
    };
    return map[page] || null;
}

function hideElements(selectors) {
    selectors.forEach((selector) => {
        document.querySelectorAll(selector).forEach((el) => {
            el.style.display = 'none';
        });
    });
}

function applyCreatePermissionVisibility() {
    const resource = getPagePermissionResource();
    if (!resource) return;
    if (canPermission(resource, 'create')) return;

    const selectors = [
        'a[href*="add-"]',
        'a[onclick*="add-"]',
        'button[onclick*="add-"]',
        '[data-bs-target*="add"]',
        '.fab',
        '.clickable-card[onclick*="add-"]',
        'button[onclick*="submitPurchase"]'
    ];
    hideElements(selectors);
}

function getFirstAllowedPage() {
    const pageResourcePairs = [
        { page: 'dashboard', resource: 'dashboard' },
        { page: 'purchases', resource: 'purchases' },
        { page: 'sales', resource: 'sales' },
        { page: 'products', resource: 'stock' },
        { page: 'patients', resource: 'patients' },
        { page: 'doctors', resource: 'doctors' },
        { page: 'appointments', resource: 'appointments' },
        { page: 'inpatients', resource: 'inpatients' },
        { page: 'suppliers', resource: 'suppliers' },
        { page: 'expenses', resource: 'expenses' },
        { page: 'warehouse-stock', resource: 'warehouse' },
        { page: 'reports', resource: 'reports' },
        { page: 'emr', resource: 'emr' },
        { page: 'laboratory', resource: 'laboratory' },
        { page: 'advanced-pharmacy', resource: 'pharmacy' }
    ];

    for (const entry of pageResourcePairs) {
        if (canPermission(entry.resource, 'read')) {
            return entry.page;
        }
    }
    return null;
}

function applyReadPermissionVisibility() {
    const perms = getStoredPermissions();
    if (!perms || Object.keys(perms).length === 0) {
        // Permissions not loaded yet; avoid redirect loops.
        return;
    }

    // Hide nav links for pages without read permission
    const linkMap = {
        'dashboard': 'dashboard',
        'purchases': 'purchases',
        'add-purchase': 'purchases',
        'sales': 'sales',
        'products': 'stock',
        'patients': 'patients',
        'doctors': 'doctors',
        'appointments': 'appointments',
        'inpatients': 'inpatients',
        'suppliers': 'suppliers',
        'expenses': 'expenses',
        'warehouse-stock': 'warehouse',
        'warehouse-stock-assignment': 'warehouse',
        'warehouse-transfers': 'warehouse',
        'warehouse-history': 'warehouse',
        'warehouses': 'warehouse',
        'stock-expense': 'stockExpense',
        'reports': 'reports',
        'profit-loss': 'reports',
        'stock-sales-report': 'reports',
        'service-sales-report': 'reports',
        'voucher-sales-report': 'reports',
        'lab-reports': 'reports',
        'emr': 'emr',
        'emr-all': 'emr',
        'laboratory': 'laboratory',
        'advanced-pharmacy': 'pharmacy'
    };

    document.querySelectorAll('a.nav-link[href]').forEach((link) => {
        const href = link.getAttribute('href') || '';
        const page = href.split('/').pop().replace('.html', '').replace('#', '');
        const resource = linkMap[page];
        if (resource && !canPermission(resource, 'read')) {
            link.style.display = 'none';
        }
    });

    // If current page is not readable, redirect to first allowed page
    const currentResource = getPagePermissionResource();
    if (currentResource && !canPermission(currentResource, 'read')) {
        const fallback = getFirstAllowedPage();
        if (fallback) {
            window.location.replace(`/pages/${fallback}`);
        } else {
            window.location.replace('/pages/no-access');
        }
    }
}

// Global logout function
function logout() {
    if (confirm('Are you sure you want to logout?')) {
        // Get token before clearing
        const token = getAuthToken();
        
        // Clear ALL storage IMMEDIATELY (synchronously) to prevent auto-login
        localStorage.clear();
        sessionStorage.clear();
        
        // Double-check by manually removing known keys
        localStorage.removeItem('authToken');
        localStorage.removeItem('token');
        localStorage.removeItem('user');
        sessionStorage.removeItem('authToken');
        sessionStorage.removeItem('token');
        sessionStorage.removeItem('user');
        
        // Try to call backend to record logout time (don't await, fire and forget)
        if (token) {
            fetch('/api/auth/logout', {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                },
                keepalive: true  // Ensures request completes even after page unloads
            }).catch(err => console.error('Logout API error:', err));
        }
        
        // Use setTimeout to ensure storage clear operations complete
        setTimeout(() => {
            // Final storage clear before redirect
            localStorage.clear();
            sessionStorage.clear();
            
            // Redirect to login page with cache buster
            window.location.replace('/?_=' + Date.now());
        }, 100);
    }
}
