/**
 * Currency Utility Functions
 * Provides consistent currency formatting across the application
 */

// Default currency settings
let currencySettings = {
    symbol: '$',
    position: 'before',
    code: 'USD'
};

/**
 * Load currency settings from the server
 */
async function loadCurrencySettings() {
    try {
        // Check if getAuthToken function is available
        const authToken = typeof getAuthToken === 'function' ? getAuthToken() : null;
        
        if (!authToken) {
            console.warn('No auth token available, using default currency settings');
            return;
        }
        
        const response = await fetch(`${API_URL}/settings/shop`, {
            headers: { 'Authorization': `Bearer ${authToken}` }
        });
        const data = await response.json();
        
        if (data.success && data.data) {
            currencySettings = {
                symbol: data.data.currency_symbol || '$',
                position: data.data.currency_position || 'before',
                code: data.data.currency || 'USD'
            };
        }
    } catch (error) {
        console.error('Error loading currency settings:', error);
    }
}

/**
 * Format a number as currency using the current settings
 * @param {number} amount - The amount to format
 * @param {boolean} showDecimals - Whether to show decimal places (default: true)
 * @returns {string} Formatted currency string
 */
function formatCurrency(amount, showDecimals = false) {
    if (amount === null || amount === undefined || isNaN(amount)) {
        return currencySettings.position === 'before' ? `${currencySettings.symbol}0` : `0${currencySettings.symbol}`;
    }
    
    const numAmount = parseFloat(amount);
    const roundedAmount = Math.round(numAmount);
    const formattedAmount = new Intl.NumberFormat('en-US').format(roundedAmount);
    
    if (currencySettings.position === 'before') {
        return `${currencySettings.symbol}${formattedAmount}`;
    } else {
        return `${formattedAmount}${currencySettings.symbol}`;
    }
}

/**
 * Format a number as currency without decimal places
 * @param {number} amount - The amount to format
 * @returns {string} Formatted currency string without decimals
 */
function formatCurrencyNoDecimals(amount) {
    return formatCurrency(amount, false);
}

/**
 * Get the currency symbol
 * @returns {string} Currency symbol
 */
function getCurrencySymbol() {
    return currencySettings.symbol;
}

/**
 * Get the currency code
 * @returns {string} Currency code
 */
function getCurrencyCode() {
    return currencySettings.code;
}

/**
 * Get the currency position
 * @returns {string} 'before' or 'after'
 */
function getCurrencyPosition() {
    return currencySettings.position;
}

/**
 * Update currency settings (for when settings change)
 * @param {Object} settings - New currency settings
 */
function updateCurrencySettings(settings) {
    currencySettings = {
        symbol: settings.currency_symbol || '$',
        position: settings.currency_position || 'before',
        code: settings.currency || 'USD'
    };
}

// Load currency settings when the script loads
if (typeof window !== 'undefined') {
    // Only load if we're in a browser environment
    loadCurrencySettings();
}
