/**
 * Frontend Translation System for Clinic Pro V3
 * Handles loading and applying translations
 */

class I18nManager {
    constructor() {
        this.currentLanguage = localStorage.getItem('language') || 'en';
        this.translations = {};
        this.fallbackTranslations = {};
        this.isLoading = false;
    }

    /**
     * Initialize translation system
     */
    async init() {
        await this.loadTranslations(this.currentLanguage);
        this.applyTranslations();
        this.setupLanguageSelector();
    }

    /**
     * Load translations from backend
     */
    async loadTranslations(lang) {
        if (this.isLoading) return;
        
        this.isLoading = true;
        
        try {
            // Load main language
            const response = await fetch(`/api/translations/${lang}`);
            if (response.ok) {
                this.translations = await response.json();
            } else {
                console.error(`Failed to load translations for ${lang}`);
                // Load fallback English if not already English
                if (lang !== 'en') {
                    const fallbackResponse = await fetch('/api/translations/en');
                    if (fallbackResponse.ok) {
                        this.fallbackTranslations = await fallbackResponse.json();
                    }
                }
            }
        } catch (error) {
            console.error('Error loading translations:', error);
            // Use fallback English if available
            if (lang !== 'en') {
                try {
                    const fallbackResponse = await fetch('/api/translations/en');
                    if (fallbackResponse.ok) {
                        this.fallbackTranslations = await fallbackResponse.json();
                    }
                } catch (fallbackError) {
                    console.error('Error loading fallback translations:', fallbackError);
                }
            }
        } finally {
            this.isLoading = false;
        }
    }

    /**
     * Get translation by key
     */
    t(key, params = {}) {
        const keys = key.split('.');
        let value = this.translations;
        
        // Try to get value from main translations
        for (const k of keys) {
            if (value && typeof value === 'object' && k in value) {
                value = value[k];
            } else {
                value = null;
                break;
            }
        }
        
        // If not found, try fallback translations
        if (!value && Object.keys(this.fallbackTranslations).length > 0) {
            value = this.fallbackTranslations;
            for (const k of keys) {
                if (value && typeof value === 'object' && k in value) {
                    value = value[k];
                } else {
                    value = null;
                    break;
                }
            }
        }
        
        // If still not found, return the key itself
        if (!value) {
            console.warn(`Translation key "${key}" not found`);
            return key;
        }
        
        // Replace parameters if any
        if (typeof value === 'string' && Object.keys(params).length > 0) {
            for (const [param, replacement] of Object.entries(params)) {
                value = value.replace(`{{${param}}}`, replacement);
            }
        }
        
        return value;
    }

    /**
     * Apply translations to DOM elements
     */
    applyTranslations() {
        // Find all elements with data-i18n attribute
        const elements = document.querySelectorAll('[data-i18n]');
        
        elements.forEach(element => {
            const key = element.getAttribute('data-i18n');
            const translation = this.t(key);
            
            // Check if element has data-i18n-attr for attribute translation
            const attrKey = element.getAttribute('data-i18n-attr');
            if (attrKey) {
                const [attr, translationKey] = attrKey.split(':');
                if (attr && translationKey) {
                    const attrTranslation = this.t(translationKey);
                    element.setAttribute(attr, attrTranslation);
                }
            } else {
                // Default: replace text content
                element.textContent = translation;
            }
        });

        // Apply translations to specific elements by ID or class
        this.applySpecificTranslations();
    }

    /**
     * Apply translations to specific elements that can't use data-i18n
     */
    applySpecificTranslations() {
        // Page title
        if (document.title === 'Clinic Pro V3') {
            document.title = this.t('app.name');
        }

        // Language selector buttons - highlight active language
        const langButtons = {
            'en': document.getElementById('langBtnEn'),
            'my': document.getElementById('langBtnMy'),
            'zh': document.getElementById('langBtnZh')
        };
        
        // Remove active class from all buttons
        Object.values(langButtons).forEach(btn => {
            if (btn) {
                btn.classList.remove('active');
                btn.style.opacity = '0.6';
                btn.style.transform = 'scale(1)';
            }
        });
        
        // Add active class to current language button
        const currentLangBtn = langButtons[this.currentLanguage];
        if (currentLangBtn) {
            currentLangBtn.classList.add('active');
            currentLangBtn.style.opacity = '1';
            currentLangBtn.style.transform = 'scale(1.1)';
        }
        
        // Also handle select dropdowns if they still exist (for backward compatibility)
        const langSelect = document.getElementById('languageSelect');
        if (langSelect) {
            langSelect.value = this.currentLanguage;
        }

        // Update placeholders
        const elementsWithPlaceholders = document.querySelectorAll('[data-i18n-placeholder]');
        elementsWithPlaceholders.forEach(element => {
            const key = element.getAttribute('data-i18n-placeholder');
            const translation = this.t(key);
            element.setAttribute('placeholder', translation);
        });

        // Update titles/tooltips
        const elementsWithTitles = document.querySelectorAll('[data-i18n-title]');
        elementsWithTitles.forEach(element => {
            const key = element.getAttribute('data-i18n-title');
            const translation = this.t(key);
            element.setAttribute('title', translation);
        });

        // Auto-translate common button and label patterns
        this.autoTranslateCommonElements();
    }

    /**
     * Automatically translate common button and label text patterns
     */
    autoTranslateCommonElements() {
        // Common button text mappings
        const buttonMappings = {
            'Save': 'common.save',
            'Cancel': 'common.cancel',
            'Edit': 'common.edit',
            'Delete': 'common.delete',
            'Search': 'common.search',
            'Filter': 'common.filter',
            'Export': 'common.export',
            'Import': 'common.import',
            'Close': 'common.close',
            'Back': 'common.back',
            'Next': 'common.next',
            'Previous': 'common.previous',
            'Yes': 'common.yes',
            'No': 'common.no',
            'Confirm': 'common.confirm',
            'Loading...': 'common.loading',
            'Loading': 'common.loading',
            'View': 'common.view',
            'Details': 'common.details',
            'Print': 'common.print',
            'Refresh': 'common.refresh',
            'Add': 'common.add',
            'Update': 'common.update',
            'Create': 'common.create',
            'Remove': 'common.remove',
            'Submit': 'common.submit',
            'Reset': 'common.reset',
            'Clear': 'common.clear',
            'Dashboard': 'nav.dashboard',
            'Patients': 'nav.patients',
            'Doctors': 'nav.doctors',
            'Appointments': 'nav.appointments',
            'Stock & Inventory': 'nav.stock',
            'Sales / POS': 'nav.sales',
            'Purchases': 'nav.purchases',
            'Reports': 'nav.reports',
            'Settings': 'nav.settings',
            'Logout': 'nav.logout',
            'Add New Patient': 'patient.add_new',
            'Add Patient': 'buttons.add_patient',
            'Add New Doctor': 'doctor.add_new',
            'Add Doctor': 'buttons.add_doctor',
            'Add New Item': 'stock.add_new',
            'Add Product': 'buttons.add_product',
            'New Sale': 'sale.new_sale',
            'New Purchase': 'purchase.new_purchase',
            'New Report': 'laboratory.new_report',
            'Add Warehouse': 'warehouse.add_warehouse',
            'Admit Patient': 'inpatient.admit_patient',
            'Admit Inpatient': 'labels.admit_inpatient',
            'Add User': 'labels.add_user',
            'Lab Reports': 'laboratory.title',
            'Point of Sale': 'sale.title',
            'Inpatients (IPD)': 'inpatient.title',
            'Patient Management': 'patient.title',
            'Doctor Management': 'doctor.title',
            'Warehouse Management': 'warehouse.title',
            'Hospital Operations': 'hospital.title',
            'Bed Management': 'hospital.bed_management',
            'Complete Sale': 'sale.complete_sale',
            'Print Voucher': 'sale.print_voucher',
            'Generate Report': 'report.generate',
            'Export to Excel': 'report.export_excel',
            'Export to PDF': 'report.export_pdf',
            'Save Product': 'labels.save_product',
            'Download Template': 'labels.download_template',
            'Choose Excel': 'labels.choose_excel',
            'Import': 'labels.import',
            'Generate': 'labels.generate',
            'Expiry': 'labels.expiry',
            'IPD Sales': 'labels.ipd_sales',
            'Products & Stock Management': 'labels.products_stock_management'
        };

        // Translate buttons
        document.querySelectorAll('button, .btn, a.btn').forEach(btn => {
            const text = btn.textContent.trim();
            // Skip if already has data-i18n or is empty
            if (btn.hasAttribute('data-i18n') || !text) return;
            
            // Check if text matches a known pattern
            for (const [pattern, key] of Object.entries(buttonMappings)) {
                if (text === pattern || text.includes(pattern)) {
                    const translation = this.t(key);
                    if (translation && translation !== key) {
                        // Preserve icon if exists
                        const icon = btn.querySelector('i');
                        if (icon) {
                            btn.innerHTML = icon.outerHTML + ' ' + translation;
                        } else {
                            btn.textContent = translation;
                        }
                        break;
                    }
                }
            }
        });

        // Translate common labels in form labels
        const labelMappings = {
            'Name': 'labels.name',
            'Code': 'labels.code',
            'Description': 'labels.description',
            'Category': 'labels.category',
            'Price': 'labels.price',
            'Quantity': 'labels.quantity',
            'Unit': 'labels.unit',
            'Total': 'labels.total',
            'Subtotal': 'labels.subtotal',
            'Discount': 'labels.discount',
            'Tax': 'labels.tax',
            'Amount': 'labels.amount',
            'Date': 'labels.date',
            'Time': 'labels.time',
            'Status': 'labels.status',
            'Type': 'labels.type',
            'Phone': 'labels.phone',
            'Email': 'labels.email',
            'Address': 'labels.address',
            'Notes': 'labels.notes',
            'Remarks': 'labels.remarks',
            'Patient Name': 'patient.name',
            'Patient ID': 'patient.patient_id',
            'Age': 'patient.age',
            'Phone Number': 'patient.phone',
            'Address': 'patient.address',
            'Doctor': 'patient.doctor',
            'Room Number': 'patient.room_no',
            'Deposit': 'patient.deposit',
            'Barcode': 'stock.barcode',
            'Item Name': 'stock.item_name',
            'Product Name': 'labels.product_name',
            'Cost Price': 'stock.cost_price',
            'Sale Price': 'stock.sale_price',
            'Expiry Date': 'stock.expiry_date',
            'Stock Quantity': 'labels.stock_quantity',
            'Stock Status': 'labels.stock_status',
            'Search Products': 'labels.search_products',
            'Voucher No.': 'sale.voucher_no',
            'Customer': 'sale.customer',
            'Cashier': 'sale.cashier',
            'Payment Method': 'sale.payment_method',
            'Paid Amount': 'sale.paid',
            'Change': 'sale.change',
            'Username': 'labels.username',
            'Password': 'labels.password',
            'Role': 'labels.role',
            'Full Name': 'labels.full_name',
            'Select Category': 'labels.select_category',
            'Select Role': 'labels.select_role',
            'Select Warehouse': 'labels.select_warehouse',
            'Select Doctor': 'labels.select_doctor',
            'All Categories': 'labels.all_categories',
            'All Stock': 'labels.all_stock',
            'Low Stock': 'labels.low_stock',
            'Out of Stock': 'labels.out_of_stock',
            'Good Stock': 'labels.good_stock',
            'Alert Before': 'labels.alert_before',
            'Product Photo': 'labels.product_photo',
            'Expiry Alerts': 'labels.expiry_alerts',
            'New Lab Order': 'labels.new_lab_order'
        };

        // Translate form labels
        document.querySelectorAll('label, .form-label, .label').forEach(label => {
            const text = label.textContent.trim();
            if (label.hasAttribute('data-i18n') || !text) return;
            
            // Remove asterisks and other markers for matching
            const cleanText = text.replace(/\s*\*.*$/, '').trim();
            
            // Sort patterns by length (longest first) to match specific patterns before generic ones
            // This ensures "Cost Price" and "Sale Price" are matched before "Price"
            const sortedMappings = Object.entries(labelMappings).sort((a, b) => b[0].length - a[0].length);
            
            let matched = false;
            for (const [pattern, key] of sortedMappings) {
                // Check for exact match first (most specific)
                if (cleanText === pattern) {
                    const translation = this.t(key);
                    if (translation && translation !== key) {
                        // Preserve required asterisk or other markers
                        const hasRequired = text.includes('*') || label.querySelector('.text-danger');
                        const originalText = label.innerHTML;
                        // Try to preserve HTML structure if it exists
                        if (originalText.includes('<')) {
                            label.innerHTML = originalText.replace(pattern, translation);
                        } else {
                            label.textContent = translation + (hasRequired ? ' *' : '');
                        }
                        matched = true;
                        break;
                    }
                }
            }
            
            // If no exact match found, try partial matches (but only for patterns that don't have longer variants)
            if (!matched) {
                for (const [pattern, key] of sortedMappings) {
                    // Only use includes for patterns that are standalone words
                    // Skip "Price" if we already have "Cost Price" or "Sale Price" in mappings
                    if (pattern === 'Price' && (labelMappings['Cost Price'] || labelMappings['Sale Price'])) {
                        // Check if this label contains "Cost Price" or "Sale Price" first
                        if (cleanText.includes('Cost Price') || cleanText.includes('Sale Price')) {
                            continue; // Skip generic "Price" if it's part of a longer pattern
                        }
                    }
                    
                    if (cleanText.startsWith(pattern + ' ') || cleanText === pattern) {
                        const translation = this.t(key);
                        if (translation && translation !== key) {
                            const hasRequired = text.includes('*') || label.querySelector('.text-danger');
                            const originalText = label.innerHTML;
                            if (originalText.includes('<')) {
                                label.innerHTML = originalText.replace(pattern, translation);
                            } else {
                                label.textContent = translation + (hasRequired ? ' *' : '');
                            }
                            break;
                        }
                    }
                }
            }
        });
        
        // Translate modal titles
        document.querySelectorAll('.modal-title, h5.modal-title').forEach(title => {
            const text = title.textContent.trim();
            if (title.hasAttribute('data-i18n') || !text) return;
            
            for (const [pattern, key] of Object.entries(buttonMappings)) {
                if (text === pattern || text.includes(pattern)) {
                    const translation = this.t(key);
                    if (translation && translation !== key) {
                        // Preserve icon if exists
                        const icon = title.querySelector('i');
                        if (icon) {
                            title.innerHTML = icon.outerHTML + ' ' + translation;
                        } else {
                            title.textContent = translation;
                        }
                        break;
                    }
                }
            }
        });
        
        // Translate placeholders
        document.querySelectorAll('input[placeholder], textarea[placeholder]').forEach(input => {
            const placeholder = input.getAttribute('placeholder');
            if (!placeholder || input.hasAttribute('data-i18n-placeholder')) return;
            
            // Common placeholder patterns
            const placeholderMappings = {
                'Search by name, barcode, or category': 'labels.search_by_name_barcode_category',
                'Enter or generate barcode': 'labels.enter_or_generate_barcode',
                'Search by name, ID, or phone': 'patient.search_placeholder',
                'e.g., pcs, mg, ml': 'stock.unit',
                'Search by ID, Name or Phone': 'patient.search_placeholder'
            };
            
            for (const [pattern, key] of Object.entries(placeholderMappings)) {
                if (placeholder.includes(pattern) || placeholder === pattern) {
                    const translation = this.t(key);
                    if (translation && translation !== key) {
                        input.setAttribute('placeholder', translation);
                        break;
                    }
                }
            }
        });
        
        // Translate option text in selects
        document.querySelectorAll('select option').forEach(option => {
            const text = option.textContent.trim();
            if (option.hasAttribute('data-i18n') || !text || !option.value) return;
            
            const optionMappings = {
                'Select Category': 'labels.select_category',
                'Select Role': 'labels.select_role',
                'Select Warehouse': 'labels.select_warehouse',
                'Select Doctor': 'labels.select_doctor',
                'All Categories': 'labels.all_categories',
                'All Stock': 'labels.all_stock',
                'Low Stock': 'labels.low_stock',
                'Out of Stock': 'labels.out_of_stock',
                'Good Stock': 'labels.good_stock',
                'All Doctors': 'patient.all_doctors',
                'All Patients': 'common.all'
            };
            
            for (const [pattern, key] of Object.entries(optionMappings)) {
                if (text === pattern) {
                    const translation = this.t(key);
                    if (translation && translation !== key) {
                        option.textContent = translation;
                        break;
                    }
                }
            }
        });

        // Translate table headers
        document.querySelectorAll('th').forEach(th => {
            const text = th.textContent.trim();
            if (th.hasAttribute('data-i18n') || !text) return;
            
            for (const [pattern, key] of Object.entries(labelMappings)) {
                if (text === pattern) {
                    const translation = this.t(key);
                    if (translation && translation !== key) {
                        th.textContent = translation;
                        break;
                    }
                }
            }
        });
    }

    /**
     * Change language
     */
    async changeLanguage(lang) {
        if (lang === this.currentLanguage) return;
        
        this.currentLanguage = lang;
        localStorage.setItem('language', lang);
        
        // Update API language preference
        if (typeof setLanguage === 'function') {
            setLanguage(lang);
        }
        
        // Reload translations
        await this.loadTranslations(lang);
        
        // Reapply translations
        this.applyTranslations();
        
        // Update language selector
        const langSelect = document.getElementById('languageSelect');
        if (langSelect) {
            langSelect.value = lang;
        }
        
        // Trigger custom event for other components
        window.dispatchEvent(new CustomEvent('languageChanged', { 
            detail: { language: lang } 
        }));
    }

    /**
     * Setup language selector event listener
     */
    setupLanguageSelector() {
        // Handle flag buttons
        const langButtons = {
            'en': document.getElementById('langBtnEn'),
            'my': document.getElementById('langBtnMy'),
            'zh': document.getElementById('langBtnZh')
        };
        
        // Add click listeners to flag buttons
        Object.entries(langButtons).forEach(([lang, btn]) => {
            if (btn) {
                btn.addEventListener('click', async () => {
                    await this.changeLanguage(lang);
                });
                
                // Set initial active state
                if (lang === this.currentLanguage) {
                    btn.classList.add('active');
                    btn.style.opacity = '1';
                    btn.style.transform = 'scale(1.1)';
                } else {
                    btn.style.opacity = '0.6';
                }
            }
        });
        
        // Also handle select dropdowns if they still exist (for backward compatibility)
        const langSelect = document.getElementById('languageSelect');
        if (langSelect) {
            langSelect.value = this.currentLanguage;
            
            langSelect.addEventListener('change', async (e) => {
                await this.changeLanguage(e.target.value);
            });
        }
    }

    /**
     * Add translations to existing elements dynamically
     */
    translateElement(element, key, attribute = null) {
        const translation = this.t(key);
        
        if (attribute) {
            element.setAttribute(attribute, translation);
        } else {
            element.textContent = translation;
        }
    }

    /**
     * Get current language
     */
    getCurrentLanguage() {
        return this.currentLanguage;
    }

    /**
     * Check if translations are loaded
     */
    isReady() {
        return Object.keys(this.translations).length > 0 || Object.keys(this.fallbackTranslations).length > 0;
    }
}

// Create global instance
window.i18n = new I18nManager();

// Auto-initialize when DOM is loaded
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        window.i18n.init();
    });
} else {
    // DOM already loaded
    window.i18n.init();
}

// Export for modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = I18nManager;
}
