/**
 * Notification Bell Functionality
 * Handles notification loading and display for all pages
 */

let notificationDropdownOpen = false;
let notificationData = [];

// Ensure dependencies are available
function ensureDependencies() {
    // Check if getAuthToken is available
    if (typeof getAuthToken === 'undefined') {
        console.error('getAuthToken function not found. Make sure api.js is loaded before notification-bell.js');
        return false;
    }
    return true;
}

// Load notifications for the bell
async function loadNotificationBell() {
    if (!ensureDependencies()) return;
    
    try {
        const response = await fetch('/api/notifications', {
            headers: { 
                'Authorization': `Bearer ${getAuthToken()}`,
                'Content-Type': 'application/json'
            }
        });
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const data = await response.json();
        
        if (data.success) {
            // Get read notifications from localStorage
            const readNotifications = JSON.parse(localStorage.getItem('readNotifications') || '[]');
            
            // Filter out read notifications
            const unreadNotifications = (data.data || []).filter(notification => 
                !readNotifications.includes(notification.id)
            );
            
            const count = unreadNotifications.length;
            updateNotificationBadge(count);
        } else {
            console.warn('Notification API returned error:', data.message);
            // Show 0 count but don't error out
            updateNotificationBadge(0);
        }
    } catch (error) {
        console.warn('Error loading notification count:', error.message);
        // Don't show error to user, just hide badge
        updateNotificationBadge(0);
    }
}

// Update notification badge
function updateNotificationBadge(count) {
    const badge = document.getElementById('notificationCount');
    if (!badge) return;
    
    if (count > 0) {
        badge.textContent = count > 99 ? '99+' : count;
        badge.style.display = 'block';
        badge.classList.add('notification-count-animate');
        setTimeout(() => badge.classList.remove('notification-count-animate'), 500);
    } else {
        badge.style.display = 'none';
    }
}

// Load recent notifications for dropdown
async function loadRecentNotifications() {
    if (!ensureDependencies()) return;
    
    try {
        const response = await fetch('/api/notifications', {
            headers: { 
                'Authorization': `Bearer ${getAuthToken()}`,
                'Content-Type': 'application/json'
            }
        });
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const data = await response.json();
        
        if (data.success) {
            // Get read notifications from localStorage
            const readNotifications = JSON.parse(localStorage.getItem('readNotifications') || '[]');
            
            // Mark notifications as read based on localStorage
            notificationData = (data.data || []).map(notification => ({
                ...notification,
                read: readNotifications.includes(notification.id)
            }));
            
            renderNotificationDropdown(notificationData.slice(0, 5)); // Show only recent 5
        } else {
            throw new Error(data.message || 'Failed to load notifications');
        }
    } catch (error) {
        console.warn('Error loading notifications:', error.message);
        const container = document.getElementById('notificationList');
        if (container) {
            container.innerHTML = `
                <div class="text-center p-3 text-muted">
                    <i class="bi bi-wifi-off"></i>
                    <p class="mb-0 mt-2 small">Cannot load notifications</p>
                </div>
            `;
        }
    }
}

function renderNotificationDropdown(notifications) {
    const container = document.getElementById('notificationList');
    if (!container) return;
    
    if (!notifications.length) {
        container.innerHTML = `
            <div class="text-center p-3">
                <i class="bi bi-bell-slash opacity-50" style="font-size: 2rem;"></i>
                <p class="mb-0 mt-2 opacity-75">No notifications</p>
            </div>
        `;
        return;
    }

    container.innerHTML = notifications.map(notification => {
        const severityClass = getSeverityClass(notification.severity);
        const severityIcon = getSeverityIcon(notification.severity);
        const timeAgo = getTimeAgo(notification.timestamp);

        return `
            <div class="notification-item ${notification.read ? '' : 'unread'}" 
                 onclick="handleNotificationClick('${notification.id}', '${notification.type}')">
                <div class="d-flex align-items-start">
                    <div class="notification-icon ${severityClass}">
                        <i class="bi ${severityIcon}"></i>
                    </div>
                    <div class="flex-grow-1">
                        <div class="notification-title">${notification.title}</div>
                        <div class="notification-message">${notification.message && notification.message.length > 60 ? notification.message.substring(0, 60) + '...' : (notification.message || '')}</div>
                        <div class="notification-time">${timeAgo}</div>
                    </div>
                    ${!notification.read ? '<div class="text-primary" style="font-size: 8px;"><i class="bi bi-circle-fill"></i></div>' : ''}
                </div>
            </div>
        `;
    }).join('');
}

function toggleNotificationDropdown() {
    const dropdown = document.getElementById('notificationDropdown');
    const bellButton = document.getElementById('notificationBell');
    if (!dropdown || !bellButton) return;
    
    if (notificationDropdownOpen) {
        dropdown.style.display = 'none';
        notificationDropdownOpen = false;
    } else {
        // Calculate bell button position
        const bellRect = bellButton.getBoundingClientRect();
        
        // Aggressive positioning with maximum z-index
        dropdown.style.cssText = `
            display: block !important;
            position: fixed !important;
            z-index: 2147483647 !important;
            top: ${bellRect.bottom + 8}px !important;
            right: ${window.innerWidth - bellRect.right}px !important;
            left: auto !important;
            margin: 0 !important;
            padding: 0 !important;
            width: 350px !important;
            max-height: 500px !important;
            overflow: visible !important;
            visibility: visible !important;
            opacity: 1 !important;
            background: rgba(255, 255, 255, 0.95) !important;
            backdrop-filter: blur(20px) !important;
            -webkit-backdrop-filter: blur(20px) !important;
            border-radius: 12px !important;
            border: 1px solid rgba(255, 255, 255, 0.2) !important;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.4) !important;
            transform: none !important;
        `;
        
        // Ensure it doesn't go off screen
        const dropdownWidth = 350;
        if (bellRect.right < dropdownWidth) {
            dropdown.style.right = 'auto';
            dropdown.style.left = bellRect.left + 'px';
        }
        
        // Move to body if not already there
        if (dropdown.parentElement !== document.body) {
            document.body.appendChild(dropdown);
        }
        
        notificationDropdownOpen = true;
        loadRecentNotifications();
        
        // Force multiple repaints
        dropdown.offsetHeight;
        setTimeout(() => dropdown.offsetHeight, 10);
        setTimeout(() => dropdown.offsetHeight, 50);
    }
}

function handleNotificationClick(notificationId, type) {
    // Mark as read
    markNotificationAsRead(notificationId);
    
    // Navigate based on type
    switch (type) {
        case 'low_stock':
        case 'out_of_stock':
            window.location.href = 'products';
            break;
        case 'expired':
        case 'expiring_soon':
            window.location.href = 'products';
            break;
        case 'appointment':
            window.location.href = 'appointments';
            break;
        case 'payment':
            window.location.href = 'inpatients';
            break;
        default:
            window.location.href = 'notifications';
    }
    
    // Close dropdown
    toggleNotificationDropdown();
}

async function markNotificationAsRead(notificationId) {
    if (!ensureDependencies()) return;
    
    try {
        // Store read status in localStorage
        const readNotifications = JSON.parse(localStorage.getItem('readNotifications') || '[]');
        if (!readNotifications.includes(notificationId)) {
            readNotifications.push(notificationId);
            localStorage.setItem('readNotifications', JSON.stringify(readNotifications));
        }
        
        // Also call backend API
        await fetch(`/api/notifications/${notificationId}/read`, {
            method: 'PUT',
            headers: { 
                'Authorization': `Bearer ${getAuthToken()}`,
                'Content-Type': 'application/json'
            }
        });
        
        // Refresh count and list
        loadNotificationBell();
        loadRecentNotifications();
    } catch (error) {
        console.warn('Error marking notification as read:', error.message);
    }
}

async function markAllNotificationsRead() {
    if (!ensureDependencies()) return;
    
    try {
        // Get all current notification IDs and mark them as read
        const response = await fetch('/api/notifications', {
            headers: { 
                'Authorization': `Bearer ${getAuthToken()}`,
                'Content-Type': 'application/json'
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            if (data.success && data.data) {
                const allNotificationIds = data.data.map(n => n.id);
                localStorage.setItem('readNotifications', JSON.stringify(allNotificationIds));
            }
        }
        
        // Call backend API
        await fetch('/api/notifications/mark-all-read', {
            method: 'PUT',
            headers: { 
                'Authorization': `Bearer ${getAuthToken()}`,
                'Content-Type': 'application/json'
            }
        });
        
        // Refresh count and list
        loadNotificationBell();
        loadRecentNotifications();
    } catch (error) {
        console.warn('Error marking all notifications as read:', error.message);
    }
}

function getSeverityClass(severity) {
    const classes = {
        critical: 'critical',
        warning: 'warning',
        info: 'info',
        success: 'success'
    };
    return classes[severity] || 'info';
}

function getSeverityIcon(severity) {
    const icons = {
        critical: 'bi-exclamation-triangle-fill',
        warning: 'bi-exclamation-circle-fill',
        info: 'bi-info-circle-fill',
        success: 'bi-check-circle-fill'
    };
    return icons[severity] || 'bi-info-circle-fill';
}

function getTimeAgo(timestamp) {
    try {
        const now = new Date();
        const time = new Date(timestamp);
        const diffMs = now - time;
        const diffMins = Math.floor(diffMs / 60000);
        const diffHours = Math.floor(diffMins / 60);
        const diffDays = Math.floor(diffHours / 24);

        if (diffMins < 1) return 'Just now';
        if (diffMins < 60) return `${diffMins}m ago`;
        if (diffHours < 24) return `${diffHours}h ago`;
        if (diffDays < 7) return `${diffDays}d ago`;
        return time.toLocaleDateString();
    } catch (error) {
        return 'Recently';
    }
}

// Close dropdown when clicking outside
function handleOutsideClick(event) {
    if (!notificationDropdownOpen) return;
    
    const container = document.querySelector('.notification-bell-container');
    const dropdown = document.getElementById('notificationDropdown');
    
    // Check if click is outside both the bell container and dropdown
    if (container && dropdown && 
        !container.contains(event.target) && 
        !dropdown.contains(event.target)) {
        dropdown.style.display = 'none';
        notificationDropdownOpen = false;
    }
}

// Reposition dropdown on window resize
function repositionDropdown() {
    if (!notificationDropdownOpen) return;
    
    const dropdown = document.getElementById('notificationDropdown');
    const bellButton = document.getElementById('notificationBell');
    if (!dropdown || !bellButton) return;
    
    const bellRect = bellButton.getBoundingClientRect();
    dropdown.style.top = (bellRect.bottom + 8) + 'px';
    dropdown.style.right = (window.innerWidth - bellRect.right) + 'px';
    dropdown.style.zIndex = '2147483647';
    dropdown.style.position = 'fixed';
    
    // Ensure it doesn't go off screen
    const dropdownWidth = 350;
    if (bellRect.right < dropdownWidth) {
        dropdown.style.right = 'auto';
        dropdown.style.left = bellRect.left + 'px';
    }
    
    // Ensure it's still on body
    if (dropdown.parentElement !== document.body) {
        document.body.appendChild(dropdown);
    }
}

// Move dropdown to body to avoid container issues
function moveDropdownToBody() {
    const dropdown = document.getElementById('notificationDropdown');
    if (dropdown && dropdown.parentElement !== document.body) {
        // Clone the dropdown to body with same ID
        const clonedDropdown = dropdown.cloneNode(true);
        document.body.appendChild(clonedDropdown);
        dropdown.remove();
    }
}

// Initialize notification bell when DOM is loaded
function initializeNotificationBell() {
    // Set up outside click handler
    document.addEventListener('click', handleOutsideClick);
    
    // Set up window resize handler
    window.addEventListener('resize', repositionDropdown);
    
    // Move dropdown to body after a short delay to ensure it's loaded
    setTimeout(moveDropdownToBody, 100);
    
    // Load initial notification count
    loadNotificationBell();
    
    // Auto-refresh every 60 seconds (reduced frequency to prevent rate limiting)
    setInterval(loadNotificationBell, 60000);
    
    console.log('Notification bell initialized successfully');
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeNotificationBell);
} else {
    // DOM already loaded
    initializeNotificationBell();
}

// Export functions for global access
window.loadNotificationBell = loadNotificationBell;
window.loadRecentNotifications = loadRecentNotifications;
window.toggleNotificationDropdown = toggleNotificationDropdown;
window.handleNotificationClick = handleNotificationClick;
window.markNotificationAsRead = markNotificationAsRead;
window.markAllNotificationsRead = markAllNotificationsRead;
