/**
 * Universal Pagination Component for Clinic Pro V3
 * iPhone Glass Theme Styling
 */

class Pagination {
    constructor(options = {}) {
        this.currentPage = options.currentPage || 1;
        this.totalPages = options.totalPages || 1;
        this.totalRecords = options.totalRecords || 0;
        this.recordsPerPage = options.recordsPerPage || 20;
        this.onPageChange = options.onPageChange || function() {};
        this.containerId = options.containerId || 'paginationContainer';
    }

    /**
     * Render pagination controls
     */
    render() {
        const container = document.getElementById(this.containerId);
        if (!container) {
            console.error(`Pagination container #${this.containerId} not found`);
            return;
        }

        // Always show pagination info, even if only 1 page
        if (this.totalRecords === 0) {
            container.innerHTML = '';
            return;
        }

        const startRecord = (this.currentPage - 1) * this.recordsPerPage + 1;
        const endRecord = Math.min(this.currentPage * this.recordsPerPage, this.totalRecords);

        let html = `
            <div class="d-flex justify-content-between align-items-center mt-4">
                <div style="color: #1e3a8a; font-weight: 500;">
                    Showing ${startRecord} to ${endRecord} of ${this.totalRecords} records
                </div>`;

        // Only show navigation controls if there are multiple pages
        if (this.totalPages > 1) {
            html += `
                <nav aria-label="Page navigation">
                    <ul class="pagination pagination-glass mb-0">
            `;

            // Previous button
            html += `
                <li class="page-item ${this.currentPage === 1 ? 'disabled' : ''}">
                    <a class="page-link" href="#" data-page="${this.currentPage - 1}">
                        <i class="bi bi-chevron-left"></i> Previous
                    </a>
                </li>
            `;

            // Page numbers
            const pageNumbers = this.getPageNumbers();
            pageNumbers.forEach(page => {
                if (page === '...') {
                    html += `
                        <li class="page-item disabled">
                            <span class="page-link">...</span>
                        </li>
                    `;
                } else {
                    html += `
                        <li class="page-item ${page === this.currentPage ? 'active' : ''}">
                            <a class="page-link" href="#" data-page="${page}">${page}</a>
                        </li>
                    `;
                }
            });

            // Next button
            html += `
                <li class="page-item ${this.currentPage === this.totalPages ? 'disabled' : ''}">
                    <a class="page-link" href="#" data-page="${this.currentPage + 1}">
                        Next <i class="bi bi-chevron-right"></i>
                    </a>
                </li>
            `;

            html += `
                    </ul>
                </nav>`;
        } else {
            // Show page info for single page with simple navigation UI
            html += `
                <nav aria-label="Page navigation">
                    <ul class="pagination pagination-glass mb-0">
                        <li class="page-item disabled">
                            <span class="page-link">
                                <i class="bi bi-chevron-left"></i> Previous
                            </span>
                        </li>
                        <li class="page-item active">
                            <span class="page-link">1</span>
                        </li>
                        <li class="page-item disabled">
                            <span class="page-link">
                                Next <i class="bi bi-chevron-right"></i>
                            </span>
                        </li>
                    </ul>
                </nav>`;
        }

        html += `
            </div>
        `;

        container.innerHTML = html;

        // Add event listeners
        this.attachEventListeners();
    }

    /**
     * Get page numbers to display
     */
    getPageNumbers() {
        const delta = 2; // Number of pages to show on each side of current page
        const pages = [];
        
        for (let i = 1; i <= this.totalPages; i++) {
            if (
                i === 1 || 
                i === this.totalPages || 
                (i >= this.currentPage - delta && i <= this.currentPage + delta)
            ) {
                pages.push(i);
            } else if (pages[pages.length - 1] !== '...') {
                pages.push('...');
            }
        }
        
        return pages;
    }

    /**
     * Attach click event listeners to pagination links
     */
    attachEventListeners() {
        const links = document.querySelectorAll(`#${this.containerId} .page-link`);
        links.forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const page = parseInt(link.getAttribute('data-page'));
                
                if (page && page >= 1 && page <= this.totalPages && page !== this.currentPage) {
                    this.currentPage = page;
                    this.onPageChange(page);
                    this.render();
                }
            });
        });
    }

    /**
     * Update pagination state
     */
    update(currentPage, totalRecords) {
        this.currentPage = currentPage;
        this.totalRecords = totalRecords;
        this.totalPages = Math.ceil(totalRecords / this.recordsPerPage);
        this.render();
    }

    /**
     * Go to specific page
     */
    goToPage(page) {
        if (page >= 1 && page <= this.totalPages) {
            this.currentPage = page;
            this.onPageChange(page);
            this.render();
        }
    }
}

// Export for use in other scripts
if (typeof module !== 'undefined' && module.exports) {
    module.exports = Pagination;
}

