/**
 * Patient Autocomplete Component
 * Reusable patient search with autocomplete for all forms
 */

// Debounce function
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Get auth token
function getAuthToken() {
    return localStorage.getItem('token') || sessionStorage.getItem('token');
}

/**
 * Setup patient autocomplete for an input field
 * @param {string} inputId - ID of the input field
 * @param {string} suggestionBoxId - ID of the suggestions container
 * @param {string} patientNameDisplayId - ID of element to display selected patient name (optional)
 * @param {function} onSelectCallback - Callback function when patient is selected (optional)
 */
function setupPatientAutocomplete(inputId, suggestionBoxId, patientNameDisplayId = null, onSelectCallback = null) {
    const input = document.getElementById(inputId);
    const box = document.getElementById(suggestionBoxId);
    
    if (!input || !box) {
        console.error('Patient autocomplete: input or suggestion box not found');
        return;
    }

    let currentPatients = [];
    let selectedIndex = -1;

    // Render suggestions
    const renderSuggestions = () => {
        if (!currentPatients.length) {
            box.style.display = 'none';
            box.innerHTML = '';
            return;
        }
        
        box.innerHTML = currentPatients.map((patient, index) => `
            <div class="autocomplete-item ${index === selectedIndex ? 'active' : ''}" 
                 onclick="selectPatient_${inputId}('${patient.patient_id}', '${escapeHtml(patient.name || '')}')">
                <div class="fw-bold">${patient.name || '-'}</div>
                <small class="text-muted">${patient.patient_id || ''} ${patient.phone ? ' | ' + patient.phone : ''}</small>
            </div>
        `).join('');
        box.style.display = 'block';
    };

    // Search patients
    const searchPatients = debounce(async () => {
        const searchTerm = input.value.trim();
        
        if (searchTerm.length < 2) {
            currentPatients = [];
            renderSuggestions();
            return;
        }

        try {
            const token = getAuthToken();
            const response = await fetch(`/api/patients/search?q=${encodeURIComponent(searchTerm)}`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.ok) {
                const data = await response.json();
                currentPatients = data.data || [];
                selectedIndex = -1;
                renderSuggestions();
            }
        } catch (error) {
            console.error('Error searching patients:', error);
        }
    }, 300);

    // Select patient function (needs to be global for onclick)
    window[`selectPatient_${inputId}`] = (patientId, patientName) => {
        input.value = patientId;
        
        if (patientNameDisplayId) {
            const nameDisplay = document.getElementById(patientNameDisplayId);
            if (nameDisplay) {
                nameDisplay.textContent = patientName || patientId;
            }
        }
        
        box.style.display = 'none';
        currentPatients = [];
        
        if (onSelectCallback) {
            onSelectCallback(patientId, patientName);
        }
    };

    // Handle keyboard navigation
    input.addEventListener('keydown', (e) => {
        if (e.key === 'ArrowDown') {
            e.preventDefault();
            selectedIndex = Math.min(selectedIndex + 1, currentPatients.length - 1);
            renderSuggestions();
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            selectedIndex = Math.max(selectedIndex - 1, -1);
            renderSuggestions();
        } else if (e.key === 'Enter' && selectedIndex >= 0) {
            e.preventDefault();
            const selected = currentPatients[selectedIndex];
            if (selected) {
                window[`selectPatient_${inputId}`](selected.patient_id, selected.name);
            }
        } else if (e.key === 'Escape') {
            box.style.display = 'none';
            currentPatients = [];
        }
    });

    // Input event
    input.addEventListener('input', searchPatients);

    // Focus event
    input.addEventListener('focus', () => {
        if (currentPatients.length > 0) {
            box.style.display = 'block';
        }
    });

    // Click outside to close
    document.addEventListener('click', (e) => {
        if (!input.contains(e.target) && !box.contains(e.target)) {
            box.style.display = 'none';
        }
    });
}

// Escape HTML
function escapeHtml(text) {
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}

