/**
 * Centralized Sidebar Manager for Clinic Pro V3
 * Handles sidebar loading, initialization, and management across all pages
 */

class SidebarManager {
    constructor() {
        this.isLoaded = false;
        this.isInitialized = false;
        this.currentPage = this.getCurrentPage();
    }

    /**
     * Get current page name from URL
     */
    getCurrentPage() {
        const path = window.location.pathname;
        const page = path.split('/').pop().replace('.html', '');
        return page || 'dashboard';
    }

    /**
     * Initialize sidebar for the current page
     */
    async init() {
        if (this.isLoaded) {
            this.initializeSidebar();
            return;
        }

        try {
            await this.loadSidebar();
            this.initializeSidebar();
            this.isLoaded = true;
            this.isInitialized = true;
        } catch (error) {
            console.error('Error initializing sidebar:', error);
        }
    }

    /**
     * Load sidebar HTML content
     */
    async loadSidebar() {
        const sidebarContainer = document.getElementById('sidebarContainer');
        if (!sidebarContainer) {
            throw new Error('Sidebar container not found');
        }

        try {
            const response = await fetch(`../includes/sidebar.html?v=${Date.now()}`);
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            const html = await response.text();
            sidebarContainer.innerHTML = html;
        } catch (error) {
            console.error('Error loading sidebar HTML:', error);
            throw error;
        }
    }

    /**
     * Initialize sidebar functionality
     */
    initializeSidebar() {
        // Wait for DOM to be ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.setupSidebar());
        } else {
            this.setupSidebar();
        }
    }

    /**
     * Setup sidebar functionality
     */
    setupSidebar() {
        // Restore sidebar state
        this.restoreSidebarState();
        
        // Restore submenu states
        this.restoreSubmenuStates();
        
        // Auto-expand current page's group
        this.autoExpandCurrentGroup();
        
        // Update active nav link
        this.updateActiveNavLink();
        
        // Initialize sidebar tooltips
        this.initSidebarTooltips();
        
        // Initialize keyboard shortcuts
        this.initKeyboardShortcuts();
        
        // Highlight current menu
        this.highlightCurrentMenu();
        
        // Update badge counts
        this.updateBadgeCounts();
    }

    /**
     * Restore sidebar minimized state
     */
    restoreSidebarState() {
        const isMinimized = localStorage.getItem('sidebar-minimized') === 'true';
        if (isMinimized) {
            document.getElementById('sidebar')?.classList.add('sidebar-minimized');
            document.body.classList.add('sidebar-minimized');
            const toggleBtn = document.getElementById('sidebarToggleBtn');
            const icon = toggleBtn?.querySelector('i');
            if (icon) {
                icon.classList.remove('bi-chevron-left');
                icon.classList.add('bi-chevron-right');
            }
        }
    }

    /**
     * Restore submenu states
     */
    restoreSubmenuStates() {
        const groups = [
            'patient-group',
            'clinical-group', 
            'inventory-group',
            'customer-group',
            'professional-group',
            'reports-group',
            'admin-group'
        ];
        
        // First, set all chevrons to rotated since all groups are collapsed by default
        groups.forEach(groupId => {
            const chevron = document.getElementById(groupId + '-chevron');
            if (chevron) {
                chevron.classList.add('rotated');
            }
        });
        
        // Then restore saved states
        groups.forEach(groupId => {
            const savedState = localStorage.getItem('sidebar-' + groupId);
            const group = document.getElementById(groupId);
            const chevron = document.getElementById(groupId + '-chevron');
            
            if (savedState === 'expanded' && group && chevron) {
                // Expand saved groups
                group.style.display = 'block';
                group.classList.remove('collapsed');
                group.classList.add('expanded');
                chevron.classList.remove('rotated');
            } else if (group) {
                // Ensure collapsed state
                group.style.display = 'none';
                group.classList.add('collapsed');
                group.classList.remove('expanded');
            }
        });
    }

    /**
     * Auto-expand current page's group
     */
    autoExpandCurrentGroup() {
        // Map pages to groups
        const pageGroups = {
            'patients': 'patient-group',
            'inpatients': 'patient-group',
            'inpatient-sales': 'patient-group',
            'emr': 'patient-group',
            'vitals-chart': 'patient-group',
            'emr-all': 'patient-group',
            'doctors': 'clinical-group',
            'appointments': 'clinical-group',
            'products': 'inventory-group',
            'services': 'inventory-group',
            'sales': 'inventory-group',
            'purchases': 'inventory-group',
            'add-purchase': 'inventory-group',
            'customers': 'customer-group',
            'suppliers': 'customer-group',
            'expenses': 'customer-group',
            'laboratory': 'professional-group',
            'hospital-operations': 'professional-group',
            'advanced-pharmacy': 'professional-group',
            'analytics-dashboard': 'professional-group',
            'reports': 'reports-group',
            'lab-reports': 'reports-group',
            'stock-sales-report': 'reports-group',
            'service-sales-report': 'reports-group',
            'voucher-sales-report': 'reports-group',
            'doctor-performance-report': 'reports-group',
            'commission-report': 'reports-group',
            'profit-loss': 'reports-group',
            'discharged-patient-report': 'reports-group',
            'inpatient-report': 'reports-group',
            'notifications': 'admin-group',
            'users': 'admin-group',
            'backup': 'admin-group'
        };
        
        const groupToExpand = pageGroups[this.currentPage];
        if (groupToExpand) {
            const group = document.getElementById(groupToExpand);
            const chevron = document.getElementById(groupToExpand + '-chevron');
            
            if (group && chevron) {
                group.style.display = 'block';
                group.classList.remove('collapsed');
                group.classList.add('expanded');
                chevron.classList.remove('rotated');
                localStorage.setItem('sidebar-' + groupToExpand, 'expanded');
            }
        }
    }

    /**
     * Update active nav link highlighting
     */
    updateActiveNavLink() {
        document.querySelectorAll('.nav-link').forEach(link => {
            const href = link.getAttribute('href');
            if (href === this.currentPage || href === this.currentPage + '.html') {
                link.classList.add('active');
            } else {
                link.classList.remove('active');
            }
        });
    }

    /**
     * Initialize sidebar tooltips for minimized mode
     */
    initSidebarTooltips() {
        const navLinks = document.querySelectorAll('.nav-link');
        navLinks.forEach(link => {
            const linkTitle = link.querySelector('.link-title');
            if (linkTitle) {
                link.setAttribute('data-tooltip', linkTitle.textContent.trim());
            } else {
                const span = link.querySelector('span:not(.badge):not(.keyboard-hint)');
                if (span) {
                    link.setAttribute('data-tooltip', span.textContent.trim());
                }
            }
        });
        
        const groupHeaders = document.querySelectorAll('.nav-group-header');
        groupHeaders.forEach(header => {
            const groupTitle = header.querySelector('.group-title');
            if (groupTitle) {
                header.setAttribute('data-tooltip', groupTitle.textContent.trim());
            } else {
                const span = header.querySelector('span');
                if (span) {
                    header.setAttribute('data-tooltip', span.textContent.trim());
                }
            }
        });
    }

    /**
     * Initialize keyboard shortcuts
     */
    initKeyboardShortcuts() {
        document.addEventListener('keydown', (e) => {
            // Ctrl+B - Toggle sidebar
            if (e.ctrlKey && e.key === 'b') {
                e.preventDefault();
                this.toggleSidebarCollapse();
            }
            // Alt+D - Dashboard
            else if (e.altKey && e.key === 'd') {
                e.preventDefault();
                window.location.href = 'dashboard';
            }
            // Alt+P - Point of Sale
            else if (e.altKey && e.key === 'p') {
                e.preventDefault();
                window.location.href = 'pos';
            }
            // Escape - Clear search
            else if (e.key === 'Escape') {
                const searchInput = document.getElementById('sidebarSearch');
                if (searchInput && searchInput.value) {
                    this.clearSearch();
                }
            }
        });
    }

    /**
     * Highlight current menu based on URL
     */
    highlightCurrentMenu() {
        const navLinks = document.querySelectorAll('.nav-link');
        
        navLinks.forEach(link => {
            const href = link.getAttribute('href');
            if (href === this.currentPage || href === this.currentPage + '.html') {
                link.classList.add('active');
                // Auto-expand parent group
                const parentGroup = link.closest('.nav-group-items');
                if (parentGroup) {
                    parentGroup.style.display = 'block';
                    const groupId = parentGroup.id;
                    const chevron = document.getElementById(groupId + '-chevron');
                    if (chevron) {
                        chevron.classList.remove('rotated');
                    }
                }
            }
        });
    }

    /**
     * Update badge counts (placeholder - can be connected to real data)
     */
    updateBadgeCounts() {
        // This function can be called from API to update counts
        // Example: sidebarManager.updateBadgeCount('opd-count', 5);
    }

    /**
     * Update specific badge count
     */
    updateBadgeCount(badgeId, count) {
        const badge = document.getElementById(badgeId);
        if (badge) {
            badge.textContent = count;
            if (count > 0) {
                badge.style.display = 'flex';
            } else {
                badge.style.display = 'none';
            }
        }
    }

    /**
     * Toggle sidebar collapse
     */
    toggleSidebarCollapse() {
        const sidebar = document.getElementById('sidebar');
        const body = document.body;
        const toggleBtn = document.getElementById('sidebarToggleBtn');
        
        if (!sidebar || !toggleBtn) return;
        
        sidebar.classList.toggle('sidebar-minimized');
        body.classList.toggle('sidebar-minimized');
        
        // Update icon
        const icon = toggleBtn.querySelector('i');
        if (sidebar.classList.contains('sidebar-minimized')) {
            icon.classList.remove('bi-chevron-left');
            icon.classList.add('bi-chevron-right');
            localStorage.setItem('sidebar-minimized', 'true');
        } else {
            icon.classList.remove('bi-chevron-right');
            icon.classList.add('bi-chevron-left');
            localStorage.setItem('sidebar-minimized', 'false');
        }
    }

    /**
     * Clear search
     */
    clearSearch() {
        const searchInput = document.getElementById('sidebarSearch');
        if (searchInput) {
            searchInput.value = '';
            this.filterSidebarMenu('');
            searchInput.focus();
        }
    }

    /**
     * Filter sidebar menu based on search term
     */
    filterSidebarMenu(searchTerm) {
        const nav = document.getElementById('sidebarNav');
        const searchClear = document.getElementById('searchClear');
        searchTerm = searchTerm.toLowerCase().trim();
        
        // Show/hide clear button
        if (searchClear) {
            searchClear.style.display = searchTerm ? 'flex' : 'none';
        }
        
        if (!searchTerm) {
            // Reset all visibility
            document.querySelectorAll('.nav-item, .nav-group').forEach(el => {
                el.style.display = '';
                el.classList.remove('search-highlight');
            });
            return;
        }
        
        let hasResults = false;
        
        // Search through all nav links
        document.querySelectorAll('.nav-item').forEach(item => {
            const link = item.querySelector('.nav-link');
            if (link) {
                const text = link.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    item.style.display = '';
                    item.classList.add('search-highlight');
                    hasResults = true;
                } else {
                    item.style.display = 'none';
                    item.classList.remove('search-highlight');
                }
            }
        });
        
        // Handle groups
        document.querySelectorAll('.nav-group').forEach(group => {
            const visibleItems = group.querySelectorAll('.nav-item:not([style*="display: none"])');
            const groupHeader = group.querySelector('.nav-group-header');
            
            if (visibleItems.length > 0) {
                group.style.display = '';
                // Auto-expand groups with matches
                const groupItems = group.querySelector('.nav-group-items');
                if (groupItems) {
                    groupItems.style.display = 'block';
                }
            } else {
                // Check if group header matches
                const headerText = groupHeader.textContent.toLowerCase();
                if (headerText.includes(searchTerm)) {
                    group.style.display = '';
                    hasResults = true;
                } else {
                    group.style.display = 'none';
                }
            }
        });
    }

    /**
     * Toggle group collapse/expand
     */
    toggleGroup(groupId) {
        const group = document.getElementById(groupId);
        const chevron = document.getElementById(groupId + '-chevron');
        
        if (!group || !chevron) return;
        
        const isExpanded = group.style.display !== 'none';
        
        if (isExpanded) {
            // Collapse
            group.style.display = 'none';
            group.classList.add('collapsed');
            group.classList.remove('expanded');
            chevron.classList.add('rotated');
            
            // Save collapsed state
            localStorage.setItem('sidebar-' + groupId, 'collapsed');
        } else {
            // Expand
            group.style.display = 'block';
            group.classList.remove('collapsed');
            group.classList.add('expanded');
            chevron.classList.remove('rotated');
            
            // Save expanded state
            localStorage.setItem('sidebar-' + groupId, 'expanded');
        }
    }
}

// Global functions for backward compatibility
window.toggleSidebarCollapse = function() {
    if (window.sidebarManager) {
        window.sidebarManager.toggleSidebarCollapse();
    }
};

window.clearSearch = function() {
    if (window.sidebarManager) {
        window.sidebarManager.clearSearch();
    }
};

window.toggleGroup = function(groupId) {
    if (window.sidebarManager) {
        window.sidebarManager.toggleGroup(groupId);
    }
};

window.updateBadgeCount = function(badgeId, count) {
    if (window.sidebarManager) {
        window.sidebarManager.updateBadgeCount(badgeId, count);
    }
};

// Initialize sidebar manager
window.sidebarManager = new SidebarManager();

// Auto-initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        window.sidebarManager.init();
    });
} else {
    window.sidebarManager.init();
}
