// Sidebar functionality for Clinic Pro V3 - Enhanced

// Toggle group collapse/expand - Global function
window.toggleGroup = function(groupId) {
    const group = document.getElementById(groupId);
    const chevron = document.getElementById(groupId + '-chevron');
    
    if (!group || !chevron) return;
    
    const isExpanded = group.style.display !== 'none';
    
    if (isExpanded) {
        // Collapse
        group.style.display = 'none';
        group.classList.add('collapsed');
        group.classList.remove('expanded');
        chevron.classList.add('rotated');
        
        // Save collapsed state
        localStorage.setItem('sidebar-' + groupId, 'collapsed');
    } else {
        // Expand
        group.style.display = 'block';
        group.classList.remove('collapsed');
        group.classList.add('expanded');
        chevron.classList.remove('rotated');
        
        // Save expanded state
        localStorage.setItem('sidebar-' + groupId, 'expanded');
    }
};

// Toggle sidebar collapse (icon-only mode) - Global function
window.toggleSidebarCollapse = function() {
    const sidebar = document.getElementById('sidebar');
    const body = document.body;
    const toggleBtn = document.getElementById('sidebarToggleBtn');
    
    if (!sidebar || !body || !toggleBtn) return;
    
    sidebar.classList.toggle('sidebar-minimized');
    body.classList.toggle('sidebar-minimized');
    
    // Update icon
    const icon = toggleBtn.querySelector('i');
    if (icon && sidebar.classList.contains('sidebar-minimized')) {
        icon.classList.remove('bi-chevron-left');
        icon.classList.add('bi-chevron-right');
        localStorage.setItem('sidebar-minimized', 'true');
    } else if (icon) {
        icon.classList.remove('bi-chevron-right');
        icon.classList.add('bi-chevron-left');
        localStorage.setItem('sidebar-minimized', 'false');
    }
};

// Clear search - Global function
window.clearSearch = function() {
    const searchInput = document.getElementById('sidebarSearch');
    if (searchInput) {
        searchInput.value = '';
        if (typeof filterSidebarMenu === 'function') {
            filterSidebarMenu('');
        }
        searchInput.focus();
    }
};

// Get current page
function getCurrentPage() {
    const path = window.location.pathname;
    const page = path.split('/').pop() || 'dashboard.html';
    return page;
}

// Set active menu item
function setActiveMenuItem() {
    const currentPage = getCurrentPage();
    const navLinks = document.querySelectorAll('.nav-link');
    
    navLinks.forEach(link => {
        const href = link.getAttribute('href');
        if (href === currentPage || href === `./${currentPage}`) {
            link.classList.add('active');
        } else {
            link.classList.remove('active');
        }
    });
}

// Toggle sidebar (for mobile) - Global function
window.toggleSidebar = function() {
    const sidebar = document.getElementById('sidebar');
    const body = document.body;
    
    if (sidebar && body) {
        sidebar.classList.toggle('show');
        
        // Add backdrop for mobile
        let backdrop = document.querySelector('.sidebar-backdrop');
        if (sidebar.classList.contains('show')) {
            if (!backdrop) {
                backdrop = document.createElement('div');
                backdrop.className = 'sidebar-backdrop';
                backdrop.style.cssText = `
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0, 0, 0, 0.5);
                    z-index: 999;
                    display: none;
                `;
                document.body.appendChild(backdrop);
                backdrop.addEventListener('click', window.toggleSidebar);
            }
            if (window.innerWidth <= 768) {
                backdrop.style.display = 'block';
                body.style.overflow = 'hidden';
            }
        } else {
            if (backdrop) {
                backdrop.style.display = 'none';
            }
            body.style.overflow = '';
        }
    }
};

// Logout function - Make globally accessible
window.logout = function() {
    if (confirm('Are you sure you want to logout?')) {
        // Get token before clearing
        const token = localStorage.getItem('authToken') || localStorage.getItem('token') || sessionStorage.getItem('token');

        // Clear ALL storage to prevent auto-login
        localStorage.clear();
        sessionStorage.clear();
        ['authToken', 'token', 'user', 'language', 'sidebar-minimized'].forEach(key => {
            localStorage.removeItem(key);
            sessionStorage.removeItem(key);
        });

        // Call backend logout (fire-and-forget)
        if (token) {
            fetch('/api/auth/logout', {
                method: 'GET',
                headers: { 'Authorization': `Bearer ${token}` },
                keepalive: true
            }).catch(err => console.error('Logout API error:', err));
        }

        // Set logout flags so login page clears cached data
        try {
            sessionStorage.setItem('LOGOUT_FLAG', '1');
            document.cookie = 'LOGOUT_FLAG=1; path=/';
        } catch (e) {}

        // Redirect to login page
        window.location.replace('/?logout=' + Date.now());
    }
};

// Backward compatibility for older templates
window.handleLogout = window.logout;

// Mobile menu toggle button handler
function initMobileToggle() {
    const mobileToggle = document.getElementById('mobileMenuToggle');
    if (mobileToggle) {
        mobileToggle.addEventListener('click', toggleSidebar);
    }
}

// Handle window resize
function handleResize() {
    const sidebar = document.getElementById('sidebar');
    const backdrop = document.querySelector('.sidebar-backdrop');
    
    if (window.innerWidth > 768) {
        if (sidebar) {
            sidebar.classList.remove('show');
        }
        if (backdrop) {
            backdrop.style.display = 'none';
        }
    }
}

// Restore sidebar state on page load - Global function
window.restoreSubmenuStates = function() {
    const groups = [
        'patient-group',
        'clinical-group', 
        'inventory-group',
        'warehouse-group',
        'customer-group',
        'professional-group',
        'reports-group',
        'admin-group'
    ];
    
    // First, set all chevrons to rotated since all groups are collapsed by default
    groups.forEach(groupId => {
        const chevron = document.getElementById(groupId + '-chevron');
        if (chevron) {
            chevron.classList.add('rotated');
        }
    });
    
    // Then restore saved states
    groups.forEach(groupId => {
        const savedState = localStorage.getItem('sidebar-' + groupId);
        const group = document.getElementById(groupId);
        const chevron = document.getElementById(groupId + '-chevron');
        
        if (savedState === 'expanded' && group && chevron) {
            // Expand saved groups
            group.style.display = 'block';
            group.classList.remove('collapsed');
            group.classList.add('expanded');
            chevron.classList.remove('rotated');
        } else if (group) {
            // Ensure collapsed state
            group.style.display = 'none';
            group.classList.add('collapsed');
            group.classList.remove('expanded');
        }
    });
};

// Auto-expand current page's group - Global function
window.autoExpandCurrentGroup = function() {
    const currentPage = window.location.pathname.split('/').pop().replace('.html', '');
    
    // Map pages to groups
    const pageGroups = {
        'patients': 'patient-group',
        'inpatients': 'patient-group',
        'emr': 'patient-group',
        'vitals-chart': 'patient-group',
        'emr-all': 'patient-group',
        'doctors': 'clinical-group',
        'appointments': 'clinical-group',
        'products': 'inventory-group',
        'barcode-labels': 'inventory-group',
        'services': 'inventory-group',
        'sales': 'inventory-group',
        'purchases': 'inventory-group',
        'add-purchase': 'inventory-group',
        'warehouses': 'warehouse-group',
        'warehouse-stock-assignment': 'warehouse-group',
        'warehouse-transfers': 'warehouse-group',
        'warehouse-stock': 'warehouse-group',
        'warehouse-history': 'warehouse-group',
        'customers': 'customer-group',
        'suppliers': 'customer-group',
        'expenses': 'customer-group',
        'laboratory': 'professional-group',
        'hospital-operations': 'professional-group',
        'advanced-pharmacy': 'professional-group',
        'analytics-dashboard': 'professional-group',
        'stock-sales-report': 'reports-group',
        'service-sales-report': 'reports-group',
        'voucher-sales-report': 'reports-group',
        'reports': 'reports-group',
        'lab-reports': 'reports-group',
        'notifications': 'admin-group',
        'users': 'admin-group',
        'backup': 'admin-group'
    };
    
    const groupToExpand = pageGroups[currentPage];
    if (groupToExpand) {
        const group = document.getElementById(groupToExpand);
        const chevron = document.getElementById(groupToExpand + '-chevron');
        
        if (group && chevron) {
            group.style.display = 'block';
            group.classList.remove('collapsed');
            group.classList.add('expanded');
            chevron.classList.remove('rotated');
            localStorage.setItem('sidebar-' + groupToExpand, 'expanded');
        }
    }
};

// Update active nav link highlighting - Global function
window.updateActiveNavLink = function() {
    const currentPage = window.location.pathname.split('/').pop().replace('.html', '');
    document.querySelectorAll('.nav-link').forEach(link => {
        const href = link.getAttribute('href');
        if (href === currentPage || href === currentPage + '.html') {
            link.classList.add('active');
        } else {
            link.classList.remove('active');
        }
    });
};

// Restore sidebar minimized state - Global function
window.restoreSidebarState = function() {
    const isMinimized = localStorage.getItem('sidebar-minimized') === 'true';
    if (isMinimized) {
        const sidebar = document.getElementById('sidebar');
        const body = document.body;
        if (sidebar && body) {
            sidebar.classList.add('sidebar-minimized');
            body.classList.add('sidebar-minimized');
            const toggleBtn = document.getElementById('sidebarToggleBtn');
            if (toggleBtn) {
                const icon = toggleBtn.querySelector('i');
                if (icon) {
                    icon.classList.remove('bi-chevron-left');
                    icon.classList.add('bi-chevron-right');
                }
            }
        }
    }
};

// Initialize sidebar when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    setActiveMenuItem();
    initMobileToggle();
    
    // Add event listener for sidebar toggle button if it exists
    const toggleBtn = document.getElementById('sidebarToggle');
    if (toggleBtn) {
        toggleBtn.addEventListener('click', toggleSidebar);
    }
    
    // Handle window resize
    window.addEventListener('resize', handleResize);

    // Load shop logo/name into sidebar
    updateSidebarShopInfo();
});

async function updateSidebarShopInfo() {
    try {
        // Use the same endpoint as invoices to get shop info
        const resp = await fetch('/api/settings/shop', {
            headers: {
                'Authorization': `Bearer ${typeof getAuthToken === 'function' ? getAuthToken() : localStorage.getItem('token') || ''}`
            }
        });
        const result = await resp.json();

        if (result && result.success && result.data) {
            const logoPath = result.data.photo_path || '';
            const shopName = result.data.name || 'Clinic Pro V3';
            const shopType = result.data.type || 'Healthcare Management';

            const logoImg = document.getElementById('sidebarShopLogo');
            const fallbackIcon = document.getElementById('sidebarFallbackIcon');
            const nameEl = document.getElementById('sidebarShopName');
            const typeEl = document.getElementById('sidebarShopType');

            if (nameEl) nameEl.textContent = shopName;
            if (typeEl) typeEl.textContent = shopType;

            if (logoImg && logoPath) {
                // If logoPath is relative, prepend origin
                const fullPath = logoPath.startsWith('http') ? logoPath : `${window.location.origin}${logoPath}`;
                logoImg.src = fullPath;
                logoImg.onload = () => {
                    logoImg.style.display = 'block';
                    if (fallbackIcon) fallbackIcon.style.display = 'none';
                };
                logoImg.onerror = () => {
                    logoImg.style.display = 'none';
                    if (fallbackIcon) fallbackIcon.style.display = 'inline-block';
                };
            }
        }
    } catch (err) {
        console.warn('Could not load shop info for sidebar:', err);
    }
}

