/**
 * Centralized Toast Notification System
 * Provides consistent user feedback across the application
 */

class ToastManager {
    constructor() {
        this.container = null;
        this.toasts = new Map();
        this.maxToasts = 5;
        this.init();
    }

    init() {
        // Create toast container if it doesn't exist
        if (!document.getElementById('toast-container-global')) {
            this.container = document.createElement('div');
            this.container.id = 'toast-container-global';
            this.container.className = 'toast-container-global';
            document.body.appendChild(this.container);
        } else {
            this.container = document.getElementById('toast-container-global');
        }
    }

    show(message, type = 'info', duration = 3000, options = {}) {
        const toastId = `toast-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
        const toast = this.createToast(toastId, message, type, options);
        
        this.container.appendChild(toast);
        this.toasts.set(toastId, toast);

        // Limit number of toasts
        if (this.toasts.size > this.maxToasts) {
            const firstToast = this.toasts.values().next().value;
            this.remove(firstToast.id);
        }

        // Trigger animation
        requestAnimationFrame(() => {
            toast.classList.add('show');
        });

        // Auto remove
        if (duration > 0) {
            setTimeout(() => {
                this.remove(toastId);
            }, duration);
        }

        return toastId;
    }

    createToast(id, message, type, options) {
        const toast = document.createElement('div');
        toast.id = id;
        toast.className = `toast-item toast-${type}`;
        toast.setAttribute('role', 'alert');
        toast.setAttribute('aria-live', 'assertive');
        toast.setAttribute('aria-atomic', 'true');

        const icons = {
            success: 'bi-check-circle-fill',
            error: 'bi-x-circle-fill',
            warning: 'bi-exclamation-triangle-fill',
            info: 'bi-info-circle-fill'
        };

        const icon = options.icon || icons[type] || icons.info;
        const title = options.title ? `<div class="toast-title">${options.title}</div>` : '';
        const action = options.action ? `<button class="toast-action" onclick="${options.action}">${options.actionLabel || 'Action'}</button>` : '';

        toast.innerHTML = `
            <div class="toast-content">
                <i class="bi ${icon} toast-icon"></i>
                <div class="toast-message">
                    ${title}
                    <div class="toast-text">${message}</div>
                </div>
                ${action}
                <button class="toast-close" onclick="toastManager.remove('${id}')" aria-label="Close">
                    <i class="bi bi-x"></i>
                </button>
            </div>
            ${options.progress ? '<div class="toast-progress"></div>' : ''}
        `;

        // Add click handler for dismissible toasts
        if (options.dismissible !== false) {
            toast.addEventListener('click', (e) => {
                if (e.target.closest('.toast-close') || options.clickToDismiss) {
                    this.remove(id);
                }
            });
        }

        return toast;
    }

    remove(id) {
        const toast = this.toasts.get(id);
        if (toast) {
            toast.classList.add('hide');
            setTimeout(() => {
                toast.remove();
                this.toasts.delete(id);
            }, 300);
        }
    }

    success(message, duration = 3000, options = {}) {
        return this.show(message, 'success', duration, options);
    }

    error(message, duration = 5000, options = {}) {
        return this.show(message, 'error', duration, options);
    }

    warning(message, duration = 4000, options = {}) {
        return this.show(message, 'warning', duration, options);
    }

    info(message, duration = 3000, options = {}) {
        return this.show(message, 'info', duration, options);
    }

    clear() {
        this.toasts.forEach((toast, id) => {
            this.remove(id);
        });
    }
}

// Initialize global toast manager
const toastManager = new ToastManager();

// Global convenience functions
function showToast(message, type = 'info', duration = 3000, options = {}) {
    return toastManager.show(message, type, duration, options);
}

function showSuccess(message, duration = 3000, options = {}) {
    return toastManager.success(message, duration, options);
}

function showError(message, duration = 5000, options = {}) {
    return toastManager.error(message, duration, options);
}

function showWarning(message, duration = 4000, options = {}) {
    return toastManager.warning(message, duration, options);
}

function showInfo(message, duration = 3000, options = {}) {
    return toastManager.info(message, duration, options);
}

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { toastManager, showToast, showSuccess, showError, showWarning, showInfo };
}

