/**
 * UX Utilities - Centralized User Experience Enhancements
 * Provides consistent UX patterns across the application
 */

// ============================================
// Confirmation Dialog System (Replaces alert/confirm)
// ============================================

class ConfirmationDialog {
    constructor() {
        this.modalId = 'ux-confirmation-modal';
        this.init();
    }

    init() {
        // Create modal if it doesn't exist
        if (!document.getElementById(this.modalId)) {
            const modal = document.createElement('div');
            modal.id = this.modalId;
            modal.className = 'modal fade';
            modal.innerHTML = `
                <div class="modal-dialog modal-dialog-centered">
                    <div class="modal-content glass-modal">
                        <div class="modal-header border-0">
                            <h5 class="modal-title" id="ux-confirmation-title">
                                <i class="bi bi-question-circle text-primary"></i> Confirm Action
                            </h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <p id="ux-confirmation-message">Are you sure you want to proceed?</p>
                            <div id="ux-confirmation-details" class="mt-3" style="display: none;"></div>
                        </div>
                        <div class="modal-footer border-0">
                            <button type="button" class="btn btn-glass" data-bs-dismiss="modal">
                                <i class="bi bi-x-circle me-2"></i>Cancel
                            </button>
                            <button type="button" class="btn btn-glass-danger" id="ux-confirmation-confirm">
                                <i class="bi bi-check-circle me-2"></i>Confirm
                            </button>
                        </div>
                    </div>
                </div>
            `;
            document.body.appendChild(modal);
        }
    }

    show(options = {}) {
        return new Promise((resolve, reject) => {
            const modal = document.getElementById(this.modalId);
            const titleEl = document.getElementById('ux-confirmation-title');
            const messageEl = document.getElementById('ux-confirmation-message');
            const detailsEl = document.getElementById('ux-confirmation-details');
            const confirmBtn = document.getElementById('ux-confirmation-confirm');

            // Set content
            if (options.title) {
                titleEl.innerHTML = `<i class="bi ${options.icon || 'bi-question-circle'} text-${options.type || 'primary'}"></i> ${options.title}`;
            }
            messageEl.textContent = options.message || 'Are you sure you want to proceed?';
            
            // Set details if provided
            if (options.details) {
                detailsEl.innerHTML = options.details;
                detailsEl.style.display = 'block';
            } else {
                detailsEl.style.display = 'none';
            }

            // Set button text
            if (options.confirmText) {
                confirmBtn.innerHTML = `<i class="bi ${options.confirmIcon || 'bi-check-circle'} me-2"></i>${options.confirmText}`;
            }

            // Set button style based on type
            confirmBtn.className = `btn btn-glass-${options.type || 'danger'}`;

            // Remove old event listeners
            const newConfirmBtn = confirmBtn.cloneNode(true);
            confirmBtn.parentNode.replaceChild(newConfirmBtn, confirmBtn);

            // Add new event listener
            newConfirmBtn.addEventListener('click', () => {
                bootstrap.Modal.getInstance(modal).hide();
                resolve(true);
            });

            // Handle cancel
            modal.addEventListener('hidden.bs.modal', () => {
                resolve(false);
            }, { once: true });

            // Show modal
            const bsModal = new bootstrap.Modal(modal);
            bsModal.show();
        });
    }
}

const confirmationDialog = new ConfirmationDialog();

// Convenience functions
async function confirmAction(message, options = {}) {
    return await confirmationDialog.show({
        message,
        ...options
    });
}

async function confirmDelete(itemName, options = {}) {
    return await confirmationDialog.show({
        title: 'Confirm Delete',
        message: `Are you sure you want to delete "${itemName}"?`,
        details: '<div class="alert alert-warning mb-0"><small><i class="bi bi-exclamation-triangle"></i> This action cannot be undone.</small></div>',
        type: 'danger',
        icon: 'bi-exclamation-triangle',
        confirmText: 'Delete',
        confirmIcon: 'bi-trash',
        ...options
    });
}

// ============================================
// Loading State Manager
// ============================================

class LoadingManager {
    constructor() {
        this.loadingOverlays = new Map();
    }

    show(element, message = 'Loading...') {
        const id = element.id || `loading-${Date.now()}`;
        
        // Remove existing overlay if any
        this.hide(element);

        const overlay = document.createElement('div');
        overlay.className = 'loading-overlay';
        overlay.id = `loading-${id}`;
        overlay.innerHTML = `
            <div class="loading-spinner">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-3 text-white">${message}</p>
            </div>
        `;

        // Make parent relative if not already
        const position = window.getComputedStyle(element).position;
        if (position === 'static') {
            element.style.position = 'relative';
        }

        element.appendChild(overlay);
        this.loadingOverlays.set(id, overlay);

        return id;
    }

    hide(element) {
        const id = element.id || Array.from(this.loadingOverlays.keys()).find(key => 
            element.contains(document.getElementById(`loading-${key}`))
        );
        
        if (id) {
            const overlay = document.getElementById(`loading-${id}`);
            if (overlay) {
                overlay.remove();
                this.loadingOverlays.delete(id);
            }
        } else {
            // Try to find and remove any loading overlay in element
            const overlay = element.querySelector('.loading-overlay');
            if (overlay) {
                overlay.remove();
            }
        }
    }

    hideAll() {
        // Remove all loading overlays
        this.loadingOverlays.forEach((overlay, id) => {
            if (overlay && overlay.parentNode) {
                overlay.remove();
            }
        });
        this.loadingOverlays.clear();
        
        // Also remove any orphaned loading overlays in the DOM
        document.querySelectorAll('.loading-overlay').forEach(overlay => {
            overlay.remove();
        });
    }

    showButton(button, loadingText = 'Loading...') {
        if (!button) return null;
        
        const originalHTML = button.innerHTML;
        const originalDisabled = button.disabled;
        
        button.disabled = true;
        button.innerHTML = `
            <span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
            ${loadingText}
        `;

        return {
            restore: () => {
                button.disabled = originalDisabled;
                button.innerHTML = originalHTML;
            }
        };
    }
}

const loadingManager = new LoadingManager();

// Export LoadingManager immediately to ensure it's available
// This prevents issues if later code has errors
try {
    window.loadingManager = loadingManager;
    window.LoadingManager = loadingManager;
} catch (e) {
    console.error('Error exporting LoadingManager:', e);
}

// ============================================
// Form Validation Helper
// ============================================

class FormValidator {
    static validateField(field, rules = {}) {
        const value = field.value.trim();
        const errors = [];

        // Required validation
        if (rules.required && !value) {
            errors.push(rules.requiredMessage || 'This field is required');
        }

        // Min length
        if (rules.minLength && value.length < rules.minLength) {
            errors.push(rules.minLengthMessage || `Minimum ${rules.minLength} characters required`);
        }

        // Max length
        if (rules.maxLength && value.length > rules.maxLength) {
            errors.push(rules.maxLengthMessage || `Maximum ${rules.maxLength} characters allowed`);
        }

        // Pattern (regex)
        if (rules.pattern && value && !rules.pattern.test(value)) {
            errors.push(rules.patternMessage || 'Invalid format');
        }

        // Custom validator
        if (rules.validator && typeof rules.validator === 'function') {
            const customError = rules.validator(value);
            if (customError) {
                errors.push(customError);
            }
        }

        // Update UI
        this.updateFieldState(field, errors.length === 0, errors[0]);

        return {
            valid: errors.length === 0,
            errors
        };
    }

    static updateFieldState(field, isValid, errorMessage) {
        // Remove existing feedback
        field.classList.remove('is-valid', 'is-invalid');
        const existingFeedback = field.parentElement.querySelector('.invalid-feedback, .valid-feedback');
        if (existingFeedback) {
            existingFeedback.remove();
        }

        if (isValid) {
            field.classList.add('is-valid');
        } else {
            field.classList.add('is-invalid');
            const feedback = document.createElement('div');
            feedback.className = 'invalid-feedback';
            feedback.textContent = errorMessage || 'Please check this field';
            field.parentElement.appendChild(feedback);
        }
    }

    static validateForm(form, rules) {
        let isValid = true;
        const errors = {};

        Object.keys(rules).forEach(fieldName => {
            const field = form.querySelector(`[name="${fieldName}"]`);
            if (field) {
                const result = this.validateField(field, rules[fieldName]);
                if (!result.valid) {
                    isValid = false;
                    errors[fieldName] = result.errors;
                }
            }
        });

        return {
            valid: isValid,
            errors
        };
    }
}

// ============================================
// Keyboard Shortcuts Manager
// ============================================

class KeyboardShortcuts {
    constructor() {
        this.shortcuts = new Map();
        this.init();
    }

    init() {
        document.addEventListener('keydown', (e) => {
            // Don't trigger shortcuts when typing in inputs
            if (e.target.tagName === 'INPUT' || 
                e.target.tagName === 'TEXTAREA' || 
                e.target.isContentEditable) {
                return;
            }

            const key = this.getKeyString(e);
            const handler = this.shortcuts.get(key);
            
            if (handler) {
                e.preventDefault();
                handler(e);
            }
        });
    }

    getKeyString(e) {
        const parts = [];
        if (e.ctrlKey || e.metaKey) parts.push('ctrl');
        if (e.shiftKey) parts.push('shift');
        if (e.altKey) parts.push('alt');
        parts.push(e.key.toLowerCase());
        return parts.join('+');
    }

    register(key, handler, description = '') {
        this.shortcuts.set(key, handler);
        return {
            key,
            description,
            unregister: () => this.shortcuts.delete(key)
        };
    }

    unregister(key) {
        this.shortcuts.delete(key);
    }
}

const keyboardShortcuts = new KeyboardShortcuts();

// Common shortcuts
keyboardShortcuts.register('ctrl+s', (e) => {
    // Find active form and submit if exists
    const activeForm = document.querySelector('form:not([style*="display: none"])');
    if (activeForm && activeForm.querySelector('button[type="submit"]')) {
        e.preventDefault();
        activeForm.querySelector('button[type="submit"]').click();
    }
}, 'Save form');

keyboardShortcuts.register('escape', (e) => {
    // Close modals
    const openModal = document.querySelector('.modal.show');
    if (openModal) {
        const bsModal = bootstrap.Modal.getInstance(openModal);
        if (bsModal) {
            bsModal.hide();
        }
    }
}, 'Close modal');

// ============================================
// Auto-save Helper
// ============================================

class AutoSave {
    constructor(form, saveFunction, interval = 30000) {
        this.form = form;
        this.saveFunction = saveFunction;
        this.interval = interval;
        this.timer = null;
        this.lastSaved = null;
    }

    start() {
        if (!this.form) return;

        this.form.addEventListener('input', () => {
            this.resetTimer();
        });
    }

    resetTimer() {
        clearTimeout(this.timer);
        this.timer = setTimeout(() => {
            this.save();
        }, this.interval);
    }

    async save() {
        if (this.saveFunction) {
            try {
                await this.saveFunction();
                this.lastSaved = new Date();
                showInfo('Auto-saved', 2000);
            } catch (error) {
                console.error('Auto-save failed:', error);
            }
        }
    }

    stop() {
        clearTimeout(this.timer);
    }
}

// ============================================
// Debounce and Throttle Utilities
// ============================================

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function throttle(func, limit) {
    let inThrottle;
    return function executedFunction(...args) {
        if (!inThrottle) {
            func.apply(this, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

// ============================================
// ConfirmDialog Wrapper (for backward compatibility)
// ============================================

const ConfirmDialog = {
    show: async function(message, options = {}) {
        return await confirmationDialog.show({
            message: typeof message === 'string' ? message : options.message || message,
            ...(typeof message === 'object' ? message : options)
        });
    },
    
    delete: async function(message, options = {}) {
        return await confirmationDialog.show({
            title: 'Confirm Delete',
            message: message || 'Are you sure you want to delete this item?',
            details: '<div class="alert alert-warning mb-0"><small><i class="bi bi-exclamation-triangle"></i> This action cannot be undone.</small></div>',
            type: 'danger',
            icon: 'bi-exclamation-triangle',
            confirmText: options.confirmText || 'Delete',
            confirmIcon: 'bi-trash',
            ...options
        });
    }
};

// ============================================
// Export Global Functions
// ============================================

// Make functions globally available
window.confirmAction = confirmAction;
window.confirmDelete = confirmDelete;
window.ConfirmDialog = ConfirmDialog; // Add ConfirmDialog wrapper
window.loadingManager = loadingManager;
window.LoadingManager = loadingManager; // Expose as LoadingManager for consistency with other pages
window.FormValidator = FormValidator;
window.keyboardShortcuts = keyboardShortcuts;
window.AutoSave = AutoSave;
window.debounce = debounce;
window.throttle = throttle;
