/**
 * Page Builder for Clinic Pro V3
 * Generates pages using centralized layout template
 */

const fs = require('fs');
const path = require('path');

class PageBuilder {
    constructor() {
        this.templatePath = path.join(__dirname, 'layout.html');
        this.pagesPath = path.join(__dirname, '..', 'pages');
        this.template = this.loadTemplate();
    }

    loadTemplate() {
        try {
            return fs.readFileSync(this.templatePath, 'utf8');
        } catch (error) {
            console.error('Error loading template:', error);
            return null;
        }
    }

    /**
     * Build a page using the centralized layout
     */
    buildPage(pageName, options = {}) {
        const {
            title = this.capitalizeFirst(pageName),
            styles = '',
            content = '',
            scripts = ''
        } = options;

        if (!this.template) {
            throw new Error('Template not loaded');
        }

        let pageContent = this.template
            .replace('{{PAGE_TITLE}}', title)
            .replace('{{PAGE_STYLES}}', styles)
            .replace('{{PAGE_CONTENT}}', content)
            .replace('{{PAGE_SCRIPTS}}', scripts);

        return pageContent;
    }

    /**
     * Save page to file
     */
    savePage(pageName, content) {
        const filePath = path.join(this.pagesPath, `${pageName}.html`);
        try {
            fs.writeFileSync(filePath, content, 'utf8');
            console.log(`✅ Generated: ${pageName}.html`);
        } catch (error) {
            console.error(`❌ Error saving ${pageName}.html:`, error);
        }
    }

    /**
     * Capitalize first letter
     */
    capitalizeFirst(str) {
        return str.charAt(0).toUpperCase() + str.slice(1);
    }

    /**
     * Generate all pages with centralized sidebar
     */
    generateAllPages() {
        const pages = [
            'dashboard',
            'pos',
            'patients',
            'inpatients',
            'inpatient-sales',
            'emr',
            'vitals-chart',
            'emr-all',
            'doctors',
            'appointments',
            'products',
            'services',
            'sales',
            'purchases',
            'add-purchase',
            'customers',
            'suppliers',
            'expenses',
            'laboratory',
            'hospital-operations',
            'advanced-pharmacy',
            'analytics-dashboard',
            'reports',
            'lab-reports',
            'stock-sales-report',
            'service-sales-report',
            'voucher-sales-report',
            'doctor-performance-report',
            'discharged-patient-report',
            'notifications',
            'users',
            'backup',
            'settings'
        ];

        pages.forEach(pageName => {
            try {
                const content = this.buildPage(pageName, {
                    title: this.getPageTitle(pageName),
                    styles: this.getPageStyles(pageName),
                    content: this.getPageContent(pageName),
                    scripts: this.getPageScripts(pageName)
                });
                this.savePage(pageName, content);
            } catch (error) {
                console.error(`Error generating ${pageName}:`, error);
            }
        });
    }

    /**
     * Get page-specific title
     */
    getPageTitle(pageName) {
        const titles = {
            'dashboard': 'Dashboard',
            'pos': 'Point of Sale',
            'patients': 'Patients (OPD)',
            'inpatients': 'Inpatients (IPD)',
            'inpatient-sales': 'Inpatient Sales',
            'emr': 'EMR / Treatment',
            'vitals-chart': 'Vitals Chart',
            'emr-all': 'EMR Overview',
            'doctors': 'Doctors',
            'appointments': 'Appointments',
            'products': 'Products/Stock',
            'services': 'Services',
            'sales': 'Sales',
            'purchases': 'Purchases',
            'add-purchase': 'Add Purchase',
            'customers': 'Customers',
            'suppliers': 'Suppliers',
            'expenses': 'Expenses',
            'laboratory': 'Laboratory (LIS)',
            'hospital-operations': 'Hospital Operations',
            'advanced-pharmacy': 'Advanced Pharmacy',
            'analytics-dashboard': 'Analytics Dashboard',
            'reports': 'Reports',
            'lab-reports': 'Laboratory Reports',
            'stock-sales-report': 'Stock Sales Report',
            'service-sales-report': 'Service Sales Report',
            'voucher-sales-report': 'Voucher Sales Report',
            'doctor-performance-report': 'Doctor Performance Report',
            'discharged-patient-report': 'Discharged Patient Report',
            'notifications': 'Notifications',
            'users': 'User Management',
            'backup': 'Backup & Restore',
            'settings': 'Settings'
        };
        return titles[pageName] || this.capitalizeFirst(pageName);
    }

    /**
     * Get page-specific styles
     */
    getPageStyles(pageName) {
        // Return empty string for now - pages can add their own styles
        return '';
    }

    /**
     * Get page-specific content
     */
    getPageContent(pageName) {
        // Return basic content structure - pages can customize as needed
        return `
        <!-- Header -->
        <div class="glass-card mb-4">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h2 class="mb-0"><i class="bi bi-${this.getPageIcon(pageName)}"></i> ${this.getPageTitle(pageName)}</h2>
                </div>
            </div>
        </div>

        <!-- Page Content -->
        <div class="glass-card">
            <div class="p-4">
                <h4>Welcome to ${this.getPageTitle(pageName)}</h4>
                <p>This page is using the centralized sidebar layout.</p>
                <p>Page-specific content will be added here.</p>
            </div>
        </div>
        `;
    }

    /**
     * Get page-specific scripts
     */
    getPageScripts(pageName) {
        return `
        <script>
            // Page-specific initialization
            document.addEventListener('DOMContentLoaded', function() {
                console.log('${pageName} page loaded with centralized sidebar');
                // Add page-specific initialization code here
            });
        </script>
        `;
    }

    /**
     * Get page icon
     */
    getPageIcon(pageName) {
        const icons = {
            'dashboard': 'speedometer2',
            'pos': 'cart3',
            'patients': 'people-fill',
            'inpatients': 'hospital',
            'inpatient-sales': 'cart-check',
            'emr': 'clipboard2-pulse',
            'vitals-chart': 'activity',
            'emr-all': 'clipboard2-data',
            'doctors': 'person-badge',
            'appointments': 'calendar-check',
            'products': 'box-seam',
            'services': 'briefcase-fill',
            'sales': 'graph-up',
            'purchases': 'bag-fill',
            'add-purchase': 'bag-plus-fill',
            'customers': 'person-lines-fill',
            'suppliers': 'truck',
            'expenses': 'cash-coin',
            'laboratory': 'clipboard2-pulse',
            'hospital-operations': 'hospital',
            'advanced-pharmacy': 'capsule',
            'analytics-dashboard': 'graph-up-arrow',
            'reports': 'file-earmark-text',
            'lab-reports': 'file-medical',
            'stock-sales-report': 'box-seam',
            'service-sales-report': 'briefcase',
            'voucher-sales-report': 'receipt-cutoff',
            'doctor-performance-report': 'person-badge',
            'discharged-patient-report': 'person-check',
            'notifications': 'bell-fill',
            'users': 'people',
            'backup': 'cloud-arrow-up',
            'settings': 'gear-fill'
        };
        return icons[pageName] || 'circle';
    }

    /**
     * Get page description
     */
    getPageDescription(pageName) {
        const descriptions = {
            'dashboard': 'Overview & Analytics',
            'pos': 'Quick Billing',
            'patients': 'Outpatient Records',
            'inpatients': 'Admitted Patients',
            'inpatient-sales': 'Daily Usage & Deposits',
            'emr': 'Medical Records',
            'vitals-chart': 'Monitor Health Data',
            'emr-all': 'All Records View',
            'doctors': 'Manage Medical Staff',
            'appointments': 'Schedule & Book',
            'products': 'Inventory Management',
            'services': 'Service Catalog',
            'sales': 'View Transactions',
            'purchases': 'Purchase History',
            'add-purchase': 'New Stock Entry',
            'customers': 'Client Database',
            'suppliers': 'Vendor Management',
            'expenses': 'Track Spending',
            'laboratory': 'Lab Information System',
            'hospital-operations': 'Beds & Resources',
            'advanced-pharmacy': 'Drug Management',
            'analytics-dashboard': 'Business Intelligence',
            'reports': 'Complete Dashboard',
            'lab-reports': 'Test Results',
            'stock-sales-report': 'Product Sales Analysis',
            'service-sales-report': 'Service Sales Analysis',
            'voucher-sales-report': 'Transaction Analysis',
            'doctor-performance-report': 'Patient Count & Revenue',
            'discharged-patient-report': 'Complete Patient Summary',
            'notifications': 'Alerts & Messages',
            'users': 'Roles & Permissions',
            'backup': 'Data Protection',
            'settings': 'System Configuration'
        };
        return descriptions[pageName] || 'Page Description';
    }
}

// Export for use in other modules
module.exports = PageBuilder;

// Run if called directly
if (require.main === module) {
    const builder = new PageBuilder();
    builder.generateAllPages();
}
