/**
 * Update all HTML files to use local vendor assets instead of CDN
 * Run with: node frontend/update-html-to-offline.js
 */

const fs = require('fs');
const path = require('path');

// Simple glob replacement using fs
function findHtmlFiles(dir, fileList = []) {
    const files = fs.readdirSync(dir);
    
    files.forEach(file => {
        const filePath = path.join(dir, file);
        const stat = fs.statSync(filePath);
        
        if (stat.isDirectory() && !filePath.includes('node_modules') && !filePath.includes('vendor')) {
            fileList = findHtmlFiles(filePath, fileList);
        } else if (file.endsWith('.html')) {
            fileList.push(filePath);
        }
    });
    
    return fileList;
}

// Patterns to replace
const replacements = [
    // Bootstrap CSS
    {
        pattern: /https:\/\/cdn\.jsdelivr\.net\/npm\/bootstrap@5\.3\.2\/dist\/css\/bootstrap\.min\.css/g,
        replacement: 'vendor/bootstrap/css/bootstrap.min.css'
    },
    // Bootstrap JS
    {
        pattern: /https:\/\/cdn\.jsdelivr\.net\/npm\/bootstrap@5\.3\.2\/dist\/js\/bootstrap\.bundle\.min\.js/g,
        replacement: 'vendor/bootstrap/js/bootstrap.bundle.min.js'
    },
    // Bootstrap Icons CSS
    {
        pattern: /https:\/\/cdn\.jsdelivr\.net\/npm\/bootstrap-icons@1\.11\.1\/font\/bootstrap-icons\.css/g,
        replacement: 'vendor/bootstrap-icons/css/bootstrap-icons.css'
    },
    // Flag Icons CSS
    {
        pattern: /https:\/\/cdnjs\.cloudflare\.com\/ajax\/libs\/flag-icon-css\/6\.6\.6\/css\/flag-icons\.min\.css/g,
        replacement: 'vendor/flag-icons/css/flag-icons.min.css'
    }
];

// Adjust path based on file location
function adjustPath(originalPath, filePath) {
    // Determine depth (how many levels from frontend root)
    const relativePath = path.relative(path.join(__dirname), filePath);
    const depth = relativePath.split(path.sep).length - 1;
    
    // If file is in pages/ or templates/, add ../ to path
    if (filePath.includes('pages') || filePath.includes('templates')) {
        return '../' + originalPath;
    }
    // If file is in root (index.html), use path as-is
    return originalPath;
}

// Update a single file
function updateFile(filePath) {
    try {
        let content = fs.readFileSync(filePath, 'utf8');
        let modified = false;
        
        replacements.forEach(({ pattern, replacement }) => {
            if (pattern.test(content)) {
                const adjustedReplacement = adjustPath(replacement, filePath);
                content = content.replace(pattern, adjustedReplacement);
                modified = true;
                console.log(`  ✓ Updated: ${pattern.source.substring(0, 50)}...`);
            }
        });
        
        if (modified) {
            fs.writeFileSync(filePath, content, 'utf8');
            console.log(`✅ Updated: ${path.relative(__dirname, filePath)}`);
            return true;
        }
        
        return false;
    } catch (error) {
        console.error(`❌ Error updating ${filePath}:`, error.message);
        return false;
    }
}

// Main function
function main() {
    console.log('🔄 Updating HTML files to use local vendor assets...\n');
    
    // Find all HTML files
    const htmlFiles = findHtmlFiles(__dirname);
    
    console.log(`Found ${htmlFiles.length} HTML files\n`);
    
    let updatedCount = 0;
    
    htmlFiles.forEach(file => {
        if (updateFile(file)) {
            updatedCount++;
        }
    });
    
    console.log(`\n✅ Updated ${updatedCount} files`);
    console.log('📝 All HTML files now use local vendor assets!');
}

// Run if executed directly
if (require.main === module) {
    main();
}

module.exports = { updateFile, adjustPath };
