const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');

// Database configuration
const dbConfig = {
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'clinic_pro_db',
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0
};

async function runMigration() {
    let connection;
    
    try {
        console.log('🔗 Connecting to database...');
        connection = await mysql.createConnection(dbConfig);
        console.log('✅ Connected to database');
        
        // Read migration file
        const migrationPath = path.join(__dirname, '..', 'database', 'migrations', '007_add_active_to_payment_method.sql');
        console.log('📄 Reading migration file:', migrationPath);
        const sql = fs.readFileSync(migrationPath, 'utf8');
        
        // Split SQL statements and execute them
        const statements = sql
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0 && !s.startsWith('--'));
        
        console.log(`\n🚀 Executing ${statements.length} SQL statements...\n`);
        
        for (const statement of statements) {
            if (statement.trim()) {
                console.log('Executing:', statement.substring(0, 100) + '...');
                await connection.execute(statement);
                console.log('✅ Success\n');
            }
        }
        
        console.log('✅ Migration completed successfully!');
        
        // Verify the column was added
        const [columns] = await connection.execute(`
            SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'tbl_payment_method'
        `, [dbConfig.database]);
        
        console.log('\n📊 Current table structure:');
        console.table(columns);
        
    } catch (error) {
        console.error('❌ Migration failed:', error);
        process.exit(1);
    } finally {
        if (connection) {
            await connection.end();
            console.log('\n🔌 Database connection closed');
        }
    }
}

runMigration();

